/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Author: Damian Waradzyn
 */

#include <QtCore/QDebug>
#include <SDL/SDL_ttf.h>
#include "VirtualKeyboard.h"
#include "../main.h"
#include "../display/texture.h"
#include "../display/glcanvas.h"

const GLfloat keyHeight = 64;

#define KEYS_IN_FIRST_ROW 10
#define KEYS_IN_SECOND_ROW 9
#define KEYS_IN_THIRD_ROW 9
#define KEYS_IN_FOURTH_ROW 5
#define KEY_COUNT (KEYS_IN_FIRST_ROW + KEYS_IN_SECOND_ROW + KEYS_IN_THIRD_ROW + KEYS_IN_FOURTH_ROW)

static Key keys[KEY_COUNT];

GLfloat keyVertices[18 * KEY_COUNT];

static GLfloat keyboardTexCoordsPortrait[][8] = { { 480 / 512.0, 0.0f, /* Top Right Of The Texture */
480 / 512.0f, 265 / 512.0f, /* Bottom Right Of The Texture */
0.0f, 0.0f, /* Bottom Left Of The Texture */
0.0f, 265 / 512.0f, /* Top Left Of The Texture */
} };

void setKeyVertices(int idx, Key* key) {
	GLfloat x = SCREEN_HEIGHT - key -> x;
	keyVertices[idx++] = key -> y;
	keyVertices[idx++] = x;
	keyVertices[idx++] = 0;

	keyVertices[idx++] = key -> y;
	keyVertices[idx++] = x - key -> width;
	keyVertices[idx++] = 0;

	keyVertices[idx++] = key -> y + keyHeight;
	keyVertices[idx++] = x;
	keyVertices[idx++] = 0;

	keyVertices[idx++] = key -> y + keyHeight;
	keyVertices[idx++] = x;
	keyVertices[idx++] = 0;

	keyVertices[idx++] = key -> y;
	keyVertices[idx++] = x - key -> width;
	keyVertices[idx++] = 0;

	keyVertices[idx++] = key -> y + keyHeight;
	keyVertices[idx++] = x - key -> width;
	keyVertices[idx++] = 0;
}

VirtualKeyboard::VirtualKeyboard() {
	layouts = KeyboardLayout::loadFromConfig();
	currentLayout = layouts -> first();
	lowercaseSurface = NULL;
	lowercaseMask = NULL;
	uppercaseSurface = NULL;
	uppercaseMask = NULL;
	lowercaseTexture = 0;
	lowercaseMaskTexture = 0;
	uppercaseTexture = 0;
	uppercaseMaskTexture = 0;
	uppercase = false;
	initKeys();
}

// TODO - landscape support
void VirtualKeyboard::initKeys() {
	int i, idx = 0;
	int padding = 2;
	int normalKeyWidth = (SCREEN_HEIGHT - padding * 10) / 10;

	for (i = 0; i < KEYS_IN_FIRST_ROW; i++) {
		keys[i].width = normalKeyWidth;
		keys[i].x = i * (padding + normalKeyWidth) + 1;
		keys[i].y = 0;
		keys[i].lowercase = QString(currentLayout -> lowercaseChars -> at(0).at(i));
		keys[i].lowercaseAlternate = QString(currentLayout -> alternateLowercaseChars -> at(0).at(i));
		keys[i].uppercase = QString(currentLayout -> uppercaseChars -> at(0).at(i));
		keys[i].uppercaseAlternate = QString(currentLayout -> alternateUppercaseChars -> at(0).at(i));
		keys[i].type = CHAR_KEY;
	}

	for (i = 0; i < KEYS_IN_SECOND_ROW; i++) {
		idx = KEYS_IN_FIRST_ROW + i;
		keys[idx].width = normalKeyWidth;
		keys[idx].x = normalKeyWidth / 2 + i * (padding + normalKeyWidth) + 1;
		keys[idx].y = keyHeight + padding;
		keys[idx].lowercase = QString(currentLayout -> lowercaseChars -> at(1).at(i));
		keys[idx].lowercaseAlternate = QString(currentLayout -> alternateLowercaseChars -> at(1).at(i));
		keys[idx].uppercase = QString(currentLayout -> uppercaseChars -> at(1).at(i));
		keys[idx].uppercaseAlternate = QString(currentLayout -> alternateUppercaseChars -> at(1).at(i));
		keys[idx].type = CHAR_KEY;
	}

	// Shift
	int shiftIdx = KEYS_IN_FIRST_ROW + KEYS_IN_SECOND_ROW;
	keys[shiftIdx].width = 69;
	keys[shiftIdx].x = 1;
	keys[shiftIdx].y = (keyHeight + padding) * 2;
	keys[shiftIdx].type = SHIFT_KEY;

	for (i = 1; i < KEYS_IN_THIRD_ROW - 1; i++) {
		idx = KEYS_IN_FIRST_ROW + KEYS_IN_SECOND_ROW + i;
		keys[idx].width = normalKeyWidth;
		keys[idx].x = (i - 1) * (padding + normalKeyWidth) + keys[shiftIdx].width + padding + 1;
		keys[idx].y = (keyHeight + padding) * 2;
		keys[idx].lowercase = QString(currentLayout -> lowercaseChars -> at(2).at(i - 1));
		keys[idx].lowercaseAlternate = QString(currentLayout -> alternateLowercaseChars -> at(2).at(i - 1));
		keys[idx].uppercase = QString(currentLayout -> uppercaseChars -> at(2).at(i - 1));
		keys[idx].uppercaseAlternate = QString(currentLayout -> alternateUppercaseChars -> at(2).at(i - 1));
		keys[idx].type = CHAR_KEY;
	}
	// Backspace
	int backspaceIdx = KEYS_IN_FIRST_ROW + KEYS_IN_SECOND_ROW + KEYS_IN_THIRD_ROW - 1;
	keys[backspaceIdx].width = 71;
	keys[backspaceIdx].x = SCREEN_HEIGHT - keys[backspaceIdx].width - 1;
	keys[backspaceIdx].y = (keyHeight + padding) * 2;
	keys[backspaceIdx].type = BACKSPACE_KEY;

	// Fourth row: language
	idx = KEY_COUNT - KEYS_IN_FOURTH_ROW;
	keys[idx].width = 90;
	keys[idx].x = 1;
	keys[idx].y = (keyHeight + padding) * 3;
	keys[idx].type = LANGUAGE_KEY;

	// Fourth row: coma
	idx++;
	keys[idx].width = normalKeyWidth;
	keys[idx].x = keys[idx - 1].x + keys[idx - 1].width + padding;
	keys[idx].y = (keyHeight + padding) * 3;
	keys[idx].lowercase = QString(currentLayout -> lowercaseChars -> at(3).at(0));
	keys[idx].lowercaseAlternate = QString(currentLayout -> alternateLowercaseChars -> at(3).at(0));
	keys[idx].uppercase = QString(currentLayout -> uppercaseChars -> at(3).at(0));
	keys[idx].uppercaseAlternate = QString(currentLayout -> alternateUppercaseChars -> at(3).at(0));
	keys[idx].type = CHAR_KEY;

	// Fourth row: space
	idx++;
	keys[idx].width = 198;
	keys[idx].x = keys[idx - 1].x + keys[idx - 1].width + padding;
	keys[idx].y = (keyHeight + padding) * 3;
	keys[idx].lowercase = QString(currentLayout -> lowercaseChars -> at(3).at(1));
	keys[idx].lowercaseAlternate = QString(currentLayout -> alternateLowercaseChars -> at(3).at(1));
	keys[idx].uppercase = QString(currentLayout -> uppercaseChars -> at(3).at(1));
	keys[idx].uppercaseAlternate = QString(currentLayout -> alternateUppercaseChars -> at(3).at(1));
	keys[idx].type = CHAR_KEY;

	// Fourth row: period
	idx++;
	keys[idx].width = normalKeyWidth;
	keys[idx].x = keys[idx - 1].x + keys[idx - 1].width + padding;
	keys[idx].y = (keyHeight + padding) * 3;
	keys[idx].lowercase = QString(currentLayout -> lowercaseChars -> at(3).at(2));
	keys[idx].lowercaseAlternate = QString(currentLayout -> alternateLowercaseChars -> at(3).at(2));
	keys[idx].uppercase = QString(currentLayout -> uppercaseChars -> at(3).at(2));
	keys[idx].uppercaseAlternate = QString(currentLayout -> alternateUppercaseChars -> at(3).at(2));
	keys[idx].type = CHAR_KEY;

	// Fourth row: enter
	idx++;
	keys[idx].width = 90;
	keys[idx].x = keys[idx - 1].x + keys[idx - 1].width + padding;
	keys[idx].y = (keyHeight + padding) * 3;
	keys[idx].type = ENTER_KEY;

	for (i = 0; i < KEY_COUNT; i++) {
		setKeyVertices(i * 18, &keys[i]);
	}
	preRenderTexture(&lowercaseSurface, &lowercaseMask, false);
	preRenderTexture(&uppercaseSurface, &uppercaseMask, true);
}

void renderText(SDL_Surface* dest, TTF_Font* font, SDL_Rect& rect, int tx, int ty, char* text, SDL_Color& color) {
	SDL_Surface* textSurface = TTF_RenderUTF8_Blended(font, text, color);

	if (!textSurface) {
		fprintf(stderr, "TTF_RenderText_Blended failed: %s\n", TTF_GetError());
		exit(1);
	} else {
		SDL_Rect srcRect;
		srcRect.x = 0;
		srcRect.y = 0;
		srcRect.w = textSurface -> w;
		srcRect.h = textSurface -> h;

		SDL_Rect dstRect;
		dstRect.x = rect.x + rect.w / 2 - srcRect.w / 2 + tx;
		dstRect.y = rect.y + rect.h / 2 - srcRect.h / 2 + ty;
		dstRect.w = srcRect.w;
		dstRect.h = srcRect.h;

		SDL_UpperBlit(textSurface, &srcRect, dest, &dstRect);
		SDL_FreeSurface(textSurface);
	}
}

void VirtualKeyboard::preRenderTexture(SDL_Surface** surfacePtr, SDL_Surface** maskPtr, bool uppercase) {
	if (*surfacePtr != NULL) {
		SDL_FreeSurface(*surfacePtr);
	}
	*surfacePtr = loadTextureFromFile("/opt/cloudgps/res/keyboard_portrait.png", NULL, maskPtr);
	if (*surfacePtr == NULL) {
		exit(1);
	}
	SDL_Surface* surface = *surfacePtr;
	SDL_Rect rect;
	rect.x = 0;
	rect.y = 0;
	rect.w = surface -> w;
	rect.h = surface -> h;

	TTF_Font* smallFont = TTF_OpenFont(options.ttfFont, 18);
	if (!smallFont) {
		fprintf(stderr, "TTF_OpenFont error: %s\n", TTF_GetError());
		exit(1);
	}

	TTF_Font* bigFont = TTF_OpenFont(options.ttfFont, 26);
	if (!bigFont) {
		fprintf(stderr, "TTF_OpenFont error: %s\n", TTF_GetError());
		exit(1);
	}

	for (int i = 0; i < KEY_COUNT; i++) {
		QString chars, alternate;
		if (uppercase) {
			chars = keys[i].uppercase;
			alternate = keys[i].uppercaseAlternate;
		} else {
			chars = keys[i].lowercase;
			alternate = keys[i].lowercaseAlternate;
		}

		SDL_Rect rect;
		rect.x = keys[i].x;
		rect.y = keys[i].y;
		rect.w = keys[i].width;
		rect.h = keyHeight;
		//SDL_FillRect(lowercaseSurface, &rect, SDL_MapRGBA(lowercaseSurface -> format, 0x7F, 0x70, 0x70, 0x7F));
		if (keys[i].type == LANGUAGE_KEY) {
			SDL_Color color = { 255, 255, 255 };
			renderText(surface, bigFont, rect, 0, 0, currentLayout -> abbreviation.toUtf8().data(), color);
		} else {
			if (!chars.isEmpty()) {
				SDL_Color color = { 255, 255, 255 };
				renderText(surface, bigFont, rect, 0, 10, chars.toUtf8().data(), color);
			}
			if (!alternate.isEmpty()) {
				SDL_Color color = { 80, 80, 255 };
				renderText(surface, smallFont, rect, 10, -12, alternate.toUtf8().data(), color);
			}
		}
	}
	TTF_CloseFont(smallFont);
	TTF_CloseFont(bigFont);
}

void VirtualKeyboard::renderBackgrounds() {
	glVertexPointer(3, GL_FLOAT, 0, keyVertices);
	GLfloat tx = SCREEN_WIDTH - (int) canvas.searchBarY + 66;
	glTranslatef(tx, 0, 0);
	glBlendFunc(GL_ZERO, GL_ONE_MINUS_SRC_ALPHA);
	glColor4f(1.0, 0.5, 0.5, 0.5);
	glBindTexture(GL_TEXTURE_2D, 0);
	glDrawArrays(GL_TRIANGLES, 0, KEY_COUNT * 2 * 3);
	glTranslatef(-tx, 0, 0);
}

void VirtualKeyboard::handleKeyHold(int pressedKey, int lastPressedKey, long lastPressedTime) {
	switch (keys[pressedKey].type) {
	case CHAR_KEY:
		if (lastPressedKey != -1) {
			canvas.searchBar -> removeChar();
		}
		if (lastPressedKey != -1 && nowMillis - lastPressedTime > 400) {
			if (uppercase) {
				canvas.searchBar -> addChars(keys[pressedKey].uppercaseAlternate);
			} else {
				canvas.searchBar -> addChars(keys[pressedKey].lowercaseAlternate);
			}
		} else {
			if (uppercase) {
				canvas.searchBar -> addChars(keys[pressedKey].uppercase);
			} else {
				canvas.searchBar -> addChars(keys[pressedKey].lowercase);
			}
		}
		break;
	case ENTER_KEY:
		break;
	case BACKSPACE_KEY:
		if (lastPressedKey == -1) {
			canvas.searchBar -> removeChar();
		} else  if (nowMillis - lastPressedTime == 0 || nowMillis - lastPressedTime > 400) {
			static long lastFrame = 0;

			if (lastFrame % 3 == 0) {
				canvas.searchBar -> removeChar();
			}
			lastFrame++;
		}
		break;
	case SHIFT_KEY:
	case LANGUAGE_KEY:
		break;
	}
}
void VirtualKeyboard::handleKeyReleased(Key& key, int lastPressedKey, long lastPressedTime) {
	switch (key.type) {
	case CHAR_KEY:
		uppercase = false;
		break;
	case ENTER_KEY:
		if (!canvas.searchBar -> getValue().isEmpty()) {
			processNewSearchQuery(canvas.searchBar -> getValue().toUtf8().data());
		}
		break;
	case BACKSPACE_KEY:
		break;
	case SHIFT_KEY:
		uppercase = !uppercase;
		break;
	case LANGUAGE_KEY:
//		canvas.searchBarActive = FALSE;
		canvas.keyboardLanguageMenu -> show();
		qDebug() << "pressed LANGUAGE";
		break;
	}
}

QList<KeyboardLayout*>* VirtualKeyboard::getLayouts() {
	return layouts;
}

KeyboardLayout* VirtualKeyboard::getCurrentLayout() {
	return currentLayout;
}

void VirtualKeyboard::setLayout(KeyboardLayout* layout) {
	if (currentLayout != layout) {
		currentLayout = layout;
		deleteTextures();
		initKeys();
	}
}

int VirtualKeyboard::handleMouse() {
	static int lastPressedKey = -1;
	static long lastPressedTime = -1;
	if (!isActive()) {
		lastPressedKey = -1;
		return FALSE;
	}
	if (mouse[0].x < SCREEN_WIDTH - (int) canvas.searchBarY + 66) {
		lastPressedKey = -1;
		return FALSE;
	}
	int pressedKey = -1;
	int x = SCREEN_HEIGHT - mouse[0].y;
	int y = mouse[0].x - (SCREEN_WIDTH - (int) canvas.searchBarY + 66);
	for (int i = 0; i < KEY_COUNT; i++) {
		if (x >= keys[i].x && x <= keys[i].x + keys[i].width && y >= keys[i].y && y <= keys[i].y + keyHeight) {
			pressedKey = i;
			break;
		}
	}

	if (pressedKey >= 0 && mouse[0].button != 0) {
		handleKeyHold(pressedKey, lastPressedKey, lastPressedTime);
	}

	if (pressedKey >= 0 && mouse[0].oldButton != 0 && mouse[0].button == 0) {
		handleKeyReleased(keys[pressedKey], lastPressedKey, lastPressedTime);
	}

	if (mouse[0].button != 0 && pressedKey != -1 && pressedKey != lastPressedKey) {
		lastPressedKey = pressedKey;
		lastPressedTime = nowMillis;
	}
	if (mouse[0].button == 0) {
		lastPressedKey = -1;
	}

	return TRUE;
}

bool VirtualKeyboard::isActive() {
	return canvas.searchBarY != 0.0;
}

void VirtualKeyboard::deleteTextures() {
	if (lowercaseTexture != 0) {
		deleteTexture(lowercaseTexture);
		lowercaseTexture = 0;
		deleteTexture(lowercaseMaskTexture);
		lowercaseMaskTexture = 0;
	}
	if (uppercaseTexture != 0) {
		deleteTexture(uppercaseTexture);
		uppercaseTexture = 0;
		deleteTexture(uppercaseMaskTexture);
		uppercaseMaskTexture = 0;
	}
}
void VirtualKeyboard::createTextures() {
	if (uppercase) {
		if (uppercaseTexture == 0) {
			uppercaseTexture = createTexture((GLushort*) uppercaseSurface -> pixels, 512, 512, FALSE);
			uppercaseMaskTexture = createTexture((GLushort*) uppercaseMask -> pixels, 512, 512, FALSE);
		}
	} else {
		if (lowercaseTexture == 0) {
			lowercaseTexture = createTexture((GLushort*) lowercaseSurface -> pixels, 512, 512, FALSE);
			lowercaseMaskTexture = createTexture((GLushort*) lowercaseMask -> pixels, 512, 512, FALSE);
		}
	}
}

void VirtualKeyboard::render() {
	if (!isActive() || canvas.searchBarY < 66) {
		deleteTextures();
		return;
	}
	createTextures();
	GLuint texture, mask;
	if (uppercase) {
		texture = uppercaseTexture;
		mask = uppercaseMaskTexture;
	} else {
		texture = lowercaseTexture;
		mask = lowercaseMaskTexture;
	}
	//renderBackgrounds();

	GLfloat tx = SCREEN_WIDTH - (int) canvas.searchBarY + 66;
	glTranslatef(tx, 0, 0);
	resetQuadStripVertices();
	setQuadStripSize(265, 480, quadStripVertices);
	glVertexPointer(3, GL_FLOAT, 0, quadStripVertices);
	glTexCoordPointer(2, GL_FLOAT, 0, keyboardTexCoordsPortrait);
	glColor4f(1.0, 1.0, 1.0, 1.0);

	glBlendFunc(GL_DST_COLOR, GL_ZERO);
	glBindTexture(GL_TEXTURE_2D, mask);
	glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);

	glBlendFunc(GL_ONE, GL_ONE_MINUS_SRC_ALPHA);
	glBindTexture(GL_TEXTURE_2D, texture);
	glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
	glTranslatef(-tx, 0, 0);
}

VirtualKeyboard::~VirtualKeyboard() {
}
