/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Author: Damian Waradzyn
 */
#ifndef GPS_H_
#define GPS_H_

#include "../main.h"
#include <QtCore/QObject>
#include <QtCore/QtDebug>
#include <QtLocation/QGeoPositionInfoSource>
#include <QtLocation/QGeoPositionInfo>
#include <QtLocation/QGeoSatelliteInfoSource>
#include <QtLocation/QGeoSatelliteInfo>

QTM_USE_NAMESPACE

extern UiElement *position;

class Gps: public QObject {
	Q_OBJECT
public:
	int satellitesInViewUpdateCounter;
	
	Gps(QObject *parent = 0) : QObject(parent)
	{
		QGeoPositionInfoSource *positionInfo = QGeoPositionInfoSource::createDefaultSource(this);
		if (positionInfo) {
			positionInfo->setPreferredPositioningMethods(QGeoPositionInfoSource::AllPositioningMethods);
			connect(positionInfo, SIGNAL(positionUpdated(QGeoPositionInfo)),
					this, SLOT(positionUpdated(QGeoPositionInfo)));
               positionInfo -> setUpdateInterval(positionInfo -> minimumUpdateInterval());
			positionInfo -> startUpdates();
			qDebug() << "Location source initialized";
		} else {
			qDebug() << "Could not initialize location source";
		}

		QGeoSatelliteInfoSource *satelliteInfo = QGeoSatelliteInfoSource::createDefaultSource(this);
		if (satelliteInfo) {
		    connect(satelliteInfo, SIGNAL(satellitesInUseUpdated(const QList<QGeoSatelliteInfo> &)),this,
		    				SLOT(satellitesInUseUpdated(const QList<QGeoSatelliteInfo> &)));
		    connect(satelliteInfo, SIGNAL(satellitesInViewUpdated(const QList<QGeoSatelliteInfo> &)),this,
		    				SLOT(satellitesInViewUpdated(const QList<QGeoSatelliteInfo> &)));
			satelliteInfo -> startUpdates();
			qDebug() << "GPS Satellite info source initialized";
		} else {
			qDebug() << "Could not initialize satellite info source";
		}
		satellitesInViewUpdateCounter = 0;
	}

private slots:
	void positionUpdated(const QGeoPositionInfo &info) {
		canvas.currentPositionInfo.positionKnown = TRUE;
		canvas.currentPositionInfo.position.latitude = info.coordinate().latitude();
		canvas.currentPositionInfo.position.longitude = info.coordinate().longitude();
		canvas.currentPositionInfo.groundSpeed = info.attribute(QGeoPositionInfo::GroundSpeed) * 3.6;
		canvas.currentPositionInfo.heading = info.attribute(QGeoPositionInfo::Direction);
		canvas.currentPositionInfo.horizontalAccuracy = info.attribute(QGeoPositionInfo::HorizontalAccuracy);
		if (info.coordinate().type() == QGeoCoordinate::Coordinate3D) {
			canvas.currentPositionInfo.altitude = info.coordinate().altitude();
		}

	}
	void satellitesInViewUpdated(const QList<QGeoSatelliteInfo> &list) {
		int i;
		canvas.currentPositionInfo.satellitesInView = list.size();
		for (i = 0; i < list.size(); i++) {
			QGeoSatelliteInfo info = list.at(i);
			canvas.currentPositionInfo.satellites[i].prn = info.prnNumber();
			canvas.currentPositionInfo.satellites[i].signalStrength = info.signalStrength();
			canvas.currentPositionInfo.satellites[i].inUse = FALSE;
			canvas.currentPositionInfo.satellites[i].elevation = info.attribute(QGeoSatelliteInfo::Elevation);
			canvas.currentPositionInfo.satellites[i].azimuth = info.attribute(QGeoSatelliteInfo::Azimuth);
		}
		if (++satellitesInViewUpdateCounter > 3) {
			canvas.currentPositionInfo.hasGpsFix = FALSE;
		}
	}
	void satellitesInUseUpdated(const QList<QGeoSatelliteInfo> &list) {
		if (list.size() == 0) {
			canvas.currentPositionInfo.hasGpsFix = FALSE;
		} else {
			satellitesInViewUpdateCounter = 0;
			canvas.currentPositionInfo.hasGpsFix = TRUE;
		}
		canvas.currentPositionInfo.satellitesInUse = list.size();

		for (int i = 0; i < list.size(); i++) {
			for (int j = 0; j < canvas.currentPositionInfo.satellitesInView; j++) {
				if (canvas.currentPositionInfo.satellites[j].prn == list.at(i).prnNumber()) {
					canvas.currentPositionInfo.satellites[j].inUse = TRUE;
				}
			}
		}
	}
};

void initGps();
void updateGps();

#endif /* GPS_H_ */
