/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Author: Damian Waradzyn
 */
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <strings.h>
#include <memory.h>
#include <curl/curl.h>
#include <malloc.h>
#include <locale.h>

#include <SDL/SDL.h>
#include <SDL/SDL_ttf.h>

#include "main.h"
#include "tile_engine/tile.h"
#include "io/network.h"
#include "display/texture.h"
#include "display/textRender.h"
#include "display/screen.h"
#include "tile_engine/tileProviders.h"
#include "display/taskman.h"
#include "tile_engine/tileengine.h"
#include "display/glcanvas.h"
#include "geocoding/geocoder.h"
#include "routing/router.h"
#include "display/buildings3d.h"
#include "io/input.h"
#include "config/config.h"
#include "sensors/battery.h"
#include "navigation/navigation.h"
#include "sensors/compass.h"
#include "sensors/gps.h"
#include "sensors/accelerometer.h"
#include "animation/AnimationManager.h"
#include "vkb/VirtualKeyboard.h"
#include "searchbar/SearchBar.h"

#include <QtCore/QtDebug>
#include <QtDBus/QDBusConnection>

Canvas canvas;
Options options;
Accelerometer accelerometer;
Mouse mouse[MAX_MOUSE_POINTERS];
GQueue* downloadQueue;

int tileCoordinateVisibility[TILES_X + 1][TILES_Y + 1];

int quit = FALSE;
long nowMillis = 0, previousMillis = 0, diffMillis = 0;
int downloaded = 0;
long lastBatteryRefresh = 0, lastGarbageCollection = 0;

volatile int syncLoadedTilesAppearance = TRUE;
int forceGarbageCollection = FALSE;
int texturesCreatedThisFrame = 0;

int currentTilesIdx;

void setQuitFlag() {
	quit = TRUE;
}

//void takeScreenshot() {
//	SDL_Surface * image = SDL_CreateRGBSurface(SDL_SWSURFACE, SCREEN_WIDTH, SCREEN_HEIGHT, 24, 0x00FF0000, 0x0000FF00, 0x000000FF, 0);
//	glReadBuffer(GL_FRONT);
//	glReadPixels(0, 0, SCREEN_WIDTH, SCREEN_HEIGHT, GL_RGB, GL_UNSIGNED_BYTE, image->pixels);
//	sprintf(strbuf, "pic-%08l.bmp", nowMillis);
//	SDL_SaveBMP(image, strbuf);
//	SDL_FreeSurface(image);
//}

#include <QtGui/QApplication>

void initializeApplication() {
	if (!QDBusConnection::systemBus().isConnected()) {
		fprintf(stderr, "system bus not connected\n");
	}
	initCompass();
	initGps();
	initAccelerometer();

	setlocale(LC_CTYPE, NULL);
	initGL();
#if defined(N900) || defined(N950)
	SDL_ShowCursor(0);
#else
	SDL_ShowCursor(1);
#endif

	initFonts();
	loadUI();
	loadConfig();
	initInput();

	canvas.mainMenu = new Menu(Menu::createMainMenuElements());
	canvas.optionsMenu = new Menu(Menu::createOptionsMenuElements());
	canvas.mapChooseMenu = new Menu(Menu::createMapChooseMenuElements());
	canvas.searchServiceMenu = new Menu(Menu::createSearchServiceMenuElements());
	canvas.searchServiceMenu -> enterComboBoxMode();
	canvas.routingServiceMenu = new Menu(Menu::createRoutingServiceMenuElements());
	canvas.routingServiceMenu -> enterComboBoxMode();
	canvas.popup = new Menu(Menu::createMapPopupMenuElements());
	canvas.popup -> enterComboBoxMode();
	canvas.virtualKeyboard = new VirtualKeyboard();
	canvas.searchBar = new SearchBar();
	canvas.keyboardLanguageMenu = new Menu(Menu::createKeyboardLanguageMenuElements());
	canvas.keyboardLanguageMenu -> enterComboBoxMode();
	canvas.aboutDialog = new AboutDialog();

	initTileEngine();
	if (options.loadLastRoute) {
		loadRoute(NULL);
	}
	initBattery();
	curl_global_init(CURL_GLOBAL_ALL);
}

int main(int argc, char *argv[]) {
	fprintf(stderr, "CloudGPS " CLOUDGPS_VERSION "\n");
	QApplication a(argc, argv, false);
	initializeApplication();

	SDL_Event event;



	/* whether or not the window is active */
	int isActive = TRUE;


	/* wait for events */
	quit = FALSE;
	while (!quit) {
		/* handle the events in the queue */
		previousMillis = nowMillis;
		nowMillis = SDL_GetTicks();
		diffMillis = nowMillis - previousMillis;
		texturesCreatedThisFrame = 0;
		a.processEvents();

		while (SDL_PollEvent(&event)) {
			switch (event.type) {
			case SDL_ACTIVEEVENT:
				if (event.active.gain == 0) {
					isActive = FALSE;
					saveConfig();
				} else {
					isActive = TRUE;
				}
				break;
			case SDL_QUIT:
				quit = TRUE;
				break;
			case SDL_KEYDOWN:
				processKeyDown(event.key);
				break;
			case SDL_KEYUP:
				processKeyUp(event.key);
				break;
				//			case SDL_TEXTINPUT:
				//				text = event.text;
				//				processTextInput(text.text);
				//				break;
			default:
				break;
			}
		}
		processKeyboard();
		processGeocoding();
		processAccelerometer();
		tileEngineUpdateCoordinates();
		processRouting();
		tileEngineUpdateMarkers();
		setupGlScene();
		processMouse();
		tileEngine_computeBoundingTrapezoid();
		tileEngine_computeTilesVisibility();
		tileEngine_handleZoomTiles();
		updateUi();
		AnimationManager::updateAnimations();
		deleteOldConsoleLines();
		updateSyncWaitTiles();
		updateBusyValue();
		updateNavigation();

		// TODO detect screen lock
		if (!isActive) {
			// when our window is inactive save system resources by limiting framerate
#if defined(N900) || defined(N950)
			SDL_Delay(50);
#endif
		}
		drawGLScene();

		// refresh battery percentage and save current view every 10 seconds
		if (nowMillis - lastBatteryRefresh > 10000) {
			refreshBattery();
			lastBatteryRefresh = nowMillis;
			saveConfig();
		}
		if (nowMillis - lastGarbageCollection > 500 || forceGarbageCollection) /*|| g_queue_get_length(allCreatedTiles) > 4 * TILES_X * TILES_Y)*/{
			processDeferredDeallocation();
			lastGarbageCollection = nowMillis;
			forceGarbageCollection = FALSE;
		}
		prevent_screen_dimming();
	}
	saveConfig();

	// clean ourselves up and exit
	shutdownTileEngine();
	TTF_Quit();
	SDL_Quit();
	return 0;
}
