/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Author: Damian Waradzyn
 */

#include "Menu.h"
#include "MapChooseMenuElement.h"
#include "AccelerometerMenuElement.h"
#include "KeyboardLayoutMenuElement.h"
#include "ProviderMenuElement.h"
#include "Toggle3dMenuElement.h"
#include "../main.h"
#include "../animation/transitions/DelayTransition.h"
#include "../animation/transitions/LinearTransition.h"
#include "../animation/transitions/BackEaseTransition.h"
#include "../animation/transitions/CubicTransition.h"
#include "../animation/transitions/SequentialTransition.h"
#include "../animation/transitions/ParallelTransition.h"
#include "../resources/TextSurfaceManager.h"
#include "../actions/DummyAction.h"
#include "../actions/AboutAction.h"
#include "../actions/ShowMenuAction.h"
#include "../actions/Toggle3DAction.h"
#include "../actions/ToggleAccelerometerAction.h"
#include "../actions/ProviderChangeAction.h"
#include "../actions/RouteFlybyAction.h"
#include "../actions/ExitMenuAction.h"
#include "../actions/MapPopupAction.h"
#include "../actions/HidePopupMenuAction.h"
#include "../display/glcanvas.h"
#include "../tile_engine/tileengine.h"

extern UiElement* busy;
GLuint menuElementBackgroundTexture;

GLfloat menuElementBackgroundTexCoords[][8] = { { 180 / 256.0f, 0.0f, /* Top Right Of The Texture */
180 / 256.0f, 180 / 256.0f, /* Bottom Right Of The Texture */
0.0f, 0.0f, /* Bottom Left Of The Texture */
0.0f, 180 / 256.0f, /* Top Left Of The Texture */
} };

Transition* Menu::createTransitionIn(MenuElement* element, int i) {
	QList<Transition*> transitions;
	transitions.append(new DelayTransition(10 + 4 * i));
	element -> position = (i % columns == 0) ? -240.0 : 240.0;
	transitions.append(new BackEaseTransition(&element -> position, element -> position, 0, 40));
	Transition* positionTransition = new SequentialTransition(transitions);
	Transition* transparencyTransition = new CubicTransition(&element -> transparency, 0.0, 1.0, 100);
	QList<Transition*> parallelTransitions;
	parallelTransitions.append(positionTransition);
	parallelTransitions.append(transparencyTransition);
	Transition* result = new ParallelTransition(parallelTransitions);
	result -> setSkipAllowed(false);
	return result;
}

Menu::Menu(QList<MenuElement*>* elements) {
	if (menuElementBackgroundTexture == 0) {
		SDL_Surface* surface = loadTextureFromFile("/opt/cloudgps/res/menu/menu_element_bg.png", NULL, NULL);
		menuElementBackgroundTexture = createTexture((GLushort*) surface -> pixels, surface -> w, surface -> h, FALSE);
	}
	this -> elements = elements;
	active = false;
	verticalRotation = false;
	renderIcon = true;
	renderBackground = true;
	position = 0.0;
	positionDelta = 0.0;
	columns = 2;
	elementWidth = 180;
	elementHeight = 180;
	padding = 40;
	rotationAnimation = new AnimatedElement();
	rotationStateLandscape = new TransitionState();
	Transition* transitionIn = new CubicTransition(&rotation, 0.0, 90.0, 60);
	transitionIn -> setSkipAllowed(false);
	rotationStateLandscape -> setTransitionIn(transitionIn);
	Transition* transitionOut = new CubicTransition(&rotation, 90.0, 0.0, 60);
	transitionOut -> setSkipAllowed(false);
	rotationStateLandscape -> setTransitionOut(transitionOut);
}

void Menu::setVerticalRotation(bool verticalRotation) {
	this -> verticalRotation = verticalRotation;
}

void Menu::setRenderBackground(bool renderBackground) {
	this -> renderBackground = renderBackground;
}

void Menu::setRenderIcon(bool renderIcon) {
	this -> renderIcon = renderIcon;
}

void Menu::setPadding(int padding) {
	this -> padding = padding;
}

void Menu::setColumns(int columns) {
	this -> columns = columns;
}

void Menu::setElemWidth(int elemWidth) {
	this -> elementWidth = elemWidth;
}

void Menu::setElemHeight(int elemHeight) {
	this -> elementHeight = elemHeight;
}
void Menu::enterComboBoxMode() {
	setColumns(1);
	setElemHeight(60);
	setElemWidth(400);
	setPadding(50);
	setVerticalRotation(true);
	setRenderBackground(false);
	setRenderIcon(false);
}

GLfloat Menu::calculateMinPosition() {
	GLfloat menuWidth = ((elements -> size() + columns - 1) / columns * (getElemHeight() + getPadding() / 2)) + getPadding();
	GLfloat min = SCREEN_WIDTH - menuWidth;
	if (min > 0.0) {
		return min/2;
	}
	return min;
}

GLfloat Menu::calculateMaxPosition() {
//	GLfloat menuWidth = ((elements -> size() + columns - 1) / columns * (getElemHeight() + getPadding() / 2)) + getPadding();
//	GLfloat max = calculateMinPosition() + menuWidth;
//	if (max < 0.0) {
//		return max/2;
//	}
	return -padding/2;
}


void Menu::show() {
	if (!active) {
		setDimmed(true);
		QListIterator<MenuElement*> iter(*elements);
		int i = 0;

		while (iter.hasNext()) {
			MenuElement* element = iter.next();
			element -> position = (i % columns == 0) ? -240.0 : 240.0;
			element -> transparency = 0.0;
			TransitionState* state = new TransitionState();
			state -> setTransitionIn(createTransitionIn(element, i));
			element -> transition.setTargetState(state);
			element -> transition.setTransitionInCompleted(false);
			element -> transition.setPaused(false);
			element -> createTexture();
			i++;
		}
		position = 0.0;
		positionDelta = 0.0;
		active = true;
		rotation = options.orientation == PORTRAIT ? 0.0 : 90.0;
	}
}

extern TransitionState* emptyState;

void Menu::hide() {
	if (active) {
		setDimmed(false);
		QListIterator<MenuElement*> iter(*elements);

		while (iter.hasNext()) {
			MenuElement* element = iter.next();
			element -> transition.setTransitionInCompleted(true);
			element -> transition.setTargetState(emptyState);
			element -> deleteTexture();
		}
		active = false;
	}
}

GLfloat Menu::getElemPositionX(MenuElement* element, int i) {
	return (i / columns * (getElemHeight() + getPadding() / 2)) + getPadding() + position;
}
GLfloat Menu::getElemPositionY(MenuElement* element, int i) {
	if (columns == 1) {
		return SCREEN_HEIGHT - getElemWidth() - getPadding();
	}
	return ((i % columns == 0) ? SCREEN_HEIGHT / 2.0 + getPadding() / 2 : getPadding()) - element -> position;
}

GLfloat Menu::getElemWidth() {
	return elementWidth;
}

GLfloat Menu::getElemHeight() {
	return elementHeight;
}

GLfloat Menu::getPadding() {
	return padding;
}

void Menu::renderTexture(GLuint texture, int height, int width, GLfloat rotation) {
	resetQuadStripVertices();
	setQuadStripSize(width, height, quadStripVertices);
	GLfloat tx = (getElemHeight() - width) / 2;
	GLfloat ty = (getElemWidth() - height) / 2;
	glTranslatef(tx, ty, 0);
	rotateQuadStripVertices(rotation, width / 2, height / 2);
	glBindTexture(GL_TEXTURE_2D, texture);
	glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
	glTranslatef(-tx, -ty, 0);
}

void Menu::renderElementBackground(MenuElement* element, GLfloat tx, GLfloat ty) {
	setQuadStripSize(getElemHeight(), getElemWidth(), quadStripVertices);

	if (verticalRotation) {
		rotateQuadStripVertices(rotation, SCREEN_WIDTH / 2 - tx, SCREEN_HEIGHT / 2 - ty);
	} else {
		rotateQuadStripVertices(rotation, getElemHeight() / 2, getElemWidth() / 2);
	}

	glBindTexture(GL_TEXTURE_2D, menuElementBackgroundTexture);
	glVertexPointer(3, GL_FLOAT, 0, quadStripVertices);
	glTexCoordPointer(2, GL_FLOAT, 0, menuElementBackgroundTexCoords);
	glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
}

void Menu::renderElementCaption(MenuElement* element, GLfloat tx, GLfloat ty) {
	TextSurfaceManager* txtMan = TextSurfaceManager::instance();
	int fontSize = 24;
	if (element -> icon == NULL) {
		fontSize = 40;
	} else {
		if (element -> name.size() > 5) {
			fontSize -= (element -> name.size() - 5) / 2;
		}
	}
	TextEntry* entry = txtMan -> getTextEntry(fontSize, element -> name);

	resetQuadStripVertices();
	setQuadStripSize(entry -> height, entry -> width, quadStripVertices);
	GLfloat texttx = 0;
	GLfloat textty = 0;
	if (verticalRotation) {
		rotateQuadStripVertices(rotation, SCREEN_WIDTH / 2 - tx, SCREEN_HEIGHT / 2 - ty);
		texttx += (getElemHeight() - entry -> height) / 2;
		textty += (getElemWidth() - entry -> width) / 2;
		rotate2d(&texttx, &textty, rotation);
		glTranslatef(texttx, textty, 0);
	} else {
		if (element -> icon != NULL) {
			texttx = 50;
			textty = 0;
			rotate2d(&texttx, &textty, rotation);
			texttx += (getElemHeight() - entry -> height) / 2;
			textty += (getElemWidth() - entry -> width) / 2;
			glTranslatef(texttx, textty, 0);
			rotateQuadStripVertices(rotation, entry -> height / 2, entry -> width / 2);
		}
	}

	glTexCoordPointer(2, GL_FLOAT, 0, txtMan -> getTexCoords(entry));
	glBindTexture(GL_TEXTURE_2D, entry -> texture);
	glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
	glTranslatef(-texttx, -textty, 0);
}

void Menu::renderElementIcon(MenuElement* element, GLfloat tx, GLfloat ty) {
	if (element -> texture == 0) {
		renderTexture(busy -> texture.name, busy -> texture.size, busy -> texture.size, nowMillis / 3.0);
		element -> createTexture();
	} else {
		glTexCoordPointer(2, GL_FLOAT, 0, element -> getIconTexCoords());
		renderTexture(element -> texture, element -> getIconSize(), element -> getIconSize(), rotation);
	}
}


void Menu::render() {
	if (!active) {
		return;
	}

	if (options.orientation == LANDSCAPE) {
		rotationAnimation -> setTargetState(rotationStateLandscape);
		rotationAnimation -> setPaused(false);
	} else {
		rotationAnimation -> setTargetState(emptyState);
		rotationAnimation -> setPaused(false);
	}

	QListIterator<MenuElement*> iter(*elements);
	int i = 0;
	glTranslatef(-SCREEN_WIDTH / 2, -SCREEN_HEIGHT / 2, 0);

	while (iter.hasNext()) {
		resetQuadStripVertices();
		MenuElement* element = iter.next();
		GLfloat tx = getElemPositionX(element, i);
		GLfloat ty = getElemPositionY(element, i);

		if (tx < -getElemHeight() || tx > SCREEN_WIDTH) {
			i++;
			continue;
		}

		GLfloat color = element -> transparency;

		if (tx < -getElemHeight() / 2) {
			color *= (tx + getElemHeight()) / (getElemHeight() / 2);
		}
		if (tx > SCREEN_WIDTH - getElemHeight() / 2) {
			color *= 1.0 - (tx - SCREEN_WIDTH + getElemHeight() / 2) / (getElemHeight() / 2);
		}

		if (element -> isHighlighted()) {
			glColor4f(color / 3.0, color / 3.0, color, color);
		} else {
			glColor4f(color, color, color, color);
		}

		glTranslatef(tx, ty, 0);
		if (renderBackground) {
			renderElementBackground(element, tx, ty);
		}
		renderElementCaption(element, tx, ty);

		glColor4f(color, color, color, color);
		if (renderIcon) {
			renderElementIcon(element, tx, ty);
		}
		glTranslatef(-tx, -ty, 0);
		i++;
	}

	resetQuadStripVertices();
	glTranslatef(SCREEN_WIDTH / 2, SCREEN_HEIGHT / 2, 0);

	if (options.orientation == LANDSCAPE && verticalRotation) {
		positionDelta += mouse[0].ydelta / 5.0;
	} else {
		positionDelta += mouse[0].xdelta / 5.0;
	}
	position += positionDelta;
	positionDelta *= 0.9;

	GLfloat minPosition = calculateMinPosition();
	GLfloat maxPosition = calculateMaxPosition();
	if (maxPosition < minPosition) {
		maxPosition = minPosition;
	}

	if (position < minPosition) {
		position -= (position - minPosition) * 0.05;
	} else if (position > maxPosition) {
		position -= (position - maxPosition) * 0.05;
	}
}

void Menu::handleMouse() {
	QListIterator<MenuElement*> iter(*elements);
	elementUnderMouse = NULL;
	int i = 0;

	while (iter.hasNext()) {
		MenuElement* elem = iter.next();
		GLfloat x1 = getElemPositionX(elem, i);
		GLfloat y1 = getElemPositionY(elem, i);

		if (options.orientation == LANDSCAPE && verticalRotation) {
			// TODO not accurate, especially during rotation
			x1 -=  SCREEN_WIDTH / 2;
			y1 -=  SCREEN_HEIGHT / 2;
			rotate2d(&x1, &y1, rotation);
//			x1 +=  SCREEN_WIDTH / 2;
			y1 +=  SCREEN_HEIGHT / 2;

			if (mouse[0].x >= x1 && mouse[0].x <= x1 + getElemWidth() && mouse[0].y >= y1 && mouse[0].y <= y1 + getElemHeight()) {
				elementUnderMouse = elem;
				break;
			}
		} else {
			if (mouse[0].x >= x1 && mouse[0].x <= x1 + getElemHeight() && mouse[0].y >= y1 && mouse[0].y <= y1 + getElemWidth()) {
				elementUnderMouse = elem;
				break;
			}
		}
		i++;
	}

	if (mouse[0].oldButton != 0 && mouse[0].button == 0 && nowMillis - mouse[0].pressed < 200 && !mouse[0].processed) {
		if ((abs(mouse[0].clickedx - mouse[0].x) < 20) && (abs(mouse[0].clickedy - mouse[0].y) < 20)) {
			if (elementUnderMouse != NULL) {
				hide();
				qDebug() << "Running menu action on element: " << elementUnderMouse -> name;
				elementUnderMouse -> action -> run();
				mouse[0].processed = true;
			}
		}
	}

}

bool Menu::isActive() {
	return active;
}

Menu::~Menu() {
}

QList<MenuElement*>* Menu::createOptionsMenuElements() {
	QList<MenuElement*>* elements = new QList<MenuElement*> ();
	elements -> append(new MenuElement("Search service", "/opt/cloudgps/res/menu/options.png", new ShowMenuAction(&canvas.searchServiceMenu)));
	elements -> append(new MenuElement("Routing service", "/opt/cloudgps/res/menu/options.png", new ShowMenuAction(&canvas.routingServiceMenu)));
//	elements -> append(new ToggleOfflineMenuElement("Offline mode", "/opt/cloudgps/res/menu/toggle3d.png", new ToggleOfflineAction()));
	elements -> append(new MenuElement("Back", "/opt/cloudgps/res/menu/back.png", new ShowMenuAction(&canvas.mainMenu)));
	return elements;

}

QList<MenuElement*>* Menu::createMainMenuElements() {
	QList<MenuElement*>* elements = new QList<MenuElement*> ();
	elements -> append(new MenuElement("Change map", "/opt/cloudgps/res/menu/choose_map.png", new ShowMenuAction(&canvas.mapChooseMenu)));
	elements -> append(new MenuElement("Options", "/opt/cloudgps/res/menu/options.png", new ShowMenuAction(&canvas.optionsMenu)));
	elements -> append(new Toggle3dMenuElement("3D view", "/opt/cloudgps/res/menu/toggle3d.png", new Toggle3DAction()));
	elements -> append(new AccelerometerMenuElement("Accelerometer", "/opt/cloudgps/res/menu/toggle_accelerometer.png", new ToggleAccelerometerAction()));
	elements -> append(new MenuElement("Route demo", "/opt/cloudgps/res/menu/route_demo.png", new RouteFlybyAction()));
	elements -> append(new MenuElement("About", "/opt/cloudgps/res/menu/about.png", new AboutAction()));
	elements -> append(new MenuElement("Back", "/opt/cloudgps/res/menu/back.png", new ExitMenuAction()));
	return elements;
}

QList<MenuElement*>* Menu::createMapChooseMenuElements() {
	QList<MenuElement*>* elements = new QList<MenuElement*> ();
	GList *provider = tileProviders;

	while (provider != NULL) {
		TileProvider *tileProvider = (TileProvider*) provider -> data;
		elements -> append(new MapChooseMenuElement(tileProvider));
		provider = provider -> next;
	}
	elements -> append(new MenuElement("Back", "/opt/cloudgps/res/menu/back.png", new ShowMenuAction(&canvas.mainMenu)));
	return elements;
}


QList<MenuElement*>* createProviderMenuElements(GList *list, BackgroundQueryProvider** target) {
	QList<MenuElement*>* elements = new QList<MenuElement*> ();
	GList *providersElem = list;

	while (providersElem != NULL) {
		BackgroundQueryProvider *provider = (BackgroundQueryProvider*) providersElem -> data;
		elements -> append(new ProviderMenuElement(provider, target, new ProviderChangeAction(provider, target)));
		providersElem = providersElem -> next;
	}
	elements -> append(new MenuElement("Back", NULL, new ShowMenuAction(&canvas.optionsMenu)));
	return elements;
}

QList<MenuElement*>* Menu::createSearchServiceMenuElements() {
	return createProviderMenuElements(geocodingProviders, &options.geocodingProvider);
}

QList<MenuElement*>* Menu::createRoutingServiceMenuElements() {
	return createProviderMenuElements(routingProviders, &options.routingProvider);
}


QList<MenuElement*>* Menu::createMapPopupMenuElements() {
	QList<MenuElement*>* elements = new QList<MenuElement*> ();
	elements -> append(new MenuElement("Directions from here", NULL, new MapPopupAction(&directionsFromHere)));
	elements -> append(new MenuElement("Directions to here", NULL, new MapPopupAction(&directionsToHere)));
	elements -> append(new MenuElement("Create marker here", NULL, new MapPopupAction(&createSearchMarkerOnWorldCoordinate)));
	elements -> append(new MenuElement("Clear map", NULL, new MapPopupAction(&clearMap)));
	elements -> append(new MenuElement("Back", NULL, new HidePopupMenuAction()));
	return elements;
}

QList<MenuElement*>* Menu::createKeyboardLanguageMenuElements() {
	QList<MenuElement*>* elements = new QList<MenuElement*> ();
	QList<KeyboardLayout*>* layouts = canvas.virtualKeyboard -> getLayouts();
	QListIterator<KeyboardLayout*> iter(*layouts);

	while (iter.hasNext()) {
		elements -> append(new KeyboardLayoutMenuElement(iter.next()));
	}

	elements -> append(new MenuElement("Back", NULL, new HidePopupMenuAction()));
	return elements;
}
