/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Author: Damian Waradzyn
 */

#include <stdlib.h>
#include <stdio.h>

#include "geocoder.h"
#include "cloudmade_geocoder.h"
#include "google_geocoder.h"
#include "geocoder.h"

volatile SearchResultStatus searchResultsStatus = NO_QUERY;

char* geocodingProviderNames[] = {"CloudMade", "Google"};
GList* geocodingResults;


#define QUERY_TMP_FILENAME "/tmp/cloudgps-query-tmp.js"

/* to_hex and url_encode taken from: http://www.geekhideout.com/urlcode.shtml */

/* Converts an integer value to its hex character*/
char to_hex(char code) {
  static char hex[] = "0123456789abcdef";
  return hex[code & 15];
}

/* Returns a url-encoded version of str */
/* IMPORTANT: be sure to free() the returned string after use */
char *url_encode(char *str) {
  char *pstr = str, *buf = malloc(strlen(str) * 3 + 1), *pbuf = buf;
  while (*pstr) {
    if (isalnum(*pstr) || *pstr == '-' || *pstr == '_' || *pstr == '.' || *pstr == '~')
      *pbuf++ = *pstr;
    else if (*pstr == ' ')
      *pbuf++ = '+';
    else
      *pbuf++ = '%', *pbuf++ = to_hex(*pstr >> 4), *pbuf++ = to_hex(*pstr & 15);
    pstr++;
  }
  *pbuf = '\0';
  return buf;
}

// Download and prepare search results.
int geocodeBgThread(void* query) {
    char *url, *response, *encoded;
    size_t length;

    encoded = url_encode((char*) query);
    free(query);



    char* urlFormat;
    void(*geocoder_parse_response)(char*);

    if (options.geocodingProvider == CLOUDMADE) {
    	urlFormat = CLOUDMADE_QUERY_URL;
    	geocoder_parse_response = cloudmade_parse_response;
    } else if (options.geocodingProvider == GOOGLE) {
    	urlFormat = GOOGLE_QUERY_URL;
    	geocoder_parse_response = google_parse_response;
    } else {
    	fprintf(stderr, "unknown geocoding provider %d\n", options.geocodingProvider);
    	return 0;
    }

    if (asprintf(&url, urlFormat, encoded) < 0) {
    	fprintf(stderr, "asprintf failed in geocode\n");
    	return 0;
    }
//	fprintf(stderr, "Search URL: %s\n", url);
    free(encoded);

    if (downloadAndSave(url, QUERY_TMP_FILENAME)) {
        if (g_file_get_contents(QUERY_TMP_FILENAME, &response, &length, NULL)) {
        	geocoder_parse_response(response);

            g_free(response);
            remove(QUERY_TMP_FILENAME);
        } else {
			searchResultsStatus = DOWNLOAD_ERROR;
            fprintf(stderr, "Unable to read query results from %s\n", QUERY_TMP_FILENAME);
        }
    }
    return 0;
}

void processNewSearchQuery(char* query) {
	if (searchResultsStatus != NO_QUERY) {
		sprintf(strbuf, "Search query: thread busy - try again");
		addConsoleLine(strbuf, 1, 1, 0);
	} else {
		char* copy = malloc(SEARCHBAR_MAX_CHARS);
		strncpy(copy, query, SEARCHBAR_MAX_CHARS);
		SDL_CreateThread(geocodeBgThread, (void*) copy);
	}
}

void processGeocoding() {
	switch (searchResultsStatus) {
	case NO_QUERY:
	case QUERY_IN_PROGRESS:
		break;
	case DOWNLOAD_ERROR:
		sprintf(strbuf, "Search download error");
		addConsoleLine(strbuf, 1, 0, 0);
		break;
	case PARSE_ERROR:
		sprintf(strbuf, "Search parse error");
		addConsoleLine(strbuf, 1, 0, 0);
		break;
	case ZERO_RESULTS:
		sprintf(strbuf, "Search query: no results");
		addConsoleLine(strbuf, 0, 1, 0);
		break;
	case RESULTS_READY:
		sprintf(strbuf, "Search query: %d results", g_list_length(geocodingResults));
		addConsoleLine(strbuf, 0, 1, 0);
	    tileEngineProcessSearchResults(geocodingResults);
		g_list_free(geocodingResults);
		geocodingResults = NULL;
		break;
	}
	searchResultsStatus = NO_QUERY;
}
