/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Author: Damian Waradzyn
 */

#define DOWNLOADING_THREADS 4

SDL_Thread* downloadThreads[DOWNLOADING_THREADS];

int downloadThread(void* queue);
void tileEngine_computeBoundingTrapezoid();
void tileEngine_computeTilesVisibility();

long lastZoomChange = 0;

int inline getCanvasShiftX() {
    return (TILES_X - (SCREEN_WIDTH / TILE_SIZE)) / 2;
}

int inline getCanvasShiftY() {
    return (TILES_Y - (SCREEN_HEIGHT / TILE_SIZE)) / 2;
}

void toScreenCoordinate(TileCoordinate *tc, GLfloat *x, GLfloat *y) {
    *x = (tc->tilex - canvas.tilex - getCanvasShiftX()) * TILE_SIZE + tc->x;
    *y = (tc->tiley - canvas.tiley - getCanvasShiftY()) * TILE_SIZE + tc->y;
}

void toTileCoordinate(WorldCoordinate* wc, TileCoordinate* result, int zoom) {
    result -> zoom = zoom;

    double p = pow(2.0, zoom);
    double lat = (wc->longitude + 180.) / 360.0 * p;
    result -> tilex = (int) lat;
    result ->x = (lat - result->tilex) * TILE_SIZE;

    double lon = (1.0 - log(tan(wc->latitude * M_PI / 180.0) + 1.0 / cos(wc->latitude * M_PI / 180.0)) / M_PI) / 2.0 * p;
    result->tiley = (int) lon;
    result ->y = (lon - result->tiley) * TILE_SIZE;
}

void toWorldCoordinate(TileCoordinate* tc, WorldCoordinate* result) {
    double p = pow(2.0, tc->zoom);
    double xx = (double) tc->tilex + (tc->x / (double) TILE_SIZE);
    result->longitude = ((xx * 360.0) / p) - 180.0;

    double yy = tc->tiley + tc->y / (double) TILE_SIZE;
    double n = M_PI - 2.0 * M_PI * yy / p;
    double m = 0.5 * (exp(n) - exp(-n));
    double a = atan(m);
    result->latitude = (180.0 / M_PI * a);
}

void canvasCenterToTileCoordinate(TileCoordinate* tc) {
    tc -> zoom = canvas.zoom;
    tc -> tilex = canvas.tilex + getCanvasShiftX();
    tc -> tiley = canvas.tiley + getCanvasShiftY();
    tc -> x = (TILE_SIZE - canvas.x) + SCREEN_WIDTH / 2;
    tc -> y = (TILE_SIZE - canvas.y) + SCREEN_HEIGHT / 2;
}

t_tile* createTile(int i, int j) {
    t_tile * tile = (t_tile*) calloc(1, sizeof(t_tile));
    if (tile == NULL) {
        fprintf(stderr, "Memory allocation error in createTile()\n");
        exit(1);
    }
    tile->tilex = canvas.tilex + i;
    tile->tiley = canvas.tiley + j;
    tile->zoom = canvas.zoom;
    tile->provider = canvas.provider;
    tile->stateChangeTime = -1000000;

    g_queue_push_tail(allCreatedTiles, tile);
    return tile;
}

long lastZoomHelperTileCreatedMilis = 0;
void tileEngine_handleZoomHelperTiles() {
    TileCoordinate tcMaxZoom, tc;
    WorldCoordinate wc;
    int i, j, newTile;
    canvasCenterToTileCoordinate(&tcMaxZoom);
    toWorldCoordinate(&tcMaxZoom, &wc);
    toTileCoordinate(&wc, &tcMaxZoom, (MAX_ZOOM_HELPER_TILES - 1));

    // Manage the array of zoom helper tiles - download them when applicable.
    for (i = 0; i < MAX_ZOOM_HELPER_TILES; i++) {
        tc.tilex = tcMaxZoom.tilex / pow(2.0, MAX_ZOOM_HELPER_TILES - i - 1);
        tc.tiley = tcMaxZoom.tiley / pow(2.0, MAX_ZOOM_HELPER_TILES - i - 1);
        tc.zoom = i;
        newTile = FALSE;
        if (currentPositionZoomHelperTiles[i] == NULL) {
            newTile = TRUE;
        } else {
            if (currentPositionZoomHelperTiles[i] -> tilex != tc.tilex || currentPositionZoomHelperTiles[i] -> tiley != tc.tiley
                    || currentPositionZoomHelperTiles[i] -> provider != canvas.provider) {
                currentPositionZoomHelperTiles[i] -> deleted = TRUE;
                newTile = TRUE;
                currentPositionZoomHelperTiles[i] = NULL;
            } else {
                if (currentPositionZoomHelperTiles[i] -> state >= STATE_LOADED && currentPositionZoomHelperTiles[i] -> texture != 0
                        && currentPositionZoomHelperTiles[i] -> textureUsedTime > 0 && nowMillis
                        - currentPositionZoomHelperTiles[i] -> textureUsedTime > 3000) {
                    //                    fprintf(stderr, "del tex %d\n", currentPositionZoomHelperTiles[i] -> texture);
                    glDeleteTextures(1, &currentPositionZoomHelperTiles[i] -> texture);
                    currentPositionZoomHelperTiles[i] -> texture = 0;
                    currentPositionZoomHelperTiles[i] -> state = STATE_GL_TEXTURE_NOT_CREATED;
                    currentPositionZoomHelperTiles[i] -> textureUsedTime = 0;
                }
            }
        }

        // If canvas is moving downloading higher zoom should be avoided.
        if ((fabs(canvas.dx) > 0.2 || fabs(canvas.dy) > 0.2) && tc.zoom > canvas.zoom) {
            newTile = FALSE;
        }

        // Limit amount of helper tiles added to queue per second.
        if (nowMillis - lastZoomHelperTileCreatedMilis < 200) {
            newTile = FALSE;
        }

        if (tc.zoom < canvas.provider -> minZoom || tc.zoom > canvas.provider -> maxZoom) {
            newTile = FALSE;
        }

        if (newTile) {
            currentPositionZoomHelperTiles[i] = createTile(0, 0);
            currentPositionZoomHelperTiles[i] -> state = STATE_QUEUED;
            currentPositionZoomHelperTiles[i] -> visible = FALSE;
            currentPositionZoomHelperTiles[i] -> zoomHelper = TRUE;
            currentPositionZoomHelperTiles[i] -> tilex = tc.tilex;
            currentPositionZoomHelperTiles[i] -> tiley = tc.tiley;
            currentPositionZoomHelperTiles[i] -> zoom = tc.zoom;
            enqueue(downloadQueue, currentPositionZoomHelperTiles[i]);
            lastZoomHelperTileCreatedMilis = nowMillis;
        }
    }

    // Go through all currently displayed tiles and set oldtile where applicable.
    for (i = 0; i < TILES_X; i++) {
        for (j = 0; j < TILES_Y; j++) {
            if (tiles[i][j][currentTilesIdx] != NULL) {
                if (tiles[i][j][currentTilesIdx] -> texture == 0 && tiles[i][j][currentTilesIdx] -> oldTiles[0] == NULL) {
                    int k, l = 1;
                    for (k = (tiles[i][j][currentTilesIdx] -> zoom); k >= 0; k--, l *= 2) {
                        if (k < MAX_ZOOM_HELPER_TILES) {
                            if (currentPositionZoomHelperTiles[k] != NULL) {
                                if (tiles[i][j][currentTilesIdx] -> tilex >= 0 && tiles[i][j][currentTilesIdx] -> tiley >= 0
                                        && (tiles[i][j][currentTilesIdx] -> tilex / l == currentPositionZoomHelperTiles[k] -> tilex)
                                        && (tiles[i][j][currentTilesIdx] -> tiley / l == currentPositionZoomHelperTiles[k] -> tiley)) {
                                    if (currentPositionZoomHelperTiles[k] -> state >= STATE_LOADED) {
                                        if (currentPositionZoomHelperTiles[k] -> texture == 0) {
                                            if (texturesCreatedThisFrame >= MAX_TEXTURES_CREATED_PER_FRAME) {
                                                continue;
                                            }
                                            currentPositionZoomHelperTiles[k] -> texture = createTexture(
                                                    currentPositionZoomHelperTiles[k] -> pixels4444, TILE_SIZE, TILE_SIZE, TRUE);
                                            currentPositionZoomHelperTiles[k] -> state = STATE_LOADED;
                                            texturesCreatedThisFrame++;
                                        }
                                    }
                                    tiles[i][j][currentTilesIdx] -> oldTiles[0] = currentPositionZoomHelperTiles[k];
                                    break;
                                }
                            }
                        }
                    }
                }

                // If the queueing was delayed enqueue now.
                if (tiles[i][j][currentTilesIdx] -> state == STATE_QUEUE_DELAYED && nowMillis - lastZoomChange > 1500) {
                    tiles[i][j][currentTilesIdx] -> state = STATE_QUEUED;
                    enqueue(downloadQueue, tiles[i][j][currentTilesIdx]);
                }
            }
        }
    }
}

void initTileEngine() {
    int i, j;

    downloadQueue = createQueue();
    allCreatedTiles = g_queue_new();

    canvas.scale = 1.0;
    canvas.viewMode = VIEW_2D;

    for (i = 0; i < DOWNLOADING_THREADS; i++) {
        //        fprintf(stderr, "before SDL_CreateThread\n");
        downloadThreads[i] = SDL_CreateThread(downloadThread, (void*) downloadQueue);
        //        fprintf(stderr, "after SDL_CreateThread\n");
        SDL_Delay(15);
    }

    for (i = 0; i < TILES_X; i++) {
        for (j = 0; j < TILES_Y; j++) {
            tiles[i][j][currentTilesIdx] = createTile(i, j);
        }
    }
    tileEngine_computeBoundingTrapezoid();
    tileEngine_computeTilesVisibility();
    for (i = 0; i < TILES_X; i++) {
        for (j = 0; j < TILES_Y; j++) {
            enqueue(downloadQueue, tiles[i][j][currentTilesIdx]);
        }
    }
    tileEngine_handleZoomHelperTiles();
}

int inline roundDown(double x) {
    return x < 0 ? (int) (x - 1.0) : (int) x;
}

int adjustShifts() {
    int horizontalShift = roundDown(canvas.x / (double) TILE_SIZE);
    int verticalShift = roundDown(canvas.y / (double) TILE_SIZE);

    canvas.x -= horizontalShift * TILE_SIZE;
    canvas.y -= verticalShift * TILE_SIZE;

    canvas.tilex += horizontalShift;
    canvas.tiley += verticalShift;
    if (horizontalShift != 0 || verticalShift != 0) {
        return TRUE;
    }
    return FALSE;
}

void recreateTiles() {
    purgeQueue(downloadQueue);
    int i, j, k;

    // start loading new tile textures:
    //    syncLoadedTilesAppearance = TRUE;
    for (i = 0; i < TILES_X; i++) {
        for (j = 0; j < TILES_Y; j++) {
            tiles[i][j][1 - currentTilesIdx] = createTile(i, j);
            if (tiles[i][j][currentTilesIdx] != NULL) {
                if (tiles[i][j][currentTilesIdx] -> texture != 0) {
                    tiles[i][j][1 - currentTilesIdx] -> oldTiles[0] = tiles[i][j][currentTilesIdx];
                    tiles[i][j][1 - currentTilesIdx] -> stateChangeTime = nowMillis;
                } else {
                    for (k = 0; k < 4; k++) {
                        tiles[i][j][1 - currentTilesIdx] -> oldTiles[k] = tiles[i][j][currentTilesIdx]-> oldTiles[k];
                    }
                }
            }
        }
    }
    currentTilesIdx = 1 - currentTilesIdx;
    tileEngine_computeBoundingTrapezoid();
    tileEngine_computeTilesVisibility();
    for (i = 0; i < TILES_X; i++) {
        for (j = 0; j < TILES_Y; j++) {
            enqueue(downloadQueue, tiles[i][j][currentTilesIdx]);
        }
    }
}

void handleZoomChange() {
    // compute new tilex, tiley, x and y for canvas using canvas.center
    int i, j;

    TileCoordinate tc;

    toTileCoordinate(&canvas.center, &tc, canvas.zoom);

    canvas.tilex = tc.tilex - getCanvasShiftX();
    canvas.tiley = tc.tiley - getCanvasShiftY();
    canvas.x = tc.x - SCREEN_WIDTH / 2;
    canvas.y = tc.y - SCREEN_HEIGHT / 2;

    adjustShifts();

    if (canvas.x != 0.0) {
        canvas.x = TILE_SIZE - canvas.x;
    }
    if (canvas.y != 0.0) {
        canvas.y = TILE_SIZE - canvas.y;
    }

    //    syncLoadedTilesAppearance = TRUE;
    lastZoomChange = nowMillis;

    if (canvas.zoom - canvas.oldZoom == 1) {
        //        fprintf(stderr, "zoomIn 1\n");
        int oldtilex = tiles[0][0][currentTilesIdx] -> tilex;
        int oldtiley = tiles[0][0][currentTilesIdx] -> tiley;

        for (i = 0; i < TILES_X; i++) {
            for (j = 0; j < TILES_Y; j++) {
                tiles[i][j][1 - currentTilesIdx] = createTile(i, j);

                int oldi = tiles[i][j][1 - currentTilesIdx] -> tilex - (oldtilex * 2);
                int oldj = tiles[i][j][1 - currentTilesIdx] -> tiley - (oldtiley * 2);

                if (oldi >= 0 && oldj >= 0) {

                    if (tiles[oldi / 2][oldj / 2][currentTilesIdx] != NULL && tiles[oldi / 2][oldj / 2][currentTilesIdx] -> state >= STATE_LOADED) {
                        tiles[i][j][1 - currentTilesIdx] -> oldTiles[0] = tiles[oldi / 2][oldj / 2][currentTilesIdx];

                        if (tiles[oldi / 2][oldj / 2][currentTilesIdx] -> state == STATE_GL_TEXTURE_NOT_CREATED) {
                            tiles[oldi / 2][oldj / 2][currentTilesIdx] -> texture = createTexture(
                                    tiles[oldi / 2][oldj / 2][currentTilesIdx] -> pixels4444, TILE_SIZE, TILE_SIZE, TRUE);
                        }

                        tiles[oldi / 2][oldj / 2][currentTilesIdx] -> state = STATE_SCALED;
                        tiles[i][j][1 - currentTilesIdx] -> transitionTime = options.zoomChangeFadeTime;
                        if (canvas.zoomBarLastDragged - nowMillis < 500) {
                            tiles[i][j][1 - currentTilesIdx] -> state = STATE_QUEUE_DELAYED;
                        } else {
                            tiles[i][j][1 - currentTilesIdx] -> state = STATE_QUEUED;
                            enqueue(downloadQueue, tiles[i][j][1 - currentTilesIdx]);
                        }
                    } else {
                        tiles[oldi / 2][oldj / 2][currentTilesIdx] = NULL;
                        tiles[i][j][1 - currentTilesIdx] -> transitionTime = options.zoomChangeFadeTime;
                        if (canvas.zoomBarLastDragged - nowMillis < 500) {
                            tiles[i][j][1 - currentTilesIdx] -> state = STATE_QUEUE_DELAYED;
                        } else {
                            tiles[i][j][1 - currentTilesIdx] -> state = STATE_QUEUED;
                            enqueue(downloadQueue, tiles[i][j][1 - currentTilesIdx]);
                        }
                    }
                }
            }
        }
        currentTilesIdx = 1 - currentTilesIdx;
        if (canvas.destScale != 0.0) {
            canvas.destScale /= 2.0;
        }
    } else if (canvas.zoom - canvas.oldZoom == -1) {
        for (i = 0; i < TILES_X; i++) {
            for (j = 0; j < TILES_Y; j++) {
                tiles[i][j][1 - currentTilesIdx] = createTile(i, j);
                if (canvas.zoomBarLastDragged - nowMillis < 500) {
                    tiles[i][j][1 - currentTilesIdx] -> state = STATE_QUEUE_DELAYED;
                } else {
                    tiles[i][j][1 - currentTilesIdx] -> state = STATE_QUEUED;
                    enqueue(downloadQueue, tiles[i][j][1 - currentTilesIdx]);
                }
                tiles[i][j][1 - currentTilesIdx] -> transitionTime = options.zoomChangeFadeTime;
            }
        }
        currentTilesIdx = 1 - currentTilesIdx;
        if (canvas.destScale != 0.0) {
            canvas.destScale *= 2.0;
        }
    } else {
        // Normally not used because changes to zoom level are supported only by one level at a time.
        // Useful for changing between tile providers that have have no zoom level common.
        recreateTiles();
    }
    forceGarbageCollection = TRUE;
}

int inline getLowerDeallocateBound(int size, int shift) {
    return shift < 0 ? 0 : (size - shift) < 0 ? 0 : (size - shift);
}

int inline getUpperDeallocateBound(int size, int shift) {
    return shift < 0 ? (-shift < size ? -shift : size) : size;
}

void deallocateTiles(int horizontalShift, int verticalShift) {

    int i, j;
    for (i = getLowerDeallocateBound(TILES_X, horizontalShift); i < getUpperDeallocateBound(TILES_X, horizontalShift); i++) {
        for (j = 0; j < TILES_Y; j++) {
            //            deallocateTile(tiles[i][j][currentTilesIdx]);
            tiles[i][j][currentTilesIdx] = NULL;
        }
    }
    for (i = 0; i < TILES_X; i++) {
        for (j = getLowerDeallocateBound(TILES_Y, verticalShift); j < getUpperDeallocateBound(TILES_Y, verticalShift); j++) {
            //            deallocateTile(tiles[i][j][currentTilesIdx]);
            tiles[i][j][currentTilesIdx] = NULL;
        }
    }
}

void updateTiles(int horizontalShift, int verticalShift) {
    deallocateTiles(horizontalShift, verticalShift);
    int i, j;
    for (i = 0; i < TILES_X; i++) {
        for (j = 0; j < TILES_Y; j++) {
            int newPosX = i - horizontalShift;
            int newPosY = j - verticalShift;
            if (newPosX >= 0 && newPosX < TILES_X && newPosY >= 0 && newPosY < TILES_Y) {
                tiles[i][j][1 - currentTilesIdx] = tiles[newPosX][newPosY][currentTilesIdx];
                tiles[newPosX][newPosY][currentTilesIdx] = NULL;
            } else {
                tiles[i][j][1 - currentTilesIdx] = createTile(i, j);
                enqueue(downloadQueue, tiles[i][j][1 - currentTilesIdx]);
            }
        }
    }
    currentTilesIdx = 1 - currentTilesIdx;
}

void tileEngineProcessMouse(int uiElemPressed) {
    canvas.dx *= 0.97;
    canvas.dy *= 0.97;

    if (uiElemPressed == FALSE) {
        if (mouse.button) {
            canvas.followingMypos = 0;
            canvas.attractionToPoint = 0;
            if (canvas.viewMode == VIEW_3D || canvas.rotation2d == 1) {
                canvas.attractedToRotZ = 0;
            }
        }

        if (canvas.viewMode == VIEW_3D || canvas.rotation2d == 1) {
            if (options.orientation == PORTRAIT && mouse.x < 150) {
                canvas.drotz -= mouse.ydelta / 18.0;
                return;
            }
            if (options.orientation == LANDSCAPE && mouse.y < 150) {
                canvas.drotz += mouse.xdelta / 25.0;
                return;
            }
        }

        canvas.dx += mouse.ydelta / 10.0 * sin(canvas.rotz * M_PI / 180.0);
        canvas.dy += mouse.ydelta / 10.0 * cos(canvas.rotz * M_PI / 180.0);
        canvas.dx += mouse.xdelta / 10.0 * sin((canvas.rotz + 90) * M_PI / 180.0);
        canvas.dy += mouse.xdelta / 10.0 * cos((canvas.rotz + 90) * M_PI / 180.0);

        if (mouse.pressed > 0 && nowMillis - mouse.pressed > 30) {
            canvas.dx *= 0.97;
            canvas.dy *= 0.97;
        }

        if (mouse.pressed > 0 && nowMillis - mouse.moved > 50) {
            canvas.dx *= 0.2;
            canvas.dy *= 0.2;
        }
    }
}
void tileEngineChangeOrientation(Orientation orientation);

void tileEngineProcessAccelerometer() {
    if (options.accelerometerEnabled == 1) {

        if (canvas.viewMode == VIEW_2D) {
            if (options.orientation == LANDSCAPE) {
                if (abs(accelerometer.x) > 70) {
                    canvas.dx += accelerometer.x / 1000.0 * sin((canvas.rotz + 90) * M_PI / 180.0);
                    canvas.dy += accelerometer.x / 1000.0 * cos((canvas.rotz + 90) * M_PI / 180.0);
                }
                if (abs(accelerometer.y) > 70) {
                    canvas.dx += accelerometer.y / 1000.0 * sin(canvas.rotz * M_PI / 180.0);
                    canvas.dy += accelerometer.y / 1000.0 * cos(canvas.rotz * M_PI / 180.0);
                }
            } else {
                if (abs(accelerometer.y) > 70) {
                    canvas.dx += accelerometer.y / 1000.0 * sin(canvas.rotz * M_PI / 180.0);
                    canvas.dy += accelerometer.y / 1000.0 * cos(canvas.rotz * M_PI / 180.0);
                }
                if (abs(accelerometer.x) > 70) {
                    canvas.dx += accelerometer.x / 1000.0 * sin((canvas.rotz + 90) * M_PI / 180.0);
                    canvas.dy += accelerometer.x / 1000.0 * cos((canvas.rotz + 90) * M_PI / 180.0);
                }
            }
        } else {
            if (options.orientation == LANDSCAPE) {
                if (abs(accelerometer.y) > 70) {
                    canvas.dx += accelerometer.y / 1000.0 * sin(canvas.rotz * M_PI / 180.0);
                    canvas.dy += accelerometer.y / 1000.0 * cos(canvas.rotz * M_PI / 180.0);
                }
            } else {
                if (abs(accelerometer.x) > 70) {
                    canvas.dx += accelerometer.x / 1000.0 * sin((canvas.rotz + 90) * M_PI / 180.0);
                    canvas.dy += accelerometer.x / 1000.0 * cos((canvas.rotz + 90) * M_PI / 180.0);
                }
            }
            if (options.orientation == LANDSCAPE) {
                canvas.drotz += accelerometer.x / 2000.0;
                if (abs(accelerometer.x) > 70) {
                    canvas.attractedToRotZ = 0;
                }
            } else {
                canvas.drotz -= accelerometer.y / 2000.0;
                if (abs(accelerometer.y) > 70) {
                    canvas.attractedToRotZ = 0;
                }
            }
        }
    }
    if (accelerometer.y + accelerometer.calibrateY < -850) {
        tileEngineChangeOrientation(LANDSCAPE);
    }
    if (accelerometer.x + accelerometer.calibrateX < -850) {
        tileEngineChangeOrientation(PORTRAIT);
    }
}

void updateCanvasCenterWorldCoordinate() {
    TileCoordinate tc;
    canvasCenterToTileCoordinate(&tc);

    int horizontalShift = roundDown(tc.x / (double) TILE_SIZE) - getCanvasShiftX();
    int verticalShift = roundDown(tc.y / (double) TILE_SIZE) - getCanvasShiftY();

    tc.x -= horizontalShift * TILE_SIZE;
    tc.y -= verticalShift * TILE_SIZE;

    tc.tilex += horizontalShift;
    tc.tiley += verticalShift;

    toWorldCoordinate(&tc, &canvas.center);
}

void tileEngineZoomKnotPressed() {
    canvas.zoomBarLastDragged = nowMillis;
    canvas.zoomBarVisibleMilis = nowMillis + 10000;
}

// Calculates zoom level from zoom knot position.
int getKnotDestZoom() {
    GLfloat screenSize = canvas.orientationTransitionLinear < 80 ? SCREEN_WIDTH : SCREEN_HEIGHT;
    int zoomLevelCount = canvas.provider -> maxZoom - canvas.provider -> minZoom + 1;
    return (canvas.orientationTransitionLinear < 80 ? zoomKnot -> x - 82 : (SCREEN_HEIGHT - zoomKnot -> y - 82 - 64)) / ((screenSize - 232)
            / zoomLevelCount) + canvas.provider -> minZoom;
}

void tileEngineZoomKnotDragged() {
    canvas.zoomBarActive = TRUE;
    canvas.zoomBarLastDragged = nowMillis;
    canvas.zoomBarVisibleMilis = nowMillis + 7000;
    canvas.zoomKnotPosition += canvas.orientationTransitionLinear < 80 ? mouse.x - mouse.oldx : mouse.y - mouse.oldy;
    if (canvas.zoomKnotPosition < 100 - 18) {
        canvas.zoomKnotPosition = 100 - 18;
    }
    int screenSize = canvas.orientationTransitionLinear < 80 ? SCREEN_WIDTH : SCREEN_HEIGHT;
    if (canvas.zoomKnotPosition > screenSize - 100 - 32 - 14) {
        canvas.zoomKnotPosition = screenSize - 100 - 32 - 14;
    }
    if (canvas.zoom != getKnotDestZoom()) {
        canvas.destScale = pow(2.0, getKnotDestZoom() - canvas.zoom);
    }
}

void tileEngineToggleSearchBar() {
    canvas.searchBarActive = 1 - canvas.searchBarActive;
}

void tileEngineUpdateCoordinates() {
    //int zoomChange = (int) round(log2((double) canvas.scale));
    int zoomChange = 0;

    canvas.x += canvas.dx;
    canvas.y += canvas.dy;

    if (canvas.followingMypos) {
        if (device -> fix -> fields | LOCATION_GPS_DEVICE_LATLONG_SET) {
            canvas.attraction.latitude = device -> fix -> latitude;
            canvas.attraction.longitude = device -> fix -> longitude;
            canvas.attractionToPoint = 1;
        }
        if (canvas.viewMode == VIEW_3D || canvas.rotation2d == 1) {
            if ((device -> fix -> fields | LOCATION_GPS_DEVICE_TRACK_SET) && device -> satellites_in_use > 0 && device -> fix -> speed > 3) {
                if (options.orientation == LANDSCAPE) {
                    canvas.destRotz = -device -> fix -> track;
                } else {
                    canvas.destRotz = 270 - device -> fix -> track;
                }
                canvas.attractedToRotZ = 1;
            }
        }
    }

    if (canvas.attractionToPoint) {
        GLfloat x, y;
        TileCoordinate tc;
        toTileCoordinate(&canvas.attraction, &tc, canvas.zoom);
        toScreenCoordinate(&tc, &x, &y);
        //            fprintf(stderr, "x = %f, y = %f\t\ttilex = %d, tiley = %d, tc.x = %f, tc.y = %f \n", x, y, tc.tilex, tc.tiley, tc.x, tc.y);
        canvas.dx = (SCREEN_WIDTH / 2.0 - (canvas.x - TILE_SIZE + x)) / 20.0;
        canvas.dy = (SCREEN_HEIGHT / 2.0 - (canvas.y - TILE_SIZE + y)) / 20.0;
    }

    if (canvas.destScale != 0.0) {
        GLfloat slowingDownSpeedFactor = 10.0;
        GLfloat tmp = canvas.destScale / canvas.scale;
        canvas.oldScale = canvas.scale;

        //        GLfloat maxSpeedBorder = 1.0 + ((canvas.maxScaleSpeed - 1.0) * slowingDownSpeedFactor);
        //        if (tmp > maxSpeedBorder) {
        //            canvas.scale *= canvas.maxScaleSpeed;
        //} else if (tmp < 1.0 / maxSpeedBorder || tmp < 0.) {
        //    canvas.scale /= canvas.maxScaleSpeed;
        //        } else {
        if (tmp < -slowingDownSpeedFactor + 1.0) {
            canvas.scale = canvas.destScale;
            canvas.destScale = 0.0;
        } else {
            canvas.scale *= 1.0 + (tmp - 1.0) / slowingDownSpeedFactor;
        }
        //        }

        if (fabs(tmp - 1.0) < 0.01 && (fabs(canvas.destScale - 1.0) < 0.01 || fabs(canvas.destScale - 2.0) < 0.01 || fabs(canvas.destScale - .5)
                < 0.01)) {
            canvas.scale = canvas.destScale;
            canvas.destScale = 0.0;
        }
        //fprintf(stderr, "%d\t%f\t%f\n", canvas.zoom, canvas.oldScale, canvas.scale);
    }
    if (canvas.scale < 1.0) {
        zoomChange = -1;
    }

    if (canvas.scale >= 2.0) {
        zoomChange = 1;
    }

    if ((canvas.zoom + zoomChange) < canvas.provider -> minZoom || (canvas.zoom + zoomChange) > canvas.provider -> maxZoom) {
        zoomChange = 0;
    }

    if (zoomChange == 0) {
        int horizontalShift = roundDown(canvas.x / TILE_SIZE);
        int verticalShift = roundDown(canvas.y / TILE_SIZE);

        if (horizontalShift != 0 || verticalShift != 0) {
            canvas.x -= horizontalShift * TILE_SIZE;
            canvas.y -= verticalShift * TILE_SIZE;

            canvas.tilex -= horizontalShift;
            canvas.tiley -= verticalShift;
            updateTiles(horizontalShift, verticalShift);
        }
    }

    updateCanvasCenterWorldCoordinate();

    if (zoomChange != 0 /* && nowMillis - lastZoomChangeMilis > 500 */) {
        canvas.oldZoom = canvas.zoom;
        canvas.zoom += zoomChange;
        handleZoomChange();
        //        if (zoomChange > 0) {
        //            canvas.scale /= 4.0/3.0;
        //        } else {
        //            canvas.scale *= 4.0/3.0;
        //        }
        //        canvas.scale /= pow(2.0, zoomChange);
        if (zoomChange == 1) {
            canvas.scale /= 2.0;
        } else if (zoomChange == -1) {
            canvas.scale *= 2.0;
        }
    }

    canvas.rotx += (canvas.destRotx - canvas.rotx) * 0.03;
    canvas.roty += (canvas.destRoty - canvas.roty) * 0.03;
    canvas.orientationTransition += ((GLfloat) options.orientation - canvas.orientationTransition) * 0.03;
    if (options.orientation == LANDSCAPE) {
        if (canvas.orientationTransitionLinear > 0) {
            canvas.orientationTransitionLinear--;
        }
    } else {
        if (canvas.orientationTransitionLinear < 160) {
            canvas.orientationTransitionLinear++;
        }
    }
    if (canvas.attractedToRotZ) {
        double dist = canvas.destRotz - canvas.rotz, dist2 = 180.0;
        if (canvas.rotz < 0.0 && canvas.destRotz > 0.0) {
            dist2 = -(180.0 + canvas.rotz) - (180.0 - canvas.destRotz);
        } else if (canvas.rotz > 0.0 && canvas.destRotz < 0.0) {
            dist2 = 180 - canvas.rotz + (180.0 + canvas.destRotz);
        }
        if (fabs(dist) < fabs(dist2)) {
            canvas.rotz += dist * 0.03;
        } else {
            canvas.rotz += dist2 * 0.03;
        }
    } else {
        canvas.rotz += canvas.drotz;
        canvas.drotz *= 0.9;
    }
    if (canvas.rotz > 180) {
        canvas.rotz -= 360;
    }
    if (canvas.rotz < -180) {
        canvas.rotz += 360;
    }
    if (canvas.orientationTransition > 0.999) {
        canvas.orientationTransition = 1.0;
    } else if (canvas.orientationTransition < 0.001) {
        canvas.orientationTransition = 0.0;
    }
}

void activateZoomBar() {
    canvas.zoomBarActive = TRUE;
    canvas.zoomBarVisibleMilis = nowMillis + 3000;
    int zoomLevelCount = canvas.provider -> maxZoom - canvas.provider -> minZoom + 2;
    GLfloat knotZoomLevel = canvas.zoom - canvas.provider -> minZoom;
    if (canvas.destScale != 0.0) {
        knotZoomLevel += round(log2(canvas.destScale));
    }
    if (canvas.orientationTransitionLinear < 80) {
        canvas.zoomKnotPosition = knotZoomLevel * (SCREEN_WIDTH - 228) / zoomLevelCount + 114;
    } else {
        canvas.zoomKnotPosition = -(knotZoomLevel * (SCREEN_HEIGHT - 232) / zoomLevelCount) + SCREEN_HEIGHT - 114-32-14;
    }
}

long zoomInLastMilis = 0;
void tileEngineZoomIn() {
    if (SDL_GetModState() & KMOD_SHIFT) {
        canvas.oldScale = canvas.scale;
        canvas.scale *= 1.005;
    } else {
        if (nowMillis - zoomInLastMilis > 100) {
            if (canvas.destScale == 0.0) {
                if (canvas.scale <= 0.5) {
                    canvas.destScale = 1.0;
                } else {
                    canvas.destScale = 2.0;
                }
            } else if (canvas.scale < 2.0 && canvas.destScale <= 2.0 && canvas.zoom < canvas.provider -> maxZoom) {
                canvas.destScale *= 2.0;
            }
            zoomInLastMilis = nowMillis;
            if (mouse.oldButton == 0 && pressedUiElem == zoomIn) {
                zoomInLastMilis += 400;
            }
            activateZoomBar();
        }
    }
}

long zoomOutLastMilis = 0;
void tileEngineZoomOut() {
    if (SDL_GetModState() & KMOD_SHIFT) {
        canvas.oldScale = canvas.scale;
        canvas.scale /= 1.005;
    } else {
        if (nowMillis - zoomOutLastMilis > 100) {
            if (canvas.destScale == 0.0) {
                if (canvas.scale >= 2.0) {
                    canvas.destScale = 1.0;
                } else {
                    canvas.destScale = 0.5;
                }
            } else if (canvas.scale > 0.25 && canvas.destScale >= 0.25 && canvas.zoom > canvas.provider -> minZoom) {
                canvas.destScale /= 2.0;
            }
            zoomOutLastMilis = nowMillis;
            if (mouse.oldButton == 0 && pressedUiElem == zoomOut) {
                zoomOutLastMilis += 400;
            }
            activateZoomBar();
        }
    }
}

void tileEngineGotomypos() {
    if (mouse.oldButton == 0) {
        canvas.followingMypos = 1 - canvas.followingMypos;
        if (canvas.followingMypos) {
            updateCanvasCenterWorldCoordinate();
            canvas.previousCenter.latitude = canvas.center.latitude;
            canvas.previousCenter.longitude = canvas.center.longitude;
        }
    }
}

void tileEngineViewMode2D() {
    view2d -> status = UI_HIDDEN;
    view3d -> status = UI_SHOWN;
    canvas.viewMode = VIEW_2D;
    if (options.orientation == LANDSCAPE) {
        canvas.destRotx = 0.0;
        canvas.destRoty = 0.0;
    } else {
        canvas.destRotx = 0.0;
        canvas.destRoty = 0.0;
    }
    if (canvas.rotation2d == 0) {
        if (options.orientation == LANDSCAPE) {
            canvas.destRotz = 0.0;
        } else {
            canvas.destRotz = -90.0;
        }
        canvas.attractedToRotZ = 1;
    }
}

void tileEngineViewMode3D() {
    view2d -> status = UI_SHOWN;
    view3d -> status = UI_HIDDEN;
    canvas.viewMode = VIEW_3D;
    if (options.orientation == LANDSCAPE) {
        canvas.destRotx = LANDSCAPE_ROTATION_X;
        canvas.destRoty = 0.0;
    } else {
        canvas.destRotx = 0.0;
        canvas.destRoty = -PORTRAIT_ROTATION_Y;
    }
}

int zoomBarInitialized = FALSE;
void tileEngineChangeOrientation(Orientation orientation) {
    if (orientation != options.orientation) {
        if (canvas.viewMode == VIEW_2D && canvas.rotation2d == 0) {
            if (orientation == LANDSCAPE) {
                canvas.destRotz = 0.0;
            } else {
                canvas.destRotz = -90.0;
            }
        } else {
            if (orientation == LANDSCAPE) {
                canvas.destRotz = canvas.rotz + 90;
                if (canvas.destRotz > 180) {
                    canvas.destRotz -= 360;
                }
            } else {
                canvas.destRotz = canvas.rotz - 90;
                if (canvas.destRotz < -180) {
                    canvas.destRotz += 360;
                }
            }
        }
        options.orientation = orientation;
        if (canvas.viewMode == VIEW_2D) {
            tileEngineViewMode2D();
        } else {
            tileEngineViewMode3D();
        }
        canvas.attractedToRotZ = 1;
        canvas.zoomBarActive = 0;
    }
}

void tileEngineToggle2dRotation() {
    if (canvas.viewMode == VIEW_2D) {
        canvas.rotation2d = 1 - canvas.rotation2d;
        if (canvas.rotation2d == 0) {
            canvas.attractedToRotZ = 1;
            if (options.orientation == LANDSCAPE) {
                canvas.destRotz = 0.0;
            } else {
                canvas.destRotz = -90.0;
            }
        } else {
            canvas.attractedToRotZ = 0;
        }
    } else {
        if (options.orientation == LANDSCAPE) {
            canvas.destRotz = 0.0;
        } else {
            canvas.destRotz = -90.0;
        }
        canvas.attractedToRotZ = 1;
    }
}

void shutdownTileEngine() {
    int i;
    for (i = 0; i < DOWNLOADING_THREADS; i++) {
        //        fprintf(stderr, "waiting for download thread %d to finish...\n", SDL_GetThreadID(downloadThreads[i]));
        //SDL_WaitThread(downloadThreads[i], NULL);
    }
}

void tileEngine_computeBoundingTrapezoid() {
    GLfloat posx, posy, posz;

    // tile plane z / (far plane z - near plane z)
    // tile plane z / (far plane z - near plane z)
    GLfloat z1 = 350.0 / (600.0 - 50.0);
    GLfloat z = z1 - canvas.roty / 15.0;
    if (z > 1.0) {
        z = 1.0;
    }

    gluUnProject(0, 0, z, modelMatrix, projectionMatrix, viewPort, &posx, &posy, &posz);
    canvas.boundingTrapezoid[0].tilex = canvas.tilex + getCanvasShiftX();
    canvas.boundingTrapezoid[0].tiley = canvas.tiley + getCanvasShiftY();
    canvas.boundingTrapezoid[0].x = posx;
    canvas.boundingTrapezoid[0].y = posy;
    canvas.boundingTrapezoid[0].zoom = canvas.zoom;

    z = z1;

    gluUnProject(SCREEN_WIDTH - 1, 0, z, modelMatrix, projectionMatrix, viewPort, &posx, &posy, &posz);

    canvas.boundingTrapezoid[1].tilex = canvas.tilex + getCanvasShiftX();
    canvas.boundingTrapezoid[1].tiley = canvas.tiley + getCanvasShiftY();
    canvas.boundingTrapezoid[1].x = posx;
    canvas.boundingTrapezoid[1].y = posy;
    canvas.boundingTrapezoid[1].zoom = canvas.zoom;

    z = z1 + canvas.rotx / 40.0;
    if (z > 1.0) {
        z = 1.0;
    }

    gluUnProject(SCREEN_WIDTH - 1, SCREEN_HEIGHT - 1, z, modelMatrix, projectionMatrix, viewPort, &posx, &posy, &posz);

    canvas.boundingTrapezoid[2].tilex = canvas.tilex + getCanvasShiftX();
    canvas.boundingTrapezoid[2].tiley = canvas.tiley + getCanvasShiftY();
    canvas.boundingTrapezoid[2].x = posx;
    canvas.boundingTrapezoid[2].y = posy;
    canvas.boundingTrapezoid[2].zoom = canvas.zoom;

    z = z1 + canvas.rotx / 40.0 - canvas.roty / 15.0;
    if (z > 1.0) {
        z = 1.0;
    }

    gluUnProject(0, SCREEN_HEIGHT - 1, z, modelMatrix, projectionMatrix, viewPort, &posx, &posy, &posz);
    canvas.boundingTrapezoid[3].tilex = canvas.tilex + getCanvasShiftX();
    canvas.boundingTrapezoid[3].tiley = canvas.tiley + getCanvasShiftY();
    canvas.boundingTrapezoid[3].x = posx;
    canvas.boundingTrapezoid[3].y = posy;
    canvas.boundingTrapezoid[3].zoom = canvas.zoom;
}

/*
 * Compute the side of point (x, y) in relation to line segment (x0, y0) - (x1, y1)
 * returns <0 when point is on right side of line segment
 * returns 0 when point is belongs to line
 * return >0 when point is on left side
 */
GLfloat computeSide(GLfloat x0, GLfloat y0, GLfloat x1, GLfloat y1, GLfloat x, GLfloat y) {
    GLfloat result = (y1 - y) * (x1 - x0) - (x1 - x) * (y1 - y0);
    //    fprintf(stderr, "    %f %f %f %f %f %f, result = %f\n", x0, y0, x1, y1, x, y, result);
    return result;
}

int tileEngine_isTileCoordinateInBoundingTrapezoid(int tilex, int tiley) {
    int i;

    for (i = 0; i < 4; i++) {
        if (computeSide(canvas.boundingTrapezoid[i].tilex + canvas.boundingTrapezoid[i].x / (GLfloat) TILE_SIZE, canvas.boundingTrapezoid[i].tiley
                + canvas.boundingTrapezoid[i].y / (GLfloat) TILE_SIZE, canvas.boundingTrapezoid[(i + 1) % 4].tilex + canvas.boundingTrapezoid[(i + 1)
                % 4].x / (GLfloat) TILE_SIZE, canvas.boundingTrapezoid[(i + 1) % 4].tiley + canvas.boundingTrapezoid[(i + 1) % 4].y
                / (GLfloat) TILE_SIZE, tilex, tiley) < 0) {
            //            fprintf(stderr, "    return FALSE\n");
            return FALSE;
        }
    }
    //    fprintf(stderr, "    return TRUE\n");
    return TRUE;
}

void tileEngine_switchTileProvider() {
}

void tileEngine_computeTilesVisibility() {
    int i, j;

    // tiles share coordinates so it is more efficient to precalculate visibility for each unique coordinate.
    for (i = 0; i < TILES_X + 1; i++) {
        for (j = 0; j < TILES_Y + 1; j++) {
            tileCoordinateVisibility[i][j] = tileEngine_isTileCoordinateInBoundingTrapezoid(canvas.tilex + i, canvas.tiley + j);
        }
    }
    for (i = 0; i < TILES_X; i++) {
        for (j = 0; j < TILES_Y; j++) {
            if (tileCoordinateVisibility[i][j] || tileCoordinateVisibility[i + 1][j] || tileCoordinateVisibility[i][j + 1]
                    || tileCoordinateVisibility[i + 1][j + 1]) {
                tiles[i][j][currentTilesIdx] -> visible = TRUE;
                // TODO parametrize how much textures can be created per frame
                if (tiles[i][j][currentTilesIdx] -> state == STATE_GL_TEXTURE_NOT_CREATED && texturesCreatedThisFrame
                        < MAX_TEXTURES_CREATED_PER_FRAME) {
                    tiles[i][j][currentTilesIdx]->texture = createTexture(tiles[i][j][currentTilesIdx] -> pixels4444, TILE_SIZE, TILE_SIZE, TRUE);
                    if (syncLoadedTilesAppearance == TRUE) {
                        tiles[i][j][currentTilesIdx] -> state = STATE_LOADED_SYNC_WAIT;
                    } else {
                        tiles[i][j][currentTilesIdx] -> state = STATE_LOADED;
                        //                        tiles[i][j][currentTilesIdx] ->stateChangeTime = 0;
                    }
                    texturesCreatedThisFrame++;
                }
            } else {
                tiles[i][j][currentTilesIdx] -> visible = FALSE;
                if (tiles[i][j][currentTilesIdx] -> state == STATE_LOADED || tiles[i][j][currentTilesIdx] -> state == STATE_LOADED_SYNC_WAIT) {
                    glDeleteTextures(1, &tiles[i][j][currentTilesIdx] -> texture);
                    tiles[i][j][currentTilesIdx] -> texture = 0;
                    tiles[i][j][currentTilesIdx] -> state = STATE_GL_TEXTURE_NOT_CREATED;
                }
            }
        }
    }
}
