/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Author: Damian Waradzyn
 */
#include <sys/stat.h>
#include <unistd.h>
#include <libgen.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/resource.h>

#define ALLOC_DEBUG 0

typedef enum {
    STATE_EMPTY = 0,
    STATE_LOADING = 1,
    STATE_ERROR = 2,
    STATE_LOADED = 3,
    STATE_SCALED_LOADING = 4,
    STATE_GL_TEXTURE_NOT_CREATED = 5,
    STATE_LOADED_SYNC_WAIT = 6
} TileState;

typedef struct {
    int zoom;
    int tilex, tiley;
    GLuint texture;
    GLuint oldTexture[4];
    GLfloat oldTextureCoords[4][8];
    GLfloat oldTextureSize;
    TileState state;
    long stateChangeTime;
    int transitionTime;
    GLubyte visible, deleted;

    // TODO
    GLushort * pixels4444; // pixels used as texture, to be freed when texture is deleted
    GLushort * oldPixels4444[4];
    TileProvider* provider;
    char *errorMessage;
} t_tile;

t_tile* tiles[TILES_X][TILES_Y][2];

typedef struct {
    int frame;
    SDL_Surface * surface;
    GLushort * oldPixels4444[4];
} TDeallocationData;

GQueue* deallocationQueue = NULL;

char* getTileFileName(TileProvider *provider, t_tile *tile);
char* getTileUrl(TileProvider *provider, t_tile *tile);
int downloadUrl(char *url, char *filename, t_tile *tile);
GLushort * convert8888to4444(SDL_Surface* surfaceRGBA);

gpointer dequeue(GQueue* queue);
void deleteElems(GQueue * queue, gpointer data);

void processDeferredDeallocation() {
    TDeallocationData * data = g_queue_peek_head(deallocationQueue);
    if (data == NULL || data -> frame + 20 > frame) {
        return;
    }
    g_queue_pop_head(deallocationQueue);
    int i;
    for (i = 0; i < 4; i++) {
        if (data -> oldPixels4444[i] != NULL) {
            if (ALLOC_DEBUG) {
                fprintf(stderr, "deferredDeallocate: free data = %p, oldPixels4444[%d] %p\n", data, i, data -> oldPixels4444[i]);
            }
            free(data -> oldPixels4444[i]);
            data -> oldPixels4444[i] = NULL;
        }
    }
    if (data -> surface != NULL) {
        if (ALLOC_DEBUG) {
            fprintf(stderr, "deferredDeallocate: free data = %p, surface %p\n", data, data -> surface);
        }
        SDL_FreeSurface(data -> surface);
        data -> surface = NULL;
    }
    free(data);
}

void deallocateTile(t_tile *tile) {
    if (tile == NULL) {
        return;
    }
    if (tile -> state == STATE_LOADING || tile -> state == STATE_SCALED_LOADING) {
        tile -> deleted = TRUE;
        return;
    }
    if (ALLOC_DEBUG) {

        fprintf(stderr, "deallocateTile(%p)\n", tile);
    }
    deleteElems(downloadQueue, (gpointer) tile);

    if (tile->texture != 0) {
        glDeleteTextures(1, &tile->texture);
        tile->texture = 0;
    }
    if (ALLOC_DEBUG) {

        fprintf(stderr, "deallocateTile, free pixels4444 %p\n", tile -> pixels4444);
    }

    // FIXME dirty hack, this should be an pointer with reference count
    int i, j, count = 0;
    for (i = 0; i < TILES_X; i++) {
        for (j = 0; j < TILES_Y; j++) {
            if (tiles[i][j][currentTilesIdx] != NULL && tiles[i][j][currentTilesIdx] != tile) {
                if (tiles[i][j][currentTilesIdx] -> pixels4444 == tile -> pixels4444) {
                    count++;
                }
            }
        }
    }

    if (count == 0) {
        free(tile -> pixels4444);
    }
    tile -> pixels4444 = NULL;

    count = 0;
    if (tile -> oldPixels4444[0] != NULL) {
        // FIXME another one
        for (i = 0; i < TILES_X; i++) {
            for (j = 0; j < TILES_Y; j++) {
                if (tiles[i][j][currentTilesIdx] != NULL && tiles[i][j][currentTilesIdx] != tile) {
                    if (tiles[i][j][currentTilesIdx] -> oldPixels4444[0] == tile -> oldPixels4444[0]) {
                        count++;
                    }
                }
            }
        }
    }

    if (count == 0) {
        for (i = 0; i < 4; i++) {
            if (tile -> oldPixels4444[i] != NULL) {
                if (ALLOC_DEBUG) {
                    fprintf(stderr, "deallocateTile, free oldpixels444[%d] %p\n", i, tile -> oldPixels4444[i]);
                }
                free(tile -> oldPixels4444[i]);
                tile -> oldPixels4444[i] = NULL;
            }
        }
    }

    //    if (tile -> errorMessage) {
    //        fprintf(stderr, "deallocateTile, free errorMessage %p\n", tile -> errorMessage);
    //        free(tile -> errorMessage);
    //        tile -> errorMessage = NULL;
    //    }
    deleteElems(downloadQueue, (gpointer) tile);
    if (ALLOC_DEBUG) {
        fprintf(stderr, "deallocateTile, free tile %p\n", tile);
    }
    free(tile);
}

// todo: move to file.c
void ensureDirExists(char *filename) {
    char *cpy = strdup(filename);
    g_mkdir_with_parents(dirname(cpy), 0755);
    free(cpy);
}

SDL_Surface* loadTextureFromFile(char *filename, t_tile* tile);

int loadTileTexture(char* filename, t_tile* tile) {
    SDL_Surface* texture = loadTextureFromFile(filename, tile);

    if (texture) {
        TDeallocationData * data;
        data = calloc(1, sizeof(TDeallocationData));
        data -> frame = frame;
        data -> surface = texture;
        tile -> pixels4444 = convert8888to4444(texture);
        g_queue_push_tail(deallocationQueue, data);

        //        fprintf(stderr, "loadTextureFromFile, tile= %p allocated pixels4444 %p\n", tile, tile -> pixels4444);
        tile -> state = STATE_GL_TEXTURE_NOT_CREATED;
        return TRUE;
    }
    return FALSE;
}

int shouldAbortLoading(t_tile * tile) {
    if (tile -> deleted) {
        tile -> state = STATE_ERROR;
        deallocateTile(tile);
        return TRUE;
    }

    if ((tile -> tilex < canvas.tilex) || (tile -> tiley < canvas.tiley) || (tile -> tilex >= (canvas.tilex + TILES_X)) || (tile -> tiley
            >= (canvas.tiley + TILES_Y)) || (tile -> zoom != canvas.zoom) || (tile -> provider -> name != canvas.provider -> name)) {
        //        fprintf(stderr, "tile load abort, tilex = %d, tiley = %d, canvasx = %d, canvasy = %d, tile.zoom = %d, canvas.zoom = %d\n",
        //                tile -> tilex, tile -> tiley, canvas.tilex, canvas.tiley, tile -> zoom, canvas.zoom);
        deallocateTile(tile);
        return TRUE;
    }
    return FALSE;
}

int debugPreventLoadingTiles = 0;

int downloadThread(void* queue) {
    t_tile * tile = NULL;

    setpriority(PRIO_PROCESS, 0, 19);

    while (!quit) {
        tile = (t_tile*) dequeue(queue);
        while (tile == NULL) {
            //            fprintf(stderr, "queue empty, waiting, thread id = %d\n", SDL_ThreadID());
            SDL_Delay(0);
            if (quit) {
                return 0;
            }
            tile = (t_tile*) dequeue(queue);
        }

        int mapSize = 1 << tile -> zoom;

        if (tile -> tilex < 0 || tile -> tiley < 0 || tile -> tilex >= mapSize || tile -> tiley >= mapSize || debugPreventLoadingTiles) {
            //            fprintf(stderr, "tile outside map\n");
            tile -> state = STATE_EMPTY;
            continue;
        }

        if (shouldAbortLoading(tile)) {
            continue;
        }

        if (tile -> state != STATE_SCALED_LOADING) {
            tile -> state = STATE_LOADING;
        }

        //        fprintf(stderr, "load tile start, thread id = %d\n", SDL_ThreadID());
        char *filename;
        filename = getTileFileName(canvas.provider, tile);
        if (!loadTileTexture(filename, tile)) {

            if (shouldAbortLoading(tile)) {
                free(filename);
                continue;
            }

            //            fprintf(stderr, "load tile from local file failed, trying to download tile\n");
            ensureDirExists(filename);
            if (downloadUrl(getTileUrl(canvas.provider, tile), filename, tile)) {

                if (shouldAbortLoading(tile)) {
                    free(filename);
                    continue;
                }

                if (loadTileTexture(filename, tile)) {
                    if (tile ->transitionTime == 0) {
                        tile ->transitionTime = 2000;
                    }
                    //                    fprintf(stderr, "download successfull\n");
                } else {
                    tile -> state = STATE_ERROR;
                    //                    fprintf(stderr, "unable to load downloaded tile\n");
                }
            } else {
                tile -> state = STATE_ERROR;
                //                fprintf(stderr, "download error\n");
            }
        } else {
            if (tile ->transitionTime == 0) {
                tile -> transitionTime = 200;
            }
        }
        free(filename);
    }
    //    fprintf(stderr, "finished thread, id = %d\n", SDL_ThreadID());
    return 0;
}
