/*!
** @file	camera.cpp
**
** @brief	Class that manages the camera on the N900
**
** Handles display of video preview as well as taking snapshots with
** the camera shutter button
*/

/*---------------------------------------------------------------------------
** Includes
*/
#include <stdlib.h>
#include <string.h>
#include <QtGui>

#include <QDBusConnection>
#include <QDBusMessage>
#include <QDBusArgument>
#include <QDBusMetaType>
#include <QDBusInterface>

#include <gst/gst.h>
#include <gst/interfaces/xoverlay.h>
#include <gst/interfaces/photography.h>

#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/videodev2.h>

#include "camera.h"

/*---------------------------------------------------------------------------
** Defines and Macros
*/
// Delcare the types used in our DBus property slot
Q_DECLARE_METATYPE(Property)
Q_DECLARE_METATYPE(QList<Property>)

#define DBUS_SHUTTER_RELEASE_BUTTON		"/org/freedesktop/Hal/devices/platform_cam_launch"
#define DBUS_FOCUS_BUTTON				"/org/freedesktop/Hal/devices/platform_cam_focus"
#define DBUS_LENS_COVER					"/org/freedesktop/Hal/devices/platform_cam_shutter"

/*---------------------------------------------------------------------------
** Typedefs
*/

/*---------------------------------------------------------------------------
** Local function prototypes
*/

/*---------------------------------------------------------------------------
** Data
*/
#define CAPTURE_WIDTH		(550/2)
#define CAPTURE_HEIGHT		(360/2)

//---------------------------------------------------------------------------

#if 1
#define VIDEO_SRC 		"v4l2camsrc"
#define VIDEO_SINK 		"xvimagesink"
#else
#define VIDEO_SRC 		"videotestsrc"
#define VIDEO_SINK 		"ximagesink"
#endif

//--------------------------------------------------------------------------
/*!
** @brief   	Constructor
*/
Camera::Camera(QWidget* parent) : QWidget(parent)
{
	m_id = 0;
	m_started = false;
	m_video_sink = 0;
	m_pipeline = NULL;

	setAttribute(Qt::WA_PaintOnScreen, true);

	m_focus_mode = CAMERA_FOCUS_MODE_AUTO;
	m_flash_mode = CAMERA_FLASH_MODE_AUTO;
	
	// Register the types used in the DBus message that we're
	// interested in
	qDBusRegisterMetaType< Property >();
	qDBusRegisterMetaType< QList<Property> >();

	// Connect the shutter notifications to our slot
 	QDBusConnection::systemBus().connect(
					QString(),
					DBUS_SHUTTER_RELEASE_BUTTON,
	 				"org.freedesktop.Hal.Device",
                    "PropertyModified",
                    this,
                    SLOT(shutter_property_modified(int, QList<Property>)));
 	QDBusConnection::systemBus().connect(
					QString(),
					DBUS_FOCUS_BUTTON,
	 				"org.freedesktop.Hal.Device",
                    "PropertyModified",
                    this,
                    SLOT(focus_property_modified(int, QList<Property>)));
	
    m_photo = QImage(64, 64, QImage::Format_Invalid);
	m_current_filename = "/home/user/camerabin.jpg";
}

//--------------------------------------------------------------------------
/*!
** @brief   	Destructor
*/
Camera::~Camera()
{
    destroy_pipeline();
}

//--------------------------------------------------------------------------
/*!
** @brief   	Set the X Window ID for rendering video overlay
**
** @param [In] id		X Window ID
**
*/
void Camera::set_x_window_id(WId id)
{
	m_id = id;
}


//--------------------------------------------------------------------------
/*!
** @brief		Start the video preview and prepare for taking
**				a snapshot
**
** @param[In]	image_directory		Folder to store photos in
**
** @param[In]	base_name			Base name for images
**
*/
void Camera::start(QString image_directory, QString base_name) {
	initialise_pipeline();

	m_image_directory = image_directory;
	m_base_filename = base_name;
	
	g_object_set(G_OBJECT(m_pipeline), "mode", 0, (void*)NULL);

	gst_element_set_state(m_pipeline, GST_STATE_PLAYING);

	// Set the default modes
	set_flash_mode(m_flash_mode);
	set_focus_mode(m_focus_mode);
}

//--------------------------------------------------------------------------
/*!
** @brief		Stop the video preview
*/
void Camera::stop()
{
	gst_element_set_state(m_pipeline, GST_STATE_NULL);
}

//--------------------------------------------------------------------------
/*!
** @brief   	Capture a still image from the camera and emit
**				snapshot signal
*/
void Camera::capture_image()
{
	// Calculate the filename for the image
	m_current_filename = m_image_directory;
	if (m_current_filename.size() == 0) {
		return;
	}
	// Make sure it has a slash on the end
	if (m_current_filename[m_current_filename.size()-1] != '/') {
		m_current_filename += "/";
	}
	m_current_filename += m_base_filename;
	// Now append the current date and time
	QDateTime	datetime = QDateTime::currentDateTime();
	m_current_filename += datetime.toString("yyyyMMddhhmmss");
	m_current_filename += ".jpg";

	printf("filename: %s\n", m_current_filename.toAscii().constData());
	g_object_set(G_OBJECT(m_pipeline), "filename", m_current_filename.toAscii().constData(), (void*)NULL);

	g_signal_emit_by_name(G_OBJECT(m_pipeline), "user-start", this, NULL);
}

//--------------------------------------------------------------------------
/*!
** @brief		Initialise the gstreamer pipeline
**
** @return		true/false
**
*/
bool Camera::initialise_pipeline()
{
	GstBus *bus;

	// Initialise the gstreamer library
	gst_init( NULL, NULL);

	//m_pipeline = gst_pipeline_new("test-camera");
	m_pipeline = gst_element_factory_make("camerabin", NULL);

	bus = gst_pipeline_get_bus(GST_PIPELINE(m_pipeline));
	gst_bus_set_sync_handler(bus, gst_bus_sync_signal_handler, this);
	g_signal_connect(bus, "sync-message::element", G_CALLBACK(element_msg_sync), this);
	gst_object_unref(GST_OBJECT(bus));

	// Create our own video sink so we can put it in a window
	m_video_sink = gst_element_factory_make(VIDEO_SINK, "xvimagesink");
	g_object_set(G_OBJECT(m_pipeline), "vfsink", m_video_sink, (void*)NULL);

	GstCaps*	caps = gst_caps_new_simple("video/x-raw-rgb",
											"width", G_TYPE_INT, CAPTURE_WIDTH,
											"height", G_TYPE_INT, CAPTURE_HEIGHT,
											NULL);
	g_object_set(G_OBJECT(m_pipeline), "preview-caps", caps, (void*)NULL);

	// Attach to the camerabin signal that tells us when the photo is complete
	g_signal_connect(m_pipeline, "img-done", G_CALLBACK(image_done_callback), this);

	g_signal_emit_by_name(G_OBJECT(m_pipeline), "user-image-res", 2584, 1938, this, NULL);

	//g_signal_emit_by_name(G_OBJECT(m_pipeline), "user-res-fps", 400, 240, 0, 0, this);

	//g_object_get(GST_OBJECT(m_pipeline), "vfsink", &m_video_sink, (void*)NULL);

	return true;
}

//--------------------------------------------------------------------------
/*!
** @brief		Initialise the gstreamer pipeline
**
** @return		true/false
**
*/
void Camera::destroy_pipeline()
{
	gst_element_set_state(m_pipeline, GST_STATE_NULL);
    gst_x_overlay_set_xwindow_id(GST_X_OVERLAY(m_video_sink), 0);
	gst_object_unref(GST_OBJECT(m_pipeline));
}

//--------------------------------------------------------------------------
/*!
** @brief		Slot called by DBus when the shutter button properties change
**
** This class uses the slot notification to determine when the
** shutter button has been pressed
**
*/
void Camera::shutter_property_modified(int num_updates, QList<Property> updates) {
	Q_UNUSED(num_updates);
	Q_UNUSED(updates);

    QDBusInterface propertyInterface("org.freedesktop.Hal",
									 DBUS_SHUTTER_RELEASE_BUTTON,
		  							 "org.freedesktop.Hal.Device",
									  QDBusConnection::systemBus());
    bool pressed = propertyInterface.call("GetProperty", "button.state.value").arguments().at(0).toBool();
    if (pressed) {
		capture_image();
    }
}

//--------------------------------------------------------------------------
/*!
** @brief		Slot called by DBus when the focus properties change
**
** This class uses the slot notification to determine when the
** shutter button is half pressed
**
*/
void Camera::focus_property_modified(int num_updates, QList<Property> updates) {
	Q_UNUSED(num_updates);
	Q_UNUSED(updates);
    
	QDBusInterface propertyInterface("org.freedesktop.Hal",
									 DBUS_FOCUS_BUTTON,
		  							 "org.freedesktop.Hal.Device",
									  QDBusConnection::systemBus());
    bool pressed = propertyInterface.call("GetProperty", "button.state.value").arguments().at(0).toBool();
    if (pressed) {
		focus();
    }
}

//--------------------------------------------------------------------------
/*!
** @brief		Perform autofocus operation
**
*/
void Camera::focus() {
	GstElement*		camera_src;
	g_object_get(GST_OBJECT(m_pipeline), "videosrc", &camera_src, (void*)NULL);
	if (camera_src) {
		if (m_focus_mode == CAMERA_FOCUS_MODE_AUTO) {
			gst_photography_set_autofocus(GST_PHOTOGRAPHY(camera_src), TRUE);
		}
		else {
			// Fixed focus. Set to infinity
			gst_photography_set_autofocus(GST_PHOTOGRAPHY(camera_src), FALSE);
			set_fixed_focus();
		}
	}
}

//--------------------------------------------------------------------------
/*!
** @brief		Set fixed focus range to infinity
**
*/
void Camera::set_fixed_focus() {
	int						fd;
	struct v4l2_queryctrl 	qctrl;
	struct v4l2_control		ctrl;

	fd = open("/dev/video0", O_RDWR | O_NONBLOCK, 0);
	if (fd == -1) {
		return;
	}
	// Get focus range
	qctrl.id = V4L2_CID_FOCUS_ABSOLUTE;
	if (ioctl(fd, VIDIOC_QUERYCTRL, &qctrl) == -1) {
		goto error;
	}
	printf("Focus min: %d, max: %d\n", qctrl.minimum, qctrl.maximum);

	// Set the range to infinity
	ctrl.id = V4L2_CID_FOCUS_ABSOLUTE;
	ctrl.value = qctrl.minimum;
	if (ioctl(fd, VIDIOC_S_CTRL, &ctrl) == -1) {
		goto error;
	}

error:
	::close(fd);
}

//--------------------------------------------------------------------------
/*!
** @brief		Receive element signals from pipeline bus
**
*/
void Camera::element_msg_sync(GstBus* bus, GstMessage* msg, gpointer data)
{
	Q_UNUSED(bus);
	GstElement *video_sink = NULL;
	Camera* camera = (Camera*)data;

	g_object_get(GST_OBJECT(camera->m_pipeline), "vfsink", &video_sink, (void*)NULL);

	g_assert(msg->type == GST_MESSAGE_ELEMENT);

	if (msg->structure == NULL)
		return;

	/* This only gets sent if we haven't set an ID yet. This is our last
	 * chance to set it before the video sink will create its own window */
	printf("element_msg_sync: %s\n", gst_structure_get_name(msg->structure));
	if ((gst_structure_has_name(msg->structure, "prepare-xwindow-id")) ||
		(gst_structure_has_name(msg->structure, "have-xwindow-id"))) {
		Camera* camera = (Camera*)data;
		g_object_set(G_OBJECT(camera->m_video_sink), "force-aspect-ratio", 1, (void*)NULL);
		gst_x_overlay_set_xwindow_id(GST_X_OVERLAY(camera->m_video_sink), camera->winId());
	}
	else if (gst_structure_has_name(msg->structure, "photo-capture-start")) {
		Camera* camera = (Camera*)data;
		camera->emit photo_start();
	}
	else if (gst_structure_has_name(msg->structure, "photo-capture-end")) {
		Camera* camera = (Camera*)data;
		camera->emit photo_end();
	}
	else if (gst_structure_has_name(msg->structure, "preview-image")) {
		const GstStructure* st = gst_message_get_structure(msg);
		if (st) {
			gpointer p;
			if (gst_structure_get((GstStructure*)st, "buffer", g_type_from_name("GstBuffer"), &p, (void*)NULL)) {
				GstBuffer* buf = (GstBuffer*)p;
				Camera* camera = (Camera*)data;
    			camera->m_photo = QImage(GST_BUFFER_DATA(buf), CAPTURE_WIDTH, CAPTURE_HEIGHT, QImage::Format_RGB888).copy();
				// Send the signal to say we've got it
				camera->emit preview(&camera->m_photo);
			}
		}
	}
	else if (gst_structure_has_name(msg->structure, "autofocus-done")) {
		const GstStructure* st = gst_message_get_structure(msg);
		if (st) {
			gint status;
			if (gst_structure_get_int(st, "status", &status)) {
				switch (status) {
					case GST_PHOTOGRAPHY_FOCUS_STATUS_FAIL:
						printf("Autofocus fail\n");
						break;
					case GST_PHOTOGRAPHY_FOCUS_STATUS_SUCCESS:
						printf("Autofocus success\n");
						break;
					case GST_PHOTOGRAPHY_FOCUS_STATUS_NONE:
						printf("Autofocus none\n");
						break;
					case GST_PHOTOGRAPHY_FOCUS_STATUS_RUNNING:
						printf("Autofocus running\n");
						break;
					default:
						break;
				}
			}
		}
	}
	else if (gst_structure_has_name(msg->structure, "image-captured")) {
		// Now handled by gstreamer signal
	}
}

//--------------------------------------------------------------------------
/*!
** @brief   	Get the current flash mode for the camera
**
** @return 		Flash mode enumeration
**
*/
CAMERA_FLASH_MODE Camera::get_flash_mode() {
	return m_flash_mode;
}

//--------------------------------------------------------------------------
/*!
** @brief   	Set the current flash mode for the camera
**
** @param[in]	mode	New flash mode
**
*/
void Camera::set_flash_mode(CAMERA_FLASH_MODE mode) {
	m_flash_mode = mode;
	if (m_pipeline) {
		// The pipeline has been initialised. Set the mode in the pipeline
		GstElement*		camera_src;
		g_object_get(GST_OBJECT(m_pipeline), "videosrc", &camera_src, (void*)NULL);
		if (camera_src) {
			// There is a 1-1 corresponence between our mode and GST modes
			gst_photography_set_flash_mode(GST_PHOTOGRAPHY(camera_src), (GstFlashMode)mode);
		}
	}
}

//--------------------------------------------------------------------------
/*!
** @brief   	Get the current focus mode for the camera
**
** @return 		Focus mode enumeration
**
*/
CAMERA_FOCUS_MODE Camera::get_focus_mode() {
	return m_focus_mode;
}

//--------------------------------------------------------------------------
/*!
** @brief   	Set the current focus mode for the camera
**
** @param[in]	mode	New focus mode
**
*/
void Camera::set_focus_mode(CAMERA_FOCUS_MODE mode) {
	m_focus_mode = mode;
}

//--------------------------------------------------------------------------
/*!
** @brief   	Signal from camerabin when photo has been written to file
**
*/
gboolean Camera::image_done_callback(void* bin, gchar* filename, gpointer user_data) {
	Q_UNUSED(bin)
	// The file write should be complete by now. Signal that the photo is ready
	Camera* camera = (Camera*)user_data;
	camera->emit photo_taken(filename);
	return TRUE;
}

//--------------------------------------------------------------------------
/*!
** @brief		Marshalling code for Property type
*/
const QDBusArgument & operator<<(QDBusArgument &arg, const Property &change) {
	arg.beginStructure();
	arg << change.name << change.added << change.removed;
	arg.endStructure();
	return arg;
}
const QDBusArgument & operator>>(const QDBusArgument &arg, Property &change) {
	arg.beginStructure();
	arg >> change.name >> change.added >> change.removed;
	arg.endStructure();
	return arg;
}
