/* This file is part of Cinaest.
 *
 * Copyright (C) 2009 Philipp Zabel
 *
 * Cinaest is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Cinaest is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cinaest. If not, see <http://www.gnu.org/licenses/>.
 */

using Gtk;
using Hildon;

public class MovieListView : PannableArea {
	public MovieListStore store;
	TreeView tree;
	public TreeSortable sorted_store;

	public signal void movie_activated (Movie movie);

	construct {
		store = new MovieListStore ();

		// Add filter wrapper
		var filtered_store = new TreeModelFilter (store, null);

		// Add sort wrapper
		sorted_store = new TreeModelSort.with_model (filtered_store);

		// Tree View
		tree = (TreeView) Hildon.gtk_tree_view_new_with_model (UIMode.NORMAL, sorted_store);
		tree.set_headers_visible (false);

		add (tree);

		tree.set_rules_hint (true);

		// Tree selection object
		var selection = tree.get_selection ();
		selection.set_mode (SelectionMode.SINGLE);

		// Title column with poster
		var title_column = new TreeViewColumn ();
		title_column.set_title (_("Movie"));
		title_column.set_sort_column_id (MovieListStore.Columns.TITLE);
		title_column.set_reorderable (false);
		title_column.set_sizing (TreeViewColumnSizing.AUTOSIZE);
		title_column.set_expand (true);

		// Add poster icon to column
		var pixbuf_renderer = new CellRendererPixbuf ();
		pixbuf_renderer.width = 64;
		title_column.pack_start (pixbuf_renderer, false);
		title_column.add_attribute (pixbuf_renderer, "pixbuf", MovieListStore.Columns.POSTER);

		// Add text to column
		var vbox_renderer = new CellRendererVBox ();

		var renderer = new CellRendererText ();
		renderer.set ("ellipsize", Pango.EllipsizeMode.END);
		title_column.add_attribute (renderer, "text", MovieListStore.Columns.TITLE);

		vbox_renderer.append (renderer, true);
		vbox_renderer.set_data ("title", renderer);

		// Add secondary text to column (Genres, Director, etc.)
		renderer = new CellRendererText ();
		renderer.set ("ellipsize", Pango.EllipsizeMode.END);

		Pango.AttrList attr_list = new Pango.AttrList ();
		var style = Gtk.rc_get_style_by_paths (this.get_settings (), "SmallSystemFont", null, typeof (void));
		if (style != null) {
			Pango.Attribute attr_font_desc = Pango.attr_font_desc_new (style.font_desc.copy ());
			attr_list.insert ((owned) attr_font_desc);
		} else {
			Pango.Attribute attr_font_scale = Pango.attr_scale_new (Pango.Scale.SMALL);
			attr_list.insert ((owned) attr_font_scale);
		}
		Gdk.Color color;
		if (!style.lookup_color ("SecondaryTextColor", out color)) {
			Gdk.Color.parse ("grey", out color);
		}
		Pango.Attribute attr_color = Pango.attr_foreground_new (color.red, color.green, color.blue);
		attr_list.insert ((owned) attr_color);
		renderer.attributes = attr_list;

		vbox_renderer.append (renderer, true);
		vbox_renderer.set_data ("secondary", renderer);

		title_column.pack_start (vbox_renderer, true);
		title_column.set_cell_data_func (vbox_renderer, vbox_data_func);

		tree.append_column (title_column);

		// Sort by title
		sorted_store.set_sort_column_id (MovieListStore.Columns.TITLE, SortType.ASCENDING);

		// Year column
		renderer = new CellRendererText ();
		var year_column = new TreeViewColumn ();
		year_column.set_title (_("Rating"));
		year_column.set_sort_column_id (MovieListStore.Columns.YEAR);
		year_column.set_reorderable (false);
		year_column.set_sort_order (SortType.DESCENDING);
		year_column.pack_start (renderer, true);
		year_column.set_cell_data_func (renderer, year_data_func);
		tree.append_column (year_column);

		// Rating column
		renderer = new CellRendererText ();
		var rating_column = new TreeViewColumn ();
		rating_column.set_title (_("Rating"));
		rating_column.set_sort_column_id (MovieListStore.Columns.RATING);
		rating_column.set_reorderable (false);
		rating_column.set_sort_order (SortType.DESCENDING);
		rating_column.pack_start (renderer, true);
		rating_column.set_cell_data_func (renderer, rating_data_func);
		rating_column.xalign = (float) 1.0;
		tree.append_column (rating_column);

		// Connect signals
		tree.row_activated.connect (on_row_activated);
	}

	public void set_hildon_ui_mode (UIMode mode) {
		var selection = tree.get_selection ();

		if (mode == UIMode.NORMAL) {
			selection.set_mode (SelectionMode.NONE);
		}
		Hildon.gtk_tree_view_set_ui_mode (tree, mode);
		if (mode == UIMode.EDIT) {
			selection.set_mode (SelectionMode.MULTIPLE);
		}
	}

	public unowned TreeSelection get_selection () {
		return tree.get_selection ();
	}

	private void on_row_activated (TreeView tree, TreePath path_, TreeViewColumn column) {
		TreePath path = path_.copy (); // FIXME - calling model.get_iter with path_ directly causes a C qualifier warning
		TreeModel model = tree.model;
		TreeIter iter;

		if (model.get_iter (out iter, path)) {
			Movie movie;
			model.get (iter, MovieListStore.Columns.MOVIE, out movie);
			movie_activated (movie);
		}
	}

	private void vbox_data_func (CellLayout cell_layout, CellRenderer cell, TreeModel model, TreeIter iter) {
		Movie movie;
		CellRendererText renderer;

		model.get (iter, MovieListStore.Columns.MOVIE, out movie);

		renderer = (CellRendererText) cell.get_data ("title");
		renderer.text = movie.title;

		renderer = (CellRendererText) cell.get_data ("secondary");
		renderer.text = movie.secondary;
	}

	private void year_data_func (CellLayout cell_layout, CellRenderer cell, TreeModel model, TreeIter iter) {
		int year;

		model.get (iter, MovieListStore.Columns.YEAR, out year);
		((CellRendererText) cell).text = (year > 0) ? year.to_string () : "";
	}

	private void rating_data_func (CellLayout cell_layout, CellRenderer cell, TreeModel model, TreeIter iter) {
		int rating;

		model.get (iter, MovieListStore.Columns.RATING, out rating);
		((CellRendererText) cell).text = (rating > 0) ? "%d.%d".printf (rating / 10, rating % 10) : "";
	}
}
