/*
 * Don Ceferino Hazaa - video game similary to Super Pang!
 * Copyright (c) 2004, 2005 Hugo Ruscitti
 * web site: http://www.loosersjuegos.com.ar
 * 
 * This file is part of Don Ceferino Hazaa (ceferino).
 * Written by Hugo Ruscitti <hugoruscitti@yahoo.com.ar>
 *
 * Don Ceferino Hazaa is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Don Ceferino Hazaa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */


#include <SDL/SDL.h>
#include <SDL/SDL_image.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include "mundo.h"
#include "utils.h"
#include "opciones.h"
#include "int.h"

#ifdef HAVE_CONFIG_H
	#include "../config.h"
#endif

#include "common.h"

mundo :: mundo()
{
	actual=NULL;
	salir=0;
	leer_teclas=1;
	key_mouse_pressed = UNDEFINED;
	savegame_present = 0;
}


/*!
 * \brief inicializa la biblioteca y los recursos bsicos
 *
 * \return 1 en caso de error
 */
int mundo :: iniciar (void)
{
	int flags=0;
	SDL_Surface *ico;

	printf(_("+ Starting the SDL library:"));

	if (SDL_Init(SDL_INIT_AUDIO | SDL_INIT_VIDEO | SDL_INIT_TIMER))
	{
		imprimir_error();
		printf("error: %s\n", SDL_GetError());
		return 1;
	}
	
	SDL_ShowCursor(SDL_DISABLE);
	SDL_WarpMouse(0,0);

	imprimir_ok();

	flags |= SDL_FULLSCREEN | SDL_DOUBLEBUF | SDL_HWSURFACE;

	screen = SDL_SetVideoMode(SCREEN_RES_X, SCREEN_RES_Y, 16,  flags);

	SDL_EnableUNICODE(1);

	fuente = new fuente2 (opciones.modo_video);

	
	if (screen == NULL)
	{
		printf("error: %s\n", SDL_GetError());
		return 1;
	}


	SDL_WM_SetCaption("Don Ceferino - ver " VERSION, NULL);

	ico = IMG_Load (DATADIR "/ima/icono.png");

	if (ico)
	{
		SDL_WM_SetIcon(ico, NULL);
		SDL_FreeSurface(ico);
	}

	imprimir_cargando();

	libgrafico.iniciar(opciones.modo_video);
	
	audio.iniciar(opciones.sonido, opciones.musica, opciones.enable_sound, opciones.enable_music);

	escena_anterior=-1;
	escena=0;

	cambiar_escena(QUESTION);
	return 0;
}


/*!
 * \brief altera la escena actual
 */
void mundo :: cambiar_escena(int nueva_escena)
{
	escena = nueva_escena;
}

/*!
 * \brief mantiene la velocidad constante del juego
 */
void mundo :: correr (void)
{
	SDL_Event evento;
	int fps=0;
	int log=0;
	int tframe = -100; // tick del ltimo fps actualizado
	int rep, i;
	
	#define TICK_POR_LOGICO (1000/100)
	#define TICK_POR_FRAME (1000/100)

	reiniciar_reloj();

	while (!salir)
	{
		if (escena != escena_anterior)
		{	
			if ((((escena == MENU) && (escena_anterior == INTRO)) ||
			     ((escena == MENU) && (escena_anterior == FINAL)) ||
			     ((escena == MENU) && (escena_anterior == JUEGO)))
				&& opciones.enable_music)
			{
				this->audio.play_musica(1);
			}
			else if ((escena == SAVE_GAME) && (escena_anterior == JUEGO))
			{
				this->audio.pause_musica(0);
			}
			else if ((escena == INTRO) && (escena_anterior == QUESTION) && opciones.enable_music)
			{
				this->audio.play_musica(0);
				this->audio.play_dummy(500);
			}

			intercambiar_escenas();
		}
		
		salir = procesar_eventos(&evento);

		t = SDL_GetTicks();
		
		if (t - tl > TICK_POR_LOGICO)	// lgica
		{
			rep = (t - tl) / TICK_POR_LOGICO;

			for (i=0; i <rep; i++)
			{
				actual->actualizar(key_mouse_pressed);
				key_mouse_pressed = UNDEFINED;
			}
	
			tl += rep * TICK_POR_LOGICO;
			log += rep;

			if (t - tg > TICK_POR_FRAME)	// grafica
			{
				actual->imprimir();
				fps++;
				tg += TICK_POR_FRAME;
			}
		}
		else
		{
			SDL_Delay(TICK_POR_LOGICO - (t-tl));
		}
	}
}

/*!
 * \brief atiende los eventos de la ventana
 * 
 * \return 1 si la aplicacion debe terminar
 */
int mundo :: procesar_eventos(SDL_Event *evento)
{
	while (SDL_PollEvent(evento))
	{
		switch (evento->type)
		{
			case SDL_QUIT:
				return 1;
				break;
				
			case SDL_KEYDOWN:
				
				if (leer_teclas)
				{
					switch (evento->key.keysym.sym)
					{
/*
						case SDLK_f:
							SDL_WM_ToggleFullScreen(screen);
							reiniciar_reloj();
						break;
*/
						case SDLK_s:
							key_mouse_pressed = 's';
							break;

						case SDLK_m:
							key_mouse_pressed = 'm';
							break;

						case SDLK_y:
							key_mouse_pressed = 'y';
							break;

						case SDLK_n:
							key_mouse_pressed = 'n';
							break;

						case SDLK_c:
							key_mouse_pressed = 'c';
							break;
					}
				}
				break;

			case SDL_ACTIVEEVENT:
				
				if (evento->active.state == SDL_APPINPUTFOCUS)
				{
					if (! evento->active.gain)
						actual->pausar();
				}
				break;
	
			case SDL_MOUSEBUTTONDOWN:
			{
				key_mouse_pressed = TOUCH_SCREEN_PRESSED;
				break;
			}
			
			case SDL_MOUSEBUTTONUP:
			{
				key_mouse_pressed = UNDEFINED;
				break;
			}
		}
	}
	
	return 0;
}


/*!
 * \brief libera recursos y termina la ejecucion del programa
 */
void mundo :: terminar(void)
{
	salir=1;
}

/*!
 * \brief termina el programa
 */
void mundo :: eliminar(void)
{
	if (actual)
		delete actual;
	
	SDL_FreeSurface(screen);
	SDL_Quit();
}

/*!
 * \brief Set default options
 */
int mundo :: set_default_options(void)
{
	opciones.saved_lives = INITIAL_LIVES;
	opciones.saved_score = INITIAL_SCORE;
	opciones.saved_level = INITIAL_LEVEL;
	opciones.saved_time  = INITIAL_TIME;
	opciones.saved_music = 2;
	opciones.saved_wallpaper = 0;
}

/*!
 * \brief Save game status
 *
 * \return 1 en caso de error
 */
int mundo :: save_options(bool flag)
{
	FILE *fp;
	char full_path[100];

#ifdef WIN32
	strcpy(full_path, CEFERINO_SAVEGAME_FILE);
#else
	strcpy(full_path, getenv("HOME"));
	strcat(full_path, "/");
	strcat(full_path, CEFERINO_SAVEGAME_FILE);
#endif

	// If the savegame file does not exist 
	// then output the error message
	if ((fp = fopen(full_path, "r+b")) == NULL)
	{
#ifdef DEBUG_PRINTF
		printf("Unable to save game status. Save file '%s' does not exist\n", full_path);
#endif
		return 0;
	}
	// Write file content
	else
	{
		// Save game status
		if (flag)
		{
			int tmp = 1;
#ifdef DEBUG_PRINTF
			printf("Game status saved successfully to the file '%s'\n", full_path);
#endif
			fwrite(&opciones.musica, sizeof(opciones.musica), 1, fp);
			fwrite(&opciones.sonido, sizeof(opciones.sonido), 1, fp);
			fwrite(&tmp, sizeof(tmp), 1, fp);
			fwrite(&tmp, sizeof(tmp), 1, fp);

			fwrite(&opciones.saved_lives, sizeof(opciones.saved_lives), 1, fp);
			fwrite(&opciones.saved_score, sizeof(opciones.saved_score), 1, fp);
			fwrite(&opciones.saved_level, sizeof(opciones.saved_level), 1, fp);
			fwrite(&opciones.saved_time, sizeof(opciones.saved_time), 1, fp);
			fwrite(&opciones.saved_music, sizeof(opciones.saved_music), 1, fp);
			fwrite(&opciones.saved_wallpaper, sizeof(opciones.saved_wallpaper), 1, fp);
			savegame_present = 1;
		}
		// erase the current saved game status
		else
		{
#ifdef DEBUG_PRINTF
			printf("Erasing current game status\n");
#endif
			fclose(fp);
			fp = fopen(full_path, "wb");
			savegame_present = 0;
		}
	}

	fclose(fp);
	return 0;
}

/*!
 * \brief Lee las opciones del archivo 'ruta'
 *
 * \return 1 en caso de error
 */
int mundo :: cargar_opciones(void)
{
	FILE *fp;
	char full_path[100];

#ifdef WIN32
	strcpy(full_path, CEFERINO_SAVEGAME_FILE);
#else
	strcpy(full_path, getenv("HOME"));
	strcat(full_path, "/");
	strcat(full_path, CEFERINO_SAVEGAME_FILE);
#endif

	opciones.musica = 1;
	opciones.sonido = 1;
	opciones.enable_music = 1;
	opciones.enable_sound = 1;
	opciones.pantalla_completa = 0;
	opciones.modo_video = 0;

	set_default_options();

	// If the savegame file does not exist then create it
	if ((fp = fopen(full_path, "r+b")) == NULL)
	{
#ifdef DEBUG_PRINTF
		printf("The savegame file '%s' does not exist\n", full_path);
#endif		
		if ((fp = fopen(full_path, "w+b")) != NULL)
		{
#ifdef DEBUG_PRINTF
			printf("The savegame file '%s' has been created\n", full_path);
#endif
			fclose(fp);
		}
		else
		{
#ifdef DEBUG_PRINTF
			printf("Error creating savegame file '%s'\n", full_path);
#endif
		}
		return 0;
	}
	// Read file content
	else
	{
		// Get file size in bytes
		fseek(fp, 0, SEEK_END);
		int lSize = ftell(fp);
		rewind(fp);

		if (lSize == 40)
		{
			savegame_present = 1;

			fread(&opciones.musica, sizeof(opciones.musica), 1, fp);
			fread(&opciones.sonido, sizeof(opciones.sonido), 1, fp);
			fread(&opciones.enable_music, sizeof(opciones.enable_music), 1, fp);
			fread(&opciones.enable_sound, sizeof(opciones.enable_sound), 1, fp);

			fread(&opciones.saved_lives, sizeof(opciones.saved_lives), 1, fp);
			fread(&opciones.saved_score, sizeof(opciones.saved_score), 1, fp);
			fread(&opciones.saved_level, sizeof(opciones.saved_level), 1, fp);
			fread(&opciones.saved_time, sizeof(opciones.saved_time), 1, fp);		
			fread(&opciones.saved_music, sizeof(opciones.saved_music), 1, fp);
			fread(&opciones.saved_wallpaper, sizeof(opciones.saved_wallpaper), 1, fp);		
#ifdef DEBUG_PRINTF
			printf("Game status restored successfully from the file '%s'\n", full_path);

			printf("musica=%d\nsonido=%d\nenable_music=%d\nenable_sound=%d\nsaved_lives=%d\nsaved_score=%d\nsaved_level=%d\nsaved_time=%d\nsaved_music=%d\nsaved_wallpaper=%d\n\n", \
				opciones.musica, opciones.sonido, opciones.enable_music, opciones.enable_sound, opciones.saved_lives, opciones.saved_score, opciones.saved_level, opciones.saved_time, \
				opciones.saved_music, opciones.saved_wallpaper);
#endif
		}
		else
		{
			savegame_present = 0;
		}
	}

	fclose(fp);
	return 0;
}


/*!
 * \brief reinicia los contadores de tiempo, por ejemplo para evitar saltos luego de cargar imagenes
 */
void mundo :: reiniciar_reloj(void)
{
	t = tl = tg = SDL_GetTicks();
}


/*!
 * \brief return the save game status (1 if present, 0 otherwise)
 */
int mundo :: is_savegame_present(void)
{
	return savegame_present;
}


/*!
 * \brief elimina una escena y carga otra
 */
void mundo :: intercambiar_escenas(void)
{
	if (actual)
		delete actual;

	switch (escena)
	{
		case QUESTION:
			actual = new class question(QUESTION_START_AUDIO);
			break;

		case SAVE_GAME:
			actual = new class question(QUESTION_SAVE_GAME);
			break;

		case RESTORE_GAME:
			actual = new class question(QUESTION_RESTORE_GAME);
			break;

		case INTRO:
			actual = new class intro;
			break;

		case MENU:
			actual = new class menu;
			break;

		case JUEGO:
			actual = new class juego;
			break;
			
		case CREDITOS:
			actual = new class creditos;
			break;
			
		case MARCAS:
			actual = new class marcas;
			break;

		case FINAL:
			actual = new class final;
			break;

		case COMOJUGAR:
			actual = new class comojugar;
			break;

		case OPCIONES:
			actual = new class opciones;
			break;
	}

	if (actual->iniciar(this, opciones.modo_video, screen))
	{
#ifdef DEBUG_PRINTF
		printf(_("Error at changing scene\n"));
#endif
		terminar();
	}
	
	escena_anterior = escena;
	reiniciar_reloj();
	
}

/*!
 * \brief deja de procesar las teclas q y f
 */
void mundo :: deshabilitar_letras (void)
{
	leer_teclas = 0;
}

/*!
 * \brief comienza a procesar las teclas q y f como eventos
 */
void mundo :: habilitar_letras (void)
{
	leer_teclas = 1;
}


/*!
 * \brief imprime el primer mensaje 'cargando...'
 */
void mundo :: imprimir_cargando (void)
{
	SDL_Surface *ima;
	SDL_Rect srcrect;

	srcrect.x = SCREEN_RES_X/2 - 70;
	srcrect.y = SCREEN_RES_Y/2 - 20;
	srcrect.w = SCREEN_RES_X;
	srcrect.h = SCREEN_RES_Y;

	ima = IMG_Load (DATADIR "/ima/cargando.png");

	if (ima)
	{
		SDL_BlitSurface(ima, NULL, screen, &srcrect);
		SDL_Flip(screen);
	}
	else
	{
		printf(_("Can't find the file '%s'\n"), DATADIR"/ima/cargando.png");
	}
}

void mundo :: reiniciar (void)
{
#ifdef DEBUG_PRINTF
	printf ("Restarting\n");
#endif
}

void mundo :: pantalla_completa (void)
{
}
