/* ccchh_dooris_desktop_widget.c */

/**
 * CCCHH DoorIs Fnord Desktop Widget
 *
 * Desktop widget that indicates the main entrance door lock state at the
 * CCC in Hamburg.
 *
 * A switch in the door frame at the CCC Hamburg registers locking / unlocking
 * events. The current state is exported to a webserver.
 *
 * This widget shows an icon on the desktop indicating whether the door is
 * open or locked and optionally notifies the user upon state changes.
 *
 *
 * It is a shared librarary that implements a derived HDHomePluginItem (a 
 * desktop widget), with the help of the HD_DEFINE_PLUGIN_MODULE() macro, which 
 * also implements some standard functions that all desktop widget libraries 
 * should export.
 *
 * Note that the API contains the name "home plugin", but the correct name 
 * is now "desktop widget".
 * 
 * As with other custom GTK+ widgets, drawing happens only in a handler for the 
 * expose-event signal. This is triggered by invalidating the desktop widget's
 * area.
 *
 */


// TODO:
// - sprinkle with type-checking macros and g_assert
// - use more HildonCaption in settings dialog?
//   - checkbox buttons look fine without Caption
//   - PickerButton could use a larger font
// - check settings dialog for object/memory leaks
// - set .so version
// - add timers to cancel async download operations
// - support multiple instances




#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>

#include <glib.h>
#include <glib/gi18n-lib.h> /* Declares _() - alias for gettext() */

#include <gtk/gtk.h>

#include <gconf/gconf-client.h>

#include <hildon/hildon.h>
#include <libhildondesktop/libhildondesktop.h>

#include <conic.h> /* libconic - Internet Connectivity library */

#include <libgnomevfs/gnome-vfs.h>

#include "ccchh-dooris-widget.h"




#define DEBUG

#ifdef DEBUG

// redefine g_debug() to g_message()
// TODO: find out how to get debug logs from hildon-desktop
#define g_debug g_message

#define USER_AGENT_VERSION_STRING (debug version)

#else

// check g_debug statements for syntax errors but don't emit code
// (compiler should optimize it away)
#define g_debug(fmt...) do { g_message(fmt); } while (0)

#define USER_AGENT_VERSION_STRING (some version number)

#endif // DEBUG




/** This is like G_DEFINE_TYPE(), but it also 
 * implements standard functions such as hd_plugin_module_load(), 
 * which hildon-desktop expects to be able to call on this library.
 */
HD_DEFINE_PLUGIN_MODULE (CCCHHDoorIsDesktopWidget, ccchh_dooris_desktop_widget,
  HD_TYPE_HOME_PLUGIN_ITEM)




static void
load_settings(CCCHHDoorIsDesktopWidget *self)
{
	GError *error;

	error = NULL;

	g_debug("ccchh-dooris-widget: load_settings()");

	gchar *strval = gconf_client_get_string(self->gconf_client,
		GCONF_DIR"/update_url", &error);

	if (error) {
		g_warning("ccchh-dooris-widget: failed to get configuration value update_url: %s",
			error->message);
		g_error_free(error);
		error = NULL;
	} else if (strval) {
		if (self->url) {
			free(self->url);
			self->url = NULL;
		}

		self->url = gnome_vfs_make_uri_from_input(strval);
		g_debug("ccchh-dooris-widget: load_settings: url = %s", self->url);

		g_free(strval);
		strval = NULL;
	}

	gint intval = gconf_client_get_int(self->gconf_client,
		GCONF_DIR"/update_interval", &error);

	if (error) {
		g_warning("ccchh-dooris-widget: failed to get configuration value update_interval: %s",
			error->message);
		g_error_free(error);
		error = NULL;
	} else if (intval) {
		self->update = intval;
		if (self->update < UPDATE_INTERVAL_MIN) {
			g_warning("ccchh-dooris-widget: self->update < UPDATE_INTERVAL_MIN, maybe a bug?");
			self->update = UPDATE_INTERVAL_MIN;
		} else if (self->update > UPDATE_INTERVAL_MAX) {
			g_warning("ccchh-dooris-widget: self->update > UPDATE_INTERVAL_MAX, maybe a bug?");
			self->update = UPDATE_INTERVAL_MAX;
		}
		g_debug("ccchh-dooris-widget: load_settings: update = %ld", (long)self->update);
	}

	gboolean boolval = gconf_client_get_bool(self->gconf_client,
		GCONF_DIR"/notify_enabled", &error);

	if (error) {
		g_warning("ccchh-dooris-widget: failed to get configuration value notify_enabled: %s",
			error->message);
		g_error_free(error);
		error = NULL;
	} else {
		self->notify = boolval;
		g_debug("ccchh-dooris-widget: load_settings: notify = %d", self->notify);
	}


	boolval = gconf_client_get_bool(self->gconf_client,
		GCONF_DIR"/use_connection", &error);

	if (error) {
		g_warning("ccchh-dooris-widget: failed to get configuration value use_connection: %s",
			error->message);
		g_error_free(error);
		error = NULL;
	} else {
		self->use_connection = boolval;
		g_debug("ccchh-dooris-widget: load_settings: use_connection = %d", self->use_connection);
	}

	boolval = gconf_client_get_bool(self->gconf_client,
		GCONF_DIR"/force_connection", &error);

	if (error) {
		g_warning("ccchh-dooris-widget: failed to get configuration value force_connection: %s",
			error->message);
		g_error_free(error);
		error = NULL;
	} else {
		self->force_connection = boolval;
		g_debug("ccchh-dooris-widget: load_settings: force_connection = %d", self->force_connection);
	}
}


static void
save_settings(CCCHHDoorIsDesktopWidget *self)
{
	GError *error;

	error = NULL;

	g_debug("ccchh-dooris-widget: save_settings()");

	if (!gconf_client_set_string(self->gconf_client,
		GCONF_DIR"/update_url", self->url, &error))
	{
		g_warning("ccchh-dooris-widget: failed to set configuration value update_url: %s",
			error->message);
		g_error_free(error);
		error = NULL;
	}

	if (!gconf_client_set_int(self->gconf_client,
		GCONF_DIR"/update_interval", self->update, &error))
	{
		g_warning("ccchh-dooris-widget: failed to set configuration value update_interval: %s",
			error->message);
		g_error_free(error);
		error = NULL;
	}

	if (!gconf_client_set_bool(self->gconf_client,
		GCONF_DIR"/notify_enabled", self->notify, &error))
	{
		g_warning("ccchh-dooris-widget: failed to set configuration value notify_enabled: %s",
			error->message);
		g_error_free(error);
		error = NULL;
	}

	if (!gconf_client_set_bool(self->gconf_client,
		GCONF_DIR"/use_connection", self->use_connection, &error))
	{
		g_warning("ccchh-dooris-widget: failed to set configuration value use_connection: %s",
			error->message);
		g_error_free(error);
		error = NULL;
	}

	if (!gconf_client_set_bool(self->gconf_client,
		GCONF_DIR"/force_connection", self->force_connection, &error))
	{
		g_warning("ccchh-dooris-widget: failed to set configuration value force_connection: %s",
			error->message);
		g_error_free(error);
		error = NULL;
	}
}

// stop update timer
// returns TRUE if timer was stopped
// returns FALSE if no timer was running
static inline gboolean
stop_update_timer(CCCHHDoorIsDesktopWidget *self)
{
	// stop timer if it was running
	if (self->timeout_handler) {
		g_debug("ccchh-dooris-widget: stopping update timer");
		g_source_remove (self->timeout_handler);
		self->timeout_handler = 0;
		return TRUE;
	}
	return FALSE;
}


// start update timer
// returns TRUE if timer was started
// returns FALSE if timer was already running
// TODO: check return code from g_timeout_add_seconds
static inline gboolean
start_update_timer(CCCHHDoorIsDesktopWidget *self)
{
	if (!self->timeout_handler) {
		g_debug("ccchh-dooris-widget: starting update timer");
		if (self->update < UPDATE_INTERVAL_MIN) {
			g_warning("ccchh-dooris-widget: self->update < UPDATE_INTERVAL_MIN, maybe a bug?");
			self->update = UPDATE_INTERVAL_MIN;
		} else if (self->update > UPDATE_INTERVAL_MAX) {
			g_warning("ccchh-dooris-widget: self->update > UPDATE_INTERVAL_MAX, maybe a bug?");
			self->update = UPDATE_INTERVAL_MAX;
		}
		self->timeout_handler = g_timeout_add_seconds (self->update*60, ccchh_dooris_desktop_widget_on_timeout, self);
		return TRUE;
	}
	return FALSE;
}


// install libconic connection event handler
// returns TRUE if handler was installed
// returns FALSE if handler was already set
// TODO: check return code from g_signal_connect
static inline gboolean
install_connection_handler(CCCHHDoorIsDesktopWidget *self)
{
	if (!self->connection_handler) {
		// set connected to FALSE, initial event may set it to TRUE
		self->connected = FALSE;
		/* Connect signal to receive connection events */
		self->connection_handler = g_signal_connect (self->connection, "connection-event",
			G_CALLBACK (ccchh_dooris_desktop_widget_connection_event), self);
		/* Set automatic events */
		g_object_set (self->connection, "automatic-connection-events", TRUE, NULL);
		g_debug("ccchh-dooris-widget: connection monitoring enabled");
		return TRUE;
	}

	return FALSE;
}


// remove libconic connection event handler
// returns TRUE if handler was removed
// returns FALSE if no handler was set
static inline gboolean
remove_connection_handler(CCCHHDoorIsDesktopWidget *self)
{
	if (self->connection_handler) {
		g_object_set (self->connection, "automatic-connection-events", FALSE, NULL);
		g_signal_handler_disconnect (self->connection, self->connection_handler);
		self->connection_handler = 0;
		g_debug("ccchh-dooris-widget: connection monitoring disabled");
		return TRUE;
	}

	return FALSE;
}





static void
draw (CCCHHDoorIsDesktopWidget *self, cairo_t *cr)
{
//  GtkWidget *widget = GTK_WIDGET(self);
  
	g_debug("ccchh-dooris-widget: draw()");

	if (!self->image_current) {
		g_debug("ccchh-dooris-widget: draw() without image");
		self->image_current = cairo_image_surface_create_from_png(IMAGE_DOORIS_FNORD);
	}

	cairo_save(cr);
	
	// transparent background
	cairo_set_source_rgba(cr, IMAGE_COLOR_BG, IMAGE_ALPHA_BG);
	cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
	cairo_paint(cr);

	// paint icon
	cairo_set_source_surface(cr, self->image_current, 5, 5);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	cairo_paint_with_alpha(cr, self->current_alpha);

	// if not connected, overlay icon with "disconnected" image
	if (!self->connected) {
		g_debug("ccchh-dooris-widget: not connected, overlay image");
		cairo_surface_t *image_overlay = cairo_image_surface_create_from_png(IMAGE_OVERLAY_DISCONNECTED);
		cairo_set_source_surface(cr, image_overlay, 5, 5);
		cairo_paint_with_alpha(cr, self->current_alpha);
		cairo_surface_destroy(image_overlay);
	}

	// if button pressed, hightligth with white mask
	if (self->highlighted) {
		cairo_set_source_rgba(cr, IMAGE_COLOR_HIGHLIGHT, IMAGE_ALPHA_HIGHLIGHT);
		cairo_mask_surface(cr, self->image_current, 5, 5);
	}

	cairo_restore(cr);
}


static void
show_status_banner (CCCHHDoorIsDesktopWidget *self)
{
	time_t timenow, timediff;
	gchar *text, *ts_text;

	g_debug("ccchh-dooris-widget: show_status_banner()");

	time (&timenow);

	// format the time elapsed since last state change
	// TODO: is it correct and portable to cast time_t to ulong?

	if (timenow != -1 && self->last_change_ts) {
		timediff = timenow - self->last_change_ts;

		if (timediff < 60) {
			ts_text = g_strdup_printf(_("(%lds ago)"), (ulong)timediff);
		} else if (timediff < 3600) {
			ts_text = g_strdup_printf(_("(%ldm %lds ago)"), (ulong)timediff / 60, (ulong)timediff % 60);
		} else {
			ts_text = g_strdup_printf(_("(%ldh %ldm %lds ago)"), (ulong)timediff / 3600, (ulong)(timediff % 3600) / 60, (ulong)timediff % 60);
		}
	} else {
		ts_text = g_strdup("");
	}

	// assemble string to display
	switch (self->current_state) {
		case DOORIS_OPEN:
			text = g_strdup_printf(("%s %s %s"),
				_("CCCHH: Door <b>unlocked</b>"), ts_text,
				(self->connected) ? ("") : _("(offline)"));
				break;
//				(self->connected) ? _("(online)") : _("(offline)"));
		case DOORIS_CLOSED:
			text = g_strdup_printf(("%s %s %s"),
				_("CCCHH: Door locked"), ts_text,
				(self->connected) ? ("") : _("(offline)"));
				break;
		default:
			text = g_strdup_printf(("%s %s"),
				_("CCCHH: Unknown"),
				(self->connected) ? ("") : _("(offline)"));
				break;
	}

	// display banner on screen
	hildon_banner_show_information_with_markup (GTK_WIDGET (self),
		NULL, /* icon_name - ignored */
		text);

	g_free (text);
	g_free (ts_text);
}


static void
force_redraw (CCCHHDoorIsDesktopWidget *self)
{
	g_debug("ccchh-dooris-widget: force_redraw()");

	/* Invalidate the drawing: */
	GtkWidget *widget = GTK_WIDGET (self);

	if (!widget->window)
		return;

	/* Redraw the cairo canvas completely by exposing it: */
	GdkRegion *region = gdk_drawable_get_clip_region (widget->window);
	if (!region)
		return;
	gdk_window_invalidate_region (widget->window, region, TRUE);

	// process expose event
	gdk_window_process_updates (widget->window, TRUE);

	gdk_region_destroy (region);
}


// end private functions




// async update with callbacks

// lock fetch mutex, redraw transparent button
static inline gboolean
fetch_update_lock (CCCHHDoorIsDesktopWidget *self)
{
	if (!g_static_mutex_trylock(&(self->fetch_lock))) {
		return FALSE;
	}

	// acknowledge button, make icon transparent
	self->current_alpha = IMAGE_ALPHA_PROGRESS;
	force_redraw(self);

	return TRUE;
}


// unlock fetch mutex, redraw opaque button
static inline void
fetch_update_unlock (CCCHHDoorIsDesktopWidget *self)
{
	self->current_alpha = IMAGE_ALPHA_NORMAL;
	force_redraw(self);

	g_static_mutex_unlock(&(self->fetch_lock));
}


// initialize async update from server, lock mutex
static void
fetch_update_start (CCCHHDoorIsDesktopWidget *self)
{
	g_debug("ccchh-dooris-widget: fetch_update_start()");

	if (!self->connected) {
		g_debug("ccchh-dooris-widget: not connected, not fetching new state");
		return;
	}

	if (!fetch_update_lock(self)) {
		g_message("ccchh-dooris-widget: fetch_lock set, not updating");
		return;
	}

	g_debug("ccchh-dooris-widget: begin update from URL: %s", self->url);

	// TODO: set timeout to cancel async open

	gnome_vfs_async_open(&self->gvfs_handle, self->url, GNOME_VFS_OPEN_READ,
		GNOME_VFS_PRIORITY_MIN, fetch_update_async_open_cb, self);
}


// handle async open callback, begin read
static void
fetch_update_async_open_cb (GnomeVFSAsyncHandle *handle, GnomeVFSResult result,
    gpointer data)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (data);

	g_debug("ccchh-dooris-widget: fetch_update_async_open_cb()");

	if (result != GNOME_VFS_OK) {
		g_warning("ccchh-dooris-widget: error opening URL: %s", gnome_vfs_result_to_string (result));
		fetch_update_process(self, DOORIS_FNORD, 0L, 0L);
		return;
	}

	// reuse input buffer, ensure it has at least MAX_HTTP_SIZE+1
	if (!(self->gvfs_input_buffer = g_try_realloc(self->gvfs_input_buffer, MAX_HTTP_SIZE+1))) {
		g_warning("ccchh-dooris-widget: failed to allocate input buffer");
		fetch_update_process(self, DOORIS_FNORD, 0L, 0L);
		return;
	}
	// initialize input buffer with zeroes, so strtok won't read beyond its end
	memset(self->gvfs_input_buffer, 0, MAX_HTTP_SIZE+1);

	// TODO: set timeout to cancel async read

	gnome_vfs_async_read (handle, self->gvfs_input_buffer, MAX_HTTP_SIZE,
		fetch_update_async_read_cb, self);
}


// handle async read callback, begin close, parse input
static void
fetch_update_async_read_cb (GnomeVFSAsyncHandle *handle, GnomeVFSResult result,
	gpointer buffer, GnomeVFSFileSize bytes_requested, GnomeVFSFileSize bytes_read,
	gpointer data)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (data);

	g_debug("ccchh-dooris-widget: fetch_update_async_read_cb()");

	if (result != GNOME_VFS_OK) {
		g_warning("ccchh-dooris-widget: read error: %s", gnome_vfs_result_to_string (result));
		fetch_update_process(self, DOORIS_FNORD, 0L, 0L);
		return;
	}

	g_debug("ccchh-dooris-widget: read %"GNOME_VFS_SIZE_FORMAT_STR" bytes", bytes_read);

	// TODO: set timeout to cancel close

	// close file
	gnome_vfs_async_close (handle, fetch_update_async_close_cb, self);

	// meanwhile, parse input buffer

	gulong ts, new_change_ts, new_report_ts;
	dooris_state_t new_state;
	char *line, *tokctx;

	new_state = DOORIS_FNORD;
	new_change_ts = 0L;
	new_report_ts = 0L;
	tokctx = NULL;

	// first line of body is either "locked" or "unlocked"

	if (!(line = strtok_r(self->gvfs_input_buffer, "\n", &tokctx))) {
		g_warning("ccchh-dooris-widget: first line of response not found");
		fetch_update_process(self, new_state, new_change_ts, new_report_ts);
		return;
	}

	if (!strcmp(line, "locked")) {
		new_state = DOORIS_CLOSED;
		g_debug("ccchh-dooris-widget: current state: closed");
	} else if (!strcmp(line, "unlocked")) {
		new_state = DOORIS_OPEN;
		g_debug("ccchh-dooris-widget: current state: open");
	} else {
		g_warning("ccchh-dooris-widget: could not parse response");
		fetch_update_process(self, new_state, new_change_ts, new_report_ts);
		return;
	}

	// second line of body is last sensor report timestamp

	if ((line = strtok_r(NULL, "\n", &tokctx))) {
		errno = 0;
		ts = strtoul(line, (char **) NULL, 10);
		if (ts && !errno) {
			new_report_ts = ts;
			g_debug("ccchh-dooris-widget: last_report_ts: %lu, errno: %i\n", ts, errno);
		} else {
			g_warning("ccchh-dooris-widget: could not parse last_report_ts");
		}

		// third line of body is last state change timestamp

		if ((line = strtok_r(NULL, "\n", &tokctx))) {
			errno = 0;
			ts = strtoul(line, (char **) NULL, 10);
			if (ts && !errno) {
				new_change_ts = ts;
				g_debug("ccchh-dooris-widget: last_change_ts: %lu, errno: %i\n", ts, errno);
			} else {
				g_warning("ccchh-dooris-widget: could not parse last_change_ts");
			}
		}
	}

	fetch_update_process(self, new_state, new_change_ts, new_report_ts);
}


// handle async close callback
static void
fetch_update_async_close_cb (GnomeVFSAsyncHandle *handle, GnomeVFSResult result,
	gpointer data)
{
//	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (data);

	g_debug("ccchh-dooris-widget: fetch_update_async_close_cb()");

	if (result != GNOME_VFS_OK) {
		g_debug("ccchh-dooris-widget: error on close: %s", gnome_vfs_result_to_string (result));
		// TODO: what now?
	}
}


// update internal state, unlock mutex
static void
fetch_update_process (CCCHHDoorIsDesktopWidget *self, dooris_state_t new_state,
	gulong new_change_ts, gulong new_report_ts)
{
	g_debug("ccchh-dooris-widget: fetch_update_process()");

	self->current_state = new_state;
	self->last_change_ts = new_change_ts;
	self->last_report_ts = new_report_ts;

	if (new_state == self->last_state) {
		g_debug("ccchh-dooris-widget: state unchanged");

		if (self->force_banner) {
			self->force_banner = FALSE;
			show_status_banner(self);
		}

		fetch_update_unlock(self);
		return;
	}

	// release old icon surface
	if (self->image_current) {
		cairo_surface_destroy(self->image_current);
		self->image_current = NULL;
	}

	// set new icon surface
	switch (self->current_state) {
		case DOORIS_OPEN:
			self->image_current = cairo_image_surface_create_from_png(IMAGE_DOORIS_OPEN);
			g_debug("ccchh-dooris-widget: image_current set to open");
			break;
		case DOORIS_CLOSED:
			self->image_current = cairo_image_surface_create_from_png(IMAGE_DOORIS_CLOSED);
			g_debug("ccchh-dooris-widget: image_current set to closed");
			break;
		default:
			self->image_current = cairo_image_surface_create_from_png(IMAGE_DOORIS_FNORD);
			g_debug("ccchh-dooris-widget: image_current set to fnord");
			break;
	}

	self->last_state = self->current_state;

	// state is valid and has changed since last known state
	if (self->current_state != DOORIS_FNORD && self->current_state != self->last_known_state) {

		self->last_known_state = self->current_state;
		g_debug("ccchh-dooris-widget: state changed");

		// state changed: redraw icon
		force_redraw(self);

		// flash banner if notifications are enabled
		if (self->notify) {
			self->force_banner = TRUE;
		}
	}

	if (self->force_banner) {
		self->force_banner = FALSE;
		show_status_banner(self);
	}

	fetch_update_unlock(self);
}




// event handlers

static gboolean
ccchh_dooris_desktop_widget_on_timeout (gpointer data)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (data);
  
	g_debug("ccchh-dooris-widget: ccchh_dooris_desktop_widget_on_timeout()");

	// stop timer when offline
	// connection event starts it again
	if (!self->connected) {
		return FALSE;
	}

	fetch_update_start(self);

	return TRUE; /* keep running this event */
}


static gboolean
ccchh_dooris_desktop_widget_expose_event (GtkWidget *widget, GdkEventExpose *event)
{    
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (widget);

	g_debug("ccchh-dooris-widget: ccchh_dooris_desktop_widget_expose_event()");

	if (!self->image_current) {
		g_debug("ccchh-dooris-widget: expose event without image");
		self->image_current = cairo_image_surface_create_from_png(IMAGE_DOORIS_FNORD);
		// no state yet, begin update
		fetch_update_start(self);
	}

	cairo_t *cr  = gdk_cairo_create (widget->window);

	/* Clip only the exposed area of the cairo context,  
	 * to potentially avoid unnecessary drawing:
	 */
	cairo_rectangle (cr,
		event->area.x, event->area.y,
		event->area.width, event->area.height);
	cairo_clip (cr);

	draw (CCCHH_DOORIS_DESKTOP_WIDGET (widget), cr);

	cairo_destroy (cr);

	return FALSE;
}


// handle the button press event, highlight icon
static gboolean 
ccchh_dooris_desktop_widget_on_button_press_event (GtkWidget *widget, 
  GdkEventButton * event G_GNUC_UNUSED, gpointer user_data G_GNUC_UNUSED)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (widget);

	g_debug("ccchh-dooris-widget: ccchh_dooris_desktop_widget_on_button_press_event()");
 
	// TODO: do we really need a button lock? fetch_update locks itself now
	if (!g_static_mutex_trylock(&(self->button_lock))) {
		g_debug("ccchh-dooris-widget: button_lock set, not updating");
		return TRUE;
	}

	self->highlighted = TRUE;
	force_redraw(self);

	return TRUE; /* Prevent further handling of this signal. */
}


// handle the leave notify event, remove highlight
static gboolean 
ccchh_dooris_desktop_widget_on_button_leave_event (GtkWidget *widget, 
  GdkEventButton * event G_GNUC_UNUSED, gpointer user_data G_GNUC_UNUSED)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (widget);

	g_debug("ccchh-dooris-widget: ccchh_dooris_desktop_widget_on_button_leave_event()");
 
	self->highlighted = FALSE;
	force_redraw(self);

	g_static_mutex_unlock(&(self->button_lock));

	return TRUE; /* Prevent further handling of this signal. */
}


// handle the button release event, begin update, remove highlight
static gboolean 
ccchh_dooris_desktop_widget_on_button_release_event (GtkWidget *widget, 
  GdkEventButton * event G_GNUC_UNUSED, gpointer user_data G_GNUC_UNUSED)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (widget);

	g_debug("ccchh-dooris-widget: ccchh_dooris_desktop_widget_on_button_release_event()");
 
	// request connection from libconic, if offline and
	// force_connection is enabled
	// TODO: this does not work immediately, the call to update()
	// will still see !connected. maybe wait a bit or use a timer event?
	if (!self->connected && self->force_connection) {
		g_debug("ccchh-dooris-widget: not connected, requesting connection");
		/* Request connection and check for the result */
		gboolean success = con_ic_connection_connect(self->connection,
			CON_IC_CONNECT_FLAG_NONE);
		if (!success) 
			g_warning("ccchh-dooris-widget: request for connection failed");
	}

	self->force_banner = TRUE;
	fetch_update_start(self);

	// if a timer was running, reset it
	if (stop_update_timer(self)) {
		start_update_timer(self);
	}

	self->highlighted = FALSE;
	force_redraw(self);

	g_static_mutex_unlock(&(self->button_lock));

	return TRUE; /* Prevent further handling of this signal. */
}


static void
ccchh_dooris_desktop_widget_connection_event(ConIcConnection *connection,
  ConIcConnectionEvent *event, gpointer user_data)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (user_data);

	g_debug("ccchh-dooris-widget: ccchh_dooris_desktop_widget_connection_event()");

	// libconic signalled a connection state change

	ConIcConnectionStatus status = con_ic_connection_event_get_status(event);
	const gchar *iap_id = con_ic_event_get_iap_id(CON_IC_EVENT(event));

	// Problem: When switching from GPRS to/from WLAN
	// we receive STATUS_CONNECTED for the new IAP first,
	// then STATUS_DISCONNECTED for the previous IAP

	// in STATUS_DISCONNECTING, save disconnecting IAP ID and
	// set self->connected to FALSE
	// STATUS_CONNECTED sets self->connected to TRUE
	// if STATUS_DISCONNECTED is called with the same IAP ID
	// and self->connected has reverted to TRUE, ignore the event

	switch (status) {
		case CON_IC_STATUS_DISCONNECTING:

			// save disconnecting IAP ID

			if (self->disconnecting) {
				g_free(self->disconnecting);
				self->disconnecting = NULL;
			}

			if (iap_id)
				self->disconnecting = g_strdup(iap_id);

			self->connected = FALSE;

			break;

		case CON_IC_STATUS_CONNECTED:

			// we are online now

			if (!self->connected) {
				self->connected = TRUE;
				g_debug("ccchh-dooris-widget: new connection state: online");

				// start timer
				start_update_timer(self);

				// force redraw to remove disconnection overlay
				force_redraw(self);
			}

			break;

		case CON_IC_STATUS_DISCONNECTED:

			if (iap_id && self->disconnecting && self->connected &&
					!strcmp(iap_id, self->disconnecting))
			{

				g_debug("ccchh-dooris-widget: ignoring offline event from disconnecting IAP");

			} else {

				// we are offline now

				self->connected = FALSE;
				g_debug("ccchh-dooris-widget: new connection state: offline");

				// don't stop timer here
				// it will deactivate itsef if !connected
				// frequent stop/start would prevent it from firing

				// force redraw to add disconnection overlay
				force_redraw(self);
			}

			if (self->disconnecting) {
				g_free(self->disconnecting);
				self->disconnecting = NULL;
			}

			break;

		case CON_IC_STATUS_NETWORK_UP:

			// Nothing to do here
			break;
	}
}


static void 
ccchh_dooris_desktop_widget_on_settings_dialog_response (GtkDialog *dialog,
  gint response_id, gpointer user_data)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (user_data);
	g_assert (self->settings_window);
	g_assert (self->settings_window == GTK_WIDGET (dialog));
  
	g_debug("ccchh-dooris-widget: ccchh_dooris_desktop_widget_on_settings_dialog_response()");

	// only use new settings if user pressed the OK button
	// TODO: add reset button to restore defaults
	if (response_id == GTK_RESPONSE_ACCEPT) {
		g_debug("ccchh-dooris-widget: response accept");

		// status URL
		if (self->url) {
			free(self->url);
			self->url = NULL;
		}

		// try to build a valid URI from user input
		self->url = gnome_vfs_make_uri_from_input(gtk_entry_get_text(GTK_ENTRY (self->url_entry)));
		g_debug("ccchh-dooris-widget: new URL: %s\n", self->url);


		// update interval
		const gchar *choice = hildon_button_get_value(HILDON_BUTTON (self->update_picker));
		self->update = CLAMP(g_ascii_strtod (choice, NULL), UPDATE_INTERVAL_MIN, UPDATE_INTERVAL_MAX);
		g_debug("ccchh-dooris-widget: new update interval: %ld\n", (long)self->update);

		// if a timer was running, reset it
		if (stop_update_timer(self)) {
			start_update_timer(self);
		}

		// banner notifications
		self->notify = hildon_check_button_get_active(HILDON_CHECK_BUTTON(self->notify_button));
		if (self->notify) {
			g_debug("ccchh-dooris-widget: notification enabled");
		} else {
			g_debug("ccchh-dooris-widget: notification disabled");
		}

		// monitor internet connection state, only update if online
		self->use_connection = hildon_check_button_get_active(HILDON_CHECK_BUTTON(self->connection_button));

		if (self->use_connection) {
			install_connection_handler(self);
		} else {
			remove_connection_handler(self);

			start_update_timer(self);

			// force connection state to online
			// redraw if necessary to remove disconnection overlay
			if (!self->connected) {
				self->connected = TRUE;
				force_redraw(self);
			}
		}

		// button forces internet connection
		self->force_connection = hildon_check_button_get_active(HILDON_CHECK_BUTTON(self->connforce_button));
		if (self->force_connection) {
			g_debug("ccchh-dooris-widget: button forces connection");
		} else {
			g_debug("ccchh-dooris-widget: button does not force connection");
		}

		save_settings(self);
	}

	gtk_widget_destroy (self->settings_window);
	// TODO: does this leak any object references such as GtkSizeGroup
	// or HildonCaption?
	// isn't there a better way to this? array+memset?
	self->settings_window = NULL;
	self->url_entry = NULL;
	self->update_picker = NULL;
	self->notify_button = NULL;
	self->connection_button = NULL;
	self->connforce_button = NULL;
}


// update interval picker callback - restrict value to UPDATE_INTERVAL_MIN - UPDATE_INTERVAL_MAX
void update_picker_changed_value_cb (HildonPickerButton *picker, gpointer data)
{
	gdouble number = 0;
	const gchar *choice = hildon_button_get_value(HILDON_BUTTON (picker));
	number = CLAMP(g_ascii_strtod (choice, NULL), UPDATE_INTERVAL_MIN, UPDATE_INTERVAL_MAX);
	gchar *value_text = g_strdup_printf ("%d", (int) number);
	hildon_button_set_value(HILDON_BUTTON (picker), value_text);
	g_free(value_text);
}


static void 
ccchh_dooris_desktop_widget_on_show_settings (GtkWidget *widget G_GNUC_UNUSED, 
  gpointer user_data G_GNUC_UNUSED)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (widget);
 
	g_debug("ccchh-dooris-widget: ccchh_dooris_desktop_widget_on_show_settings()");

	// could this be called when the settings window is already open?
	if (!self->settings_window) {
		/* The Hildon HIG says "Dialogs should be used for small tasks" so 
		 * we use GtkDialog instead of HildonWindow: */
		self->settings_window = gtk_dialog_new_with_buttons(
			_("CCCHH DoorIs Widget Settings"), 
			NULL /* parent window */, 
			GTK_DIALOG_NO_SEPARATOR /* flags */, 
			GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
			NULL);
		// call dialog_response() on close
		g_signal_connect (self->settings_window, "response", 
			G_CALLBACK (ccchh_dooris_desktop_widget_on_settings_dialog_response), 
			self);

		// arrange the inputs
		// we put controls and labels into hbox-es
		// and stack them into the dialog's vbox

		GtkWidget *vbox = GTK_DIALOG(self->settings_window)->vbox;
		gtk_container_set_border_width(GTK_CONTAINER (vbox), HILDON_MARGIN_DEFAULT);

		GtkWidget *hbox = gtk_hbox_new (FALSE, HILDON_MARGIN_DEFAULT);
		gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

		GtkSizeGroup *sizegroup = gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL);

		// status url
		self->url_entry = hildon_entry_new (HILDON_SIZE_FINGER_HEIGHT);
		GtkWidget *caption = hildon_caption_new(NULL, _("Status URL:"),
			self->url_entry, NULL, HILDON_CAPTION_MANDATORY);
		gtk_box_pack_start (GTK_BOX (hbox), caption, TRUE, TRUE, 0);

		hbox = gtk_hbox_new (FALSE, HILDON_MARGIN_DEFAULT);
		gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

		// predefined choices for update interval selector
		HildonTouchSelector *selector = HILDON_TOUCH_SELECTOR (hildon_touch_selector_entry_new_text());
		hildon_touch_selector_append_text (selector, "5");
		hildon_touch_selector_append_text (selector, "10");
		hildon_touch_selector_append_text (selector, "15");
		hildon_touch_selector_append_text (selector, "20");
		hildon_touch_selector_append_text (selector, "25");
		hildon_touch_selector_append_text (selector, "30");
		hildon_touch_selector_append_text (selector, "45");
		hildon_touch_selector_append_text (selector, "60");

		// update interval
		self->update_picker = hildon_picker_button_new (HILDON_SIZE_FINGER_HEIGHT,
			HILDON_BUTTON_ARRANGEMENT_VERTICAL);
		hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (self->update_picker), selector);
		// install callback function to validate input
		g_signal_connect (G_OBJECT (self->update_picker), "value-changed",
			G_CALLBACK (update_picker_changed_value_cb), NULL);
		// align left, center vertically
		gtk_button_set_alignment(GTK_BUTTON(self->update_picker), 0, 0.5);
		caption = hildon_caption_new(sizegroup, _("Update interval (minutes):"),
			self->update_picker, NULL, HILDON_CAPTION_MANDATORY);
		gtk_box_pack_start (GTK_BOX (hbox), caption, TRUE, TRUE, 0);

		hbox = gtk_hbox_new (FALSE, HILDON_MARGIN_DEFAULT);
		gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
    
		// banner notifications
		self->notify_button = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
		gtk_button_set_label(GTK_BUTTON (self->notify_button), _("Notify on state change"));
		// align left, center vertically
		gtk_button_set_alignment(GTK_BUTTON(self->notify_button), 0, 0.5);
		gtk_box_pack_start (GTK_BOX (hbox), self->notify_button, TRUE, TRUE, 0);

		hbox = gtk_hbox_new (FALSE, HILDON_MARGIN_DEFAULT);
		gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

		// monitor internet connection, only update when online
		self->connection_button = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
		gtk_button_set_label(GTK_BUTTON (self->connection_button), _("Only update when online"));
		gtk_button_set_alignment(GTK_BUTTON(self->connection_button), 0, 0.5);
		gtk_box_pack_start (GTK_BOX (hbox), self->connection_button, TRUE, TRUE, 0);

		hbox = gtk_hbox_new (FALSE, HILDON_MARGIN_DEFAULT);
		gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

		// button forces connection
		self->connforce_button = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
		gtk_button_set_label(GTK_BUTTON (self->connforce_button), _("Button requests internet connection"));
		gtk_button_set_alignment(GTK_BUTTON(self->connforce_button), 0, 0.5);
		gtk_box_pack_start (GTK_BOX (hbox), self->connforce_button, TRUE, TRUE, 0);

		g_object_unref(sizegroup);
	}

	// set current values
	gtk_entry_set_text(GTK_ENTRY(self->url_entry), self->url);

	// no default selection
	// TODO: find the selector nearest to the current value and activate it
	hildon_picker_button_set_active (HILDON_PICKER_BUTTON (self->update_picker), -1);
	char *value_text = g_strdup_printf ("%d", self->update);
	hildon_button_set_value(HILDON_BUTTON (self->update_picker), value_text);
	g_free(value_text);

	hildon_check_button_set_active(HILDON_CHECK_BUTTON(self->notify_button), self->notify);
	hildon_check_button_set_active(HILDON_CHECK_BUTTON(self->connection_button), self->use_connection);
	hildon_check_button_set_active(HILDON_CHECK_BUTTON(self->connforce_button), self->force_connection);

	gtk_widget_show_all (self->settings_window);
}




static void
ccchh_dooris_desktop_widget_realize (GtkWidget *widget)
{
	g_debug("ccchh-dooris-widget: ccchh_dooris_desktop_widget_realize()");

	/* Use An RGBA colormap rather than RGB, 
	 * so we can use transparency in our expose_event() implementation.
	 */
	GdkScreen *screen = gtk_widget_get_screen (widget);
	gtk_widget_set_colormap (widget,
		gdk_screen_get_rgba_colormap (screen));

	gtk_widget_set_app_paintable (widget, TRUE);

	/* Call the base class's implementation: */
	GTK_WIDGET_CLASS (ccchh_dooris_desktop_widget_parent_class)->realize (widget);
}


// TODO: this function is called twice. why?
static void
ccchh_dooris_desktop_widget_dispose (GObject *object)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (object);
 
	g_debug("ccchh-dooris-widget: ccchh_dooris_desktop_widget_dispose()");

	// remove settings window if it is still open
	if (self->settings_window) {
		gtk_widget_destroy (self->settings_window);
		self->settings_window = NULL;
		self->url_entry = NULL;
	}
   
	/* Call the base class's implementation: */
	G_OBJECT_CLASS (ccchh_dooris_desktop_widget_parent_class)->dispose (object);
}


static void
ccchh_dooris_desktop_widget_finalize (GObject *object)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (object);

	g_debug("ccchh-dooris-widget: ccchh_dooris_desktop_widget_finalize()");
 
	stop_update_timer(self);
  
	remove_connection_handler(self);

	// destroy icon surface if we still hold one
	if (self->image_current && cairo_surface_get_reference_count(self->image_current) > 0) {
		cairo_surface_destroy(self->image_current);
	}

	// don'shut down gnome vfs library here
	// other parts of the desktop might still use it
 
	if (self->url) {
		free(self->url);
		self->url = NULL;
	}

	g_static_mutex_free(&(self->button_lock));
	g_static_mutex_free(&(self->fetch_lock));

/*
	if (self->gvfs_handle) {
		g_object_unref(self->gvfs_handle);
		self->gvfs_handle = NULL;
	}
*/

	if (self->gvfs_input_buffer) {
		g_free(self->gvfs_input_buffer);
		self->gvfs_input_buffer = NULL;
	}

	if (self->disconnecting) {
		g_free(self->disconnecting);
		self->disconnecting = NULL;
	}

	if (self->gconf_client) {
		g_object_unref(self->gconf_client);
		self->gconf_client = NULL;
	}

	/* Call the base class's implementation: */
	G_OBJECT_CLASS (ccchh_dooris_desktop_widget_parent_class)->finalize (object);
}


static void
ccchh_dooris_desktop_widget_class_init (CCCHHDoorIsDesktopWidgetClass *klass)
{

	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);

	g_debug("ccchh-dooris-widget: ccchh_dooris_desktop_widget_class_init()");

	object_class->dispose = ccchh_dooris_desktop_widget_dispose;
	object_class->finalize = ccchh_dooris_desktop_widget_finalize;
 
	widget_class->realize = ccchh_dooris_desktop_widget_realize;
	widget_class->expose_event = ccchh_dooris_desktop_widget_expose_event;
}


static void
ccchh_dooris_desktop_widget_class_finalize (CCCHHDoorIsDesktopWidgetClass *klass G_GNUC_UNUSED)
{
	g_debug("ccchh-dooris-widget: ccchh_dooris_desktop_widget_class_finalize()");
}


static void
ccchh_dooris_desktop_widget_init (CCCHHDoorIsDesktopWidget *self)
{
	g_debug("ccchh-dooris-widget: ccchh_dooris_desktop_widget_init()");

/*
	// initialize GnomeVFS library
	if (!gnome_vfs_init()) {
		// FIXME: crashes the whole desktop process
		g_error("ccchh-dooris-widget: could not initialize GnomeVFS library");
		abort();
	}
*/
 
	// default settings
	self->notify = TRUE;
	self->update = 23;
	// TODO: this should be moved to turing.hamburg.ccc.de
	self->url = gnome_vfs_make_uri_from_input("http://dooris.koalo.de/door.txt");
//	self->url = g_strdup("http://dooris.koalo.de/door.txt");
//	self->url = gnome_vfs_make_uri_from_input("http://formularfetischisten.de/~packbart/lustitsch/whale.mov");
//	self->url = g_strdup("http://formularfetischisten.de/~packbart/temp/door.txt");
//	self->url = gnome_vfs_make_uri_from_input("http://formularfetischisten.de/~packbart/temp/door.txt");
	self->use_connection = TRUE;
	self->force_connection = TRUE;

	// initialize state
	// this combination ensures that update() sets an image on first call
	// TODO: is there a better way?
	self->current_state = DOORIS_FNORD;
	self->last_state = -1;
	self->last_known_state = DOORIS_FNORD;

	self->last_change_ts = 0L;
	self->last_report_ts = 0L;

	// set image to NULL, first expose event will then call update()
	self->image_current = NULL;
	self->current_alpha = IMAGE_ALPHA_NORMAL;

	g_static_mutex_init(&(self->button_lock));
	g_static_mutex_init(&(self->fetch_lock));

	self->gconf_client = gconf_client_get_default();

	self->force_banner = FALSE;

	self->disconnecting = NULL;

	self->gvfs_handle = NULL;
	self->gvfs_input_buffer = NULL;

	// set widget size
	// TODO: is this the right way?
	gtk_widget_set_size_request (GTK_WIDGET (self), IMAGE_WIDGET_WIDTH, IMAGE_WIDGET_HEIGHT);
	gtk_window_resize (GTK_WINDOW (self), IMAGE_WIDGET_WIDTH, IMAGE_WIDGET_HEIGHT);

	/* Allow this widget to handle button-press events:
	 * Note that gtk_widget_add_events would only work after the widget is realized:
	 */
	gint mask = gtk_widget_get_events (GTK_WIDGET (self)) | GDK_BUTTON_PRESS_MASK |
		GDK_BUTTON_RELEASE_MASK | GDK_LEAVE_NOTIFY_MASK;
	gtk_widget_set_events (GTK_WIDGET (self), mask);
     
	/* Handle a mouse button press: */
	g_signal_connect (self, "button-press-event", 
		G_CALLBACK (ccchh_dooris_desktop_widget_on_button_press_event), NULL);

	g_signal_connect (self, "button-release-event", 
		G_CALLBACK (ccchh_dooris_desktop_widget_on_button_release_event), NULL);

	g_signal_connect (self, "leave-notify-event", 
		G_CALLBACK (ccchh_dooris_desktop_widget_on_button_leave_event), NULL);

	/* Specify that a settings button should be shown in layout mode, 
	 * and handle a request to configure the settings:
	 */
	hd_home_plugin_item_set_settings (HD_HOME_PLUGIN_ITEM (self), TRUE);
	g_signal_connect (self, "show-settings", 
		G_CALLBACK (ccchh_dooris_desktop_widget_on_show_settings), NULL);   

	if (gconf_client_dir_exists(self->gconf_client, GCONF_DIR, NULL)) {
		load_settings(self);
	}

	/* Create connection object */
	self->connection = con_ic_connection_new ();
	self->connection_handler = 0;

	if (self->use_connection) {
		// connection event will eventually
		// set self->connected and start timer
		install_connection_handler(self);
	} else {
		self->connected = TRUE;
		start_update_timer(self);
	}
	g_debug("ccchh-dooris-widget: initial URL: %s\n", self->url);
}


// this function is never called, is it?
CCCHHDoorIsDesktopWidget*
ccchh_dooris_desktop_widget_new (void)
{
	g_debug("ccchh-dooris-widget: ccchh_dooris_desktop_widget_new()");
	return g_object_new (TYPE_CCCHH_DOORIS_DESKTOP_WIDGET, NULL);
}
