/* ccchh_dooris_desktop_widget.c */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>

#include <glib.h>
#include <glib/gi18n-lib.h> /* Declares _(). */
#include <gtk/gtk.h>

#include <hildon/hildon.h>
#include <libhildondesktop/libhildondesktop.h>
#include <conic.h> /* libconic - Internet Connectivity library */

#include <libgnomevfs/gnome-vfs.h>

#include "ccchh-dooris-widget.h"

// max download size
#define MAX_HTTP_SIZE 1024

// minimum/maximum selectable update interval
// in minutes
#define UPDATE_INTERVAL_MIN	5
#define UPDATE_INTERVAL_MAX	1440

// paths to icon files
#define IMAGE_DOORIS_OPEN			DATADIR"/dooris_open.png"
#define IMAGE_DOORIS_CLOSED			DATADIR"/dooris_closed.png"
#define IMAGE_DOORIS_FNORD			DATADIR"/dooris_fnord.png"
#define IMAGE_OVERLAY_DISCONNECTED	DATADIR"/dooris_notavail.png"

// widget size
#define IMAGE_WIDGET_WIDTH	102
#define IMAGE_WIDGET_HEIGHT	72

// transparency
#define IMAGE_ALPHA_NORMAL		1.0
#define IMAGE_ALPHA_HIGHLIGHT	0.65

// #define _DEBUG
#ifdef _DEBUG
// redefine // g_debug() to g_message()
// TODO: find out how to get debug logs from hildon-desktop
#define // g_debug g_message
#define USER_AGENT_VERSION_STRING (debug version)
#else
#define USER_AGENT_VERSION_STRING (some version number)
#endif

/**
 * CCCHH DoorIs Fnord Desktop Widget
 *
 * Desktop widget that indicates the main entrance door lock state at the
 * CCC in Hamburg.
 *
 * A switch in the door frame at the CCC Hamburg registers locking / unlocking
 * events. The current state is exported to a webserver.
 *
 * This widget shows an icon on the desktop indicating whether the door is
 * open or locked and optionally notifies the user upon state changes.
 *
 *
 * It is a shared librarary that implements a derived HDHomePluginItem (a 
 * desktop widget), with the help of the HD_DEFINE_PLUGIN_MODULE() macro, which 
 * also implements some standard functions that all desktop widget libraries 
 * should export.
 *
 * Note that the API contains the name "home plugin", but the correct name 
 * is now "desktop widget".
 * 
 * As with other custom GTK+ widgets, drawing happens only in a handler for the 
 * expose-event signal. This is triggered by a timeout handler that invalidates
 * the desktop widget's area.
 *
 */

/** This is like G_DEFINE_TYPE(), but it also 
 * implements standard functions such as hd_plugin_module_load(), 
 * which hildon-desktop expects to be able to call on this library.
 */
HD_DEFINE_PLUGIN_MODULE (CCCHHDoorIsDesktopWidget, ccchh_dooris_desktop_widget,
  HD_TYPE_HOME_PLUGIN_ITEM)


// private functions, only called from within the widget

 
// Code to be reworked:

// TODO: is this really the right thing to do?
// code copied from the internets
// maybe use gtk-timers instead of "for" loop
// running the main loop from within events
// or constructors can lead to strange results

static void gtk_sleep (unsigned int ms)
{
  int i;
  fflush (stdout);
//  ms = ms / 22;
  if (ms == 0)
    ms ++;
  for (i=0; i<ms; i++)
  {
    usleep(1000);
    while (gtk_events_pending ())
      gtk_main_iteration ();
  }
}


// TODO: better implementation.
// maybe use glibcurl (http://atterer.net/glibcurl/)
// or GnomeVFS
static dooris_state_t
fetch_current_state (CCCHHDoorIsDesktopWidget *self)
{
	gulong ts;
	dooris_state_t state;
	char *line, *tokctx;

	GnomeVFSHandle *gvfs_handle;
	GnomeVFSResult gvfs_errno;
	GnomeVFSFileSize bytes_read;

	char buffer[MAX_HTTP_SIZE+1];
	memset(buffer, 0, sizeof(buffer));

	// g_debug("ccchh-dooris-widget: fetch_current_state()");
	// g_debug("ccchh-dooris-widget: fetching state from URL: %s", self->url);

// why bother? we check error on open
/*
GnomeVFSURI *uri_struct = gnome_vfs_uri_new(self->url);
if (!gnome_vfs_uri_exists(uri_struct)) {
	g_warning("ccchh-dooris-widget: URL does not exist");
	return DOORIS_FNORD;
}
gnome_vfs_uri_unref(uri_struct);
*/

	// g_debug("ccchh-dooris-widget: gnome_vfs_open()");
	gvfs_errno = gnome_vfs_open(&gvfs_handle, self->url, GNOME_VFS_OPEN_READ);

	if (gvfs_errno != GNOME_VFS_OK) {
		g_warning("ccchh-dooris-widget: gnome_vfs_open: %s (%d)\n", gnome_vfs_result_to_string(gvfs_errno), gvfs_errno);
		return DOORIS_FNORD;
	}

	// g_debug("ccchh-dooris-widget: gnome_vfs_read()");
	gvfs_errno = gnome_vfs_read(gvfs_handle, buffer, MAX_HTTP_SIZE, &bytes_read);

	if (gvfs_errno != GNOME_VFS_OK) {
		g_warning("ccchh-dooris-widget: gnome_vfs_read: %s (%d)\n", gnome_vfs_result_to_string(gvfs_errno), gvfs_errno);
		return DOORIS_FNORD;
	}

	// g_debug("ccchh-dooris-widget: read %lld bytes", bytes_read);
// rest of buffer already zeroed by memset()
//	buffer[bytes_read] = '\0';

	// g_debug("ccchh-dooris-widget: gnome_vfs_close()");
	gnome_vfs_close(gvfs_handle);
 
	self->last_change_ts = 0L;
	self->last_report_ts = 0L;
	tokctx = NULL;

	// first line of body is either "locked" or "unlocked"

	if (!(line = strtok_r(buffer, "\n", &tokctx))) {
		g_warning("ccchh-dooris-widget: first line of response not found");
		return DOORIS_FNORD;
	}

	if (!strcmp(line, "locked")) {
		state = DOORIS_CLOSED;
		// g_debug("ccchh-dooris-widget: current state: closed");
	} else if (!strcmp(line, "unlocked")) {
		state = DOORIS_OPEN;
		// g_debug("ccchh-dooris-widget: current state: open");
	}
	else {
		g_warning("ccchh-dooris-widget: could not parse response");
		return DOORIS_FNORD;
	}

	// second line of body is last sensor report timestamp

	if ((line = strtok_r(NULL, "\n", &tokctx))) {
		errno = 0;
		ts = strtoul(line, (char **) NULL, 10);
		if (ts && !errno) {
			self->last_report_ts = ts;
		}
		// g_debug("ccchh-dooris-widget: last_report_ts: %lu, errno: %i\n", ts, errno);

		// third line of body is last state change timestamp

		if ((line = strtok_r(NULL, "\n", &tokctx))) {
			errno = 0;
			ts = strtoul(line, (char **) NULL, 10);
			if (ts && !errno) {
				self->last_change_ts = ts;
			}
			// g_debug("ccchh-dooris-widget: last_change_ts: %lu, errno: %i\n", ts, errno);
		}
	}

 	return state;
}

// stop update timer
// returns TRUE if timer was stopped
// returns FALSE if no timer was running
static inline gboolean
stop_update_timer(CCCHHDoorIsDesktopWidget *self)
{
	// stop timer if it was running
	if (self->timeout_handler) {
		// g_debug("ccchh-dooris-widget: stopping update timer");
		g_source_remove (self->timeout_handler);
		self->timeout_handler = 0;
		return TRUE;
	}
	return FALSE;
}

// start update timer
// returns TRUE if timer was started
// returns FALSE if timer was already running
// TODO: check return code from g_timeout_add_seconds
static inline gboolean
start_update_timer(CCCHHDoorIsDesktopWidget *self)
{
	if (!self->timeout_handler) {
		// g_debug("ccchh-dooris-widget: starting update timer");
		if (self->update < UPDATE_INTERVAL_MIN) {
			g_warning("ccchh-dooris-widget: self->update < UPDATE_INTERVAL_MIN, maybe a bug?");
			self->update = UPDATE_INTERVAL_MIN;
		} else if (self->update > UPDATE_INTERVAL_MAX) {
			g_warning("ccchh-dooris-widget: self->update > UPDATE_INTERVAL_MAX, maybe a bug?");
			self->update = UPDATE_INTERVAL_MAX;
		}
		self->timeout_handler = g_timeout_add_seconds (self->update*60, ccchh_dooris_desktop_widget_on_timeout, self);
		return TRUE;
	}
	return FALSE;
}

// install libconic connection event handler
// returns TRUE if handler was installed
// returns FALSE if handler was already set
// TODO: check return code from g_signal_connect
static inline gboolean
install_connection_handler(CCCHHDoorIsDesktopWidget *self)
{
	if (!self->connection_handler) {
		// set connected to FALSE, initial event may set it to TRUE
		self->connected = FALSE;
		/* Connect signal to receive connection events */
		self->connection_handler = g_signal_connect (self->connection, "connection-event",
			G_CALLBACK (ccchh_dooris_desktop_widget_connection_event), self);
		/* Set automatic events */
		g_object_set (self->connection, "automatic-connection-events", TRUE, NULL);
		// g_debug("ccchh-dooris-widget: connection monitoring enabled");
		return TRUE;
	}

	return FALSE;
}

// remove libconic connection event handler
// returns TRUE if handler was removed
// returns FALSE if no handler was set
static inline gboolean
remove_connection_handler(CCCHHDoorIsDesktopWidget *self)
{
	if (self->connection_handler) {
		g_object_set (self->connection, "automatic-connection-events", FALSE, NULL);
		g_signal_handler_disconnect (self->connection, self->connection_handler);
		self->connection_handler = 0;
		// g_debug("ccchh-dooris-widget: connection monitoring disabled");
		return TRUE;
	}

	return FALSE;
}

// update internal state
// returns TRUE on significant state change
// (i.e. state is valid and has changed since last valid state)
// TODO: glib locking?
// this function should not be called simultaneously
// timer and button events could call update() at the same time
static gboolean
update (CCCHHDoorIsDesktopWidget *self)
{
	// g_debug("ccchh-dooris-widget: update()");

	if (!self->connected) {
		// g_debug("ccchh-dooris-widget: not connected, not fetching new state");
		return FALSE;
	}

	// retrieve state from server
	self->current_state = fetch_current_state(self);

	if (self->current_state == self->last_state) {
		// g_debug("ccchh-dooris-widget: state unchanged");
		return FALSE;
	}

	// release old icon surface
	if (self->image_current) {
		cairo_surface_destroy(self->image_current);
		self->image_current = NULL;
	}

	// set new icon surface
	switch (self->current_state) {
		case DOORIS_OPEN:
			self->image_current = cairo_image_surface_create_from_png(IMAGE_DOORIS_OPEN);
			// g_debug("ccchh-dooris-widget: image_current set to open");
			break;
		case DOORIS_CLOSED:
			self->image_current = cairo_image_surface_create_from_png(IMAGE_DOORIS_CLOSED);
			// g_debug("ccchh-dooris-widget: image_current set to closed");
			break;
		default:
			self->image_current = cairo_image_surface_create_from_png(IMAGE_DOORIS_FNORD);
			// g_debug("ccchh-dooris-widget: image_current set to fnord");
			break;
	}

	self->last_state = self->current_state;

	// state is valid and has changed since last known state
	if (self->current_state != DOORIS_FNORD && self->current_state != self->last_known_state) {
		self->last_known_state = self->current_state;
		// g_debug("ccchh-dooris-widget: state changed");
		return TRUE;
	}

	return FALSE;
}


static void
draw (CCCHHDoorIsDesktopWidget *self, cairo_t *cr)
{
//  GtkWidget *widget = GTK_WIDGET(self);
  
	// g_debug("ccchh-dooris-widget: draw()");

	if (!self->image_current) {
		// g_debug("ccchh-dooris-widget: draw() without image");
		self->image_current = cairo_image_surface_create_from_png(IMAGE_DOORIS_FNORD);
	}

	cairo_save(cr);

	// paint icon
	cairo_set_source_surface(cr, self->image_current, 5, 5);
	cairo_paint_with_alpha(cr, self->current_alpha);

	// if not connected, overlay icon with "disconnected" image
	if (!self->connected) {
		// g_debug("ccchh-dooris-widget: not connected, overlay image");
		cairo_surface_t *image_overlay = cairo_image_surface_create_from_png(IMAGE_OVERLAY_DISCONNECTED);
		cairo_set_source_surface(cr, image_overlay, 5, 5);
		cairo_paint_with_alpha(cr, self->current_alpha);
		cairo_surface_destroy(image_overlay);
	}

	cairo_restore(cr);
}


static void
show_status_banner (CCCHHDoorIsDesktopWidget *self)
{
	time_t timenow, timediff;
	gchar *text, *ts_text;

	// g_debug("ccchh-dooris-widget: show_status_banner()");

	time (&timenow);

	// format the time elapsed since last state change
	// TODO: is it correct and portable to cast time_t to ulong?

	if (timenow != -1 && self->last_change_ts) {
		timediff = timenow - self->last_change_ts;

		if (timediff < 60) {
			ts_text = g_strdup_printf(_("(%lds ago)"), (ulong)timediff);
		} else if (timediff < 3600) {
			ts_text = g_strdup_printf(_("(%ldm %lds ago)"), (ulong)timediff / 60, (ulong)timediff % 60);
		} else {
			ts_text = g_strdup_printf(_("(%ldh %ldm %lds ago)"), (ulong)timediff / 3600, (ulong)(timediff % 3600) / 60, (ulong)timediff % 60);
		}
	} else {
		ts_text = g_strdup("");
	}

	// assemble string to display
	switch (self->current_state) {
		case DOORIS_OPEN:
			text = g_strdup_printf(("%s %s %s"),
				_("CCCHH: Door <b>unlocked</b>"), ts_text,
				(self->connected) ? ("") : _("(offline)"));
				break;
//				(self->connected) ? _("(online)") : _("(offline)"));
		case DOORIS_CLOSED:
			text = g_strdup_printf(("%s %s %s"),
				_("CCCHH: Door locked"), ts_text,
				(self->connected) ? ("") : _("(offline)"));
				break;
		default:
			text = g_strdup_printf(("%s %s"),
				_("CCCHH: Unknown"),
				(self->connected) ? ("") : _("(offline)"));
				break;
	}

	// display banner on screen
	hildon_banner_show_information_with_markup (GTK_WIDGET (self),
		NULL, /* icon_name - ignored */
		text);

	g_free (text);
	g_free (ts_text);
}


static void
force_redraw (CCCHHDoorIsDesktopWidget *self)
{
	// g_debug("ccchh-dooris-widget: force_redraw()");

	/* Invalidate the drawing: */
	GtkWidget *widget = GTK_WIDGET (self);

	if (!widget->window)
		return;

	/* Redraw the cairo canvas completely by exposing it: */
	GdkRegion *region = gdk_drawable_get_clip_region (widget->window);
	if (!region)
		return;
	gdk_window_invalidate_region (widget->window, region, TRUE);

	// process expose event
	gdk_window_process_updates (widget->window, TRUE);

	gdk_region_destroy (region);
}


// end private functions


static gboolean
ccchh_dooris_desktop_widget_on_timeout (gpointer data)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (data);
  
	// g_debug("ccchh-dooris-widget: ccchh_dooris_desktop_widget_on_timeout()");

// TODO: glib locking?
// update() must be called only once at a time
if (self->updating) {
	// g_debug("ccchh-dooris-widget: update lock set, not updating");
	return TRUE;
}
self->updating = TRUE;

	// stop event when offline
	// connection event starts the timer again
	if (!self->connected) {
self->updating = FALSE;
		return FALSE;
	}

	// update internal state
	if (!update(self)) {
self->updating = FALSE;
		// nothing significant to report, keep the timer running
		// TODO: this could miss to redraw a change to/from DOORIS_FNORD
		return TRUE;
	}

	// state changed: redraw icon
	force_redraw(self);

	// flash banner if notifications are enabled
	if (self->notify)
		show_status_banner(self);

self->updating = FALSE;

	return TRUE; /* keep running this event */
}


static gboolean
ccchh_dooris_desktop_widget_expose_event (GtkWidget *widget, GdkEventExpose *event)
{    
	cairo_t *cr  = gdk_cairo_create (widget->window);
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (widget);

	// g_debug("ccchh-dooris-widget: ccchh_dooris_desktop_widget_expose_event()");

	if (!self->image_current) {
		// g_debug("ccchh-dooris-widget: expose event without image");
		// no state yet, call update()
		// flash banner if necessary and enabled
		if (update(self) && self->notify) {
			show_status_banner(self);
		}
	}

	/* Clip only the exposed area of the cairo context,  
	 * to potentially avoid unnecessary drawing:
	 */
	cairo_rectangle (cr,
		event->area.x, event->area.y,
		event->area.width, event->area.height);
	cairo_clip (cr);

	draw (CCCHH_DOORIS_DESKTOP_WIDGET (widget), cr);

	cairo_destroy (cr);
	return FALSE;
}


// handle the button_press event
// TODO: highlight on button_press, call update() on button_release
static gboolean 
ccchh_dooris_desktop_widget_on_button_press_event (GtkWidget *widget, 
  GdkEventButton * event G_GNUC_UNUSED, gpointer user_data G_GNUC_UNUSED)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (widget);

	// g_debug("ccchh-dooris-widget: ccchh_dooris_desktop_widget_on_button_press_event()");
 
// TODO: glib locking?
// update() must be called only once at a time
if (self->updating) {
	// g_debug("ccchh-dooris-widget: update lock set, not updating");
	return TRUE;
}

self->updating = TRUE;

	// acknowledge button, make icon transparent
	// TODO: not really highlighting the button
	// also it doesn't always redraw in time
	self->current_alpha = IMAGE_ALPHA_HIGHLIGHT;
	force_redraw(self);

	// request connection from libconic, if offline and
	// force_connection is enabled
	// TODO: this does not work immediately, the call to update()
	// will still see !connected. maybe wait a bit or use a timer event?
	if (!self->connected && self->force_connection) {
		// g_debug("ccchh-dooris-widget: not connected, requesting connection");
		/* Request connection and check for the result */
		gboolean success = con_ic_connection_connect(self->connection,
			CON_IC_CONNECT_FLAG_NONE);
		if (!success) 
			g_warning("ccchh-dooris-widget: request for connection failed");
	}

	// TODO: think again
	// we want the button to stay transparent for a few ms,
	// even if update() returns fast
	// also, running the main loop from within a button event
	// could call the function again. re-ent-what?
	gtk_sleep(100);

	// update internal state, redraw on change
	// TODO: this could miss to redraw a change to/from DOORIS_FNORD
	// although the next force_redraw() catches it
	if (update(self)) {
		force_redraw(self);
	}

	// always display the banner when the button is pressed
	show_status_banner(self);

	// redraw the icon opaque
	self->current_alpha = IMAGE_ALPHA_NORMAL;
	force_redraw(self);

	// TODO: think again, see above
	gtk_sleep(100);

	// if a timer was running, reset it
	if (stop_update_timer(self)) {
		start_update_timer(self);
	}

self->updating = FALSE;

	return TRUE; /* Prevent further handling of this signal. */
}


static void
ccchh_dooris_desktop_widget_connection_event(ConIcConnection *connection,
  ConIcConnectionEvent *event, gpointer user_data)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (user_data);

	// g_debug("ccchh-dooris-widget: ccchh_dooris_desktop_widget_connection_event()");

	// libconic signalled a connection state change

	ConIcConnectionStatus status = con_ic_connection_event_get_status(event);

	if (status == CON_IC_STATUS_CONNECTED) {

		// we are online now
		self->connected = TRUE;
		// g_debug("ccchh-dooris-widget: new connection state: online");

		// start timer
		start_update_timer(self);

		// force redraw to remove disconnection overlay
		force_redraw(self);
	} else if (status == CON_IC_STATUS_DISCONNECTED) {

		// we are offline now
		self->connected = FALSE;
		// g_debug("ccchh-dooris-widget: new connection state: offline");

//const gchar *iap_id = con_ic_event_get_iap_id(CON_IC_EVENT(event));
//const gchar *bearer = con_ic_event_get_bearer_type(CON_IC_EVENT(event));
//gchar *text = g_strdup_printf("
// // g_debug("Hey, we are connected to IAP %s with bearer %s!", iap_id, bearer);
// error = con_ic_connection_event_get_error(event);
//hildon_banner_show_information_with_markup (GTK_WIDGET (self),
//	NULL, /* icon_name - ignored */
//	text);
//g_free (text);

		// don't stop timer here
		// it will deactivate itsef if !connected
		// frequent stop/start would prevent it from firing
//		// stop timer
//		stop_update_timer(self);

		// force redraw to add disconnection overlay
		force_redraw(self);
	}

	// don't do anything on CON_IC_STATUS_DISCONNECTING
	// or CON_IC_STATUS_NETWORK_UP
}


// TODO: save settings

static void 
ccchh_dooris_desktop_widget_on_settings_dialog_response (GtkDialog *dialog,
  gint response_id, gpointer user_data)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (user_data);
	g_assert (self->settings_window);
	g_assert (self->settings_window == GTK_WIDGET (dialog));
  
	// g_debug("ccchh-dooris-widget: ccchh_dooris_desktop_widget_on_settings_dialog_response()");

	// only use new settings if user pressed the OK button
	// TODO: add reset button to restore defaults
	if (response_id == GTK_RESPONSE_ACCEPT) {
		// g_debug("ccchh-dooris-widget: response accept");

		// status URL
		if (self->url) {
			// is there any difference between free() and g_free()?
//			g_free(self->url);
			free(self->url);
			self->url = NULL;
		}
//		self->url = g_strdup(gtk_entry_get_text(GTK_ENTRY (self->url_entry)));
		// try to build a valid URI from user input
		self->url = gnome_vfs_make_uri_from_input(gtk_entry_get_text(GTK_ENTRY (self->url_entry)));
		// g_debug("ccchh-dooris-widget: new URL: %s\n", self->url);


		// update interval
		const gchar *choice = hildon_button_get_value(HILDON_BUTTON (self->update_picker));
		self->update = CLAMP(g_ascii_strtod (choice, NULL), UPDATE_INTERVAL_MIN, UPDATE_INTERVAL_MAX);
		// g_debug("ccchh-dooris-widget: new update interval: %ld\n", (long)self->update);

		// if a timer was running, reset it
		if (stop_update_timer(self)) {
			start_update_timer(self);
		}

		// banner notifications
		self->notify = hildon_check_button_get_active(HILDON_CHECK_BUTTON(self->notify_button));
		if (self->notify) {
			// g_debug("ccchh-dooris-widget: notification enabled");
		} else {
			// g_debug("ccchh-dooris-widget: notification disabled");
		}

		// monitor internet connection state, only update if online
		self->use_connection = hildon_check_button_get_active(HILDON_CHECK_BUTTON(self->connection_button));

		if (self->use_connection) {
			install_connection_handler(self);
		} else {
			remove_connection_handler(self);

			start_update_timer(self);

			// force connection state to online
			// redraw if necessary to remove disconnection overlay
			if (!self->connected) {
				self->connected = TRUE;
				force_redraw(self);
			}
		}

		// button forces internet connection
		self->force_connection = hildon_check_button_get_active(HILDON_CHECK_BUTTON(self->connforce_button));
		if (self->force_connection) {
			// g_debug("ccchh-dooris-widget: button forces connection");
		} else {
			// g_debug("ccchh-dooris-widget: button does not force connection");
		}
	}

	gtk_widget_destroy (self->settings_window);
	// TODO: isn't there a better way to this? array+memset?
	self->settings_window = NULL;
	self->url_entry = NULL;
	self->update_picker = NULL;
	self->notify_button = NULL;
	self->connection_button = NULL;
	self->connforce_button = NULL;
}


// update interval picker callback - restrict value to UPDATE_INTERVAL_MIN - UPDATE_INTERVAL_MAX
void update_picker_changed_value_cb (HildonPickerButton *picker, gpointer data)
{
	gdouble number = 0;
	const gchar *choice = hildon_button_get_value(HILDON_BUTTON (picker));
	number = CLAMP(g_ascii_strtod (choice, NULL), UPDATE_INTERVAL_MIN, UPDATE_INTERVAL_MAX);
	gchar *value_text = g_strdup_printf ("%d", (int) number);
	hildon_button_set_value(HILDON_BUTTON (picker), value_text);
	g_free(value_text);
}


static void 
ccchh_dooris_desktop_widget_on_show_settings (GtkWidget *widget G_GNUC_UNUSED, 
  gpointer user_data G_GNUC_UNUSED)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (widget);
 
	// g_debug("ccchh-dooris-widget: ccchh_dooris_desktop_widget_on_show_settings()");

	// could this be called when the settings window is already open?
	if (!self->settings_window) {
		/* The Hildon HIG says "Dialogs should be used for small tasks" so 
		 * we use GtkDialog instead of HildonWindow: */
		self->settings_window = gtk_dialog_new_with_buttons(
			_("CCCHH DoorIs Widget Settings"), 
			NULL /* parent window */, 
			GTK_DIALOG_NO_SEPARATOR /* flags */, 
			GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
			NULL);
		// call dialog_response() on close
		g_signal_connect (self->settings_window, "response", 
			G_CALLBACK (ccchh_dooris_desktop_widget_on_settings_dialog_response), 
			self);

		// arrange the inputs
		// we put controls and labels into hbox-es
		// and stack them into the dialog's vbox

		GtkWidget *vbox = GTK_DIALOG(self->settings_window)->vbox;
		gtk_container_set_border_width(GTK_CONTAINER (vbox), HILDON_MARGIN_DEFAULT);

		GtkWidget *hbox = gtk_hbox_new (FALSE, HILDON_MARGIN_DEFAULT);
		gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

		// TODO: use HildonCaption

		// status url
		GtkWidget *label = gtk_label_new (_("Status URL:"));
		gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
		self->url_entry = hildon_entry_new (HILDON_SIZE_FINGER_HEIGHT);
		gtk_box_pack_start (GTK_BOX (hbox), self->url_entry, TRUE, TRUE, 0);

		hbox = gtk_hbox_new (FALSE, HILDON_MARGIN_DEFAULT);
		gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

		// predefined choices for update interval selector
		HildonTouchSelector *selector = HILDON_TOUCH_SELECTOR (hildon_touch_selector_entry_new_text());
		hildon_touch_selector_append_text (selector, "5");
		hildon_touch_selector_append_text (selector, "10");
		hildon_touch_selector_append_text (selector, "15");
		hildon_touch_selector_append_text (selector, "20");
		hildon_touch_selector_append_text (selector, "25");
		hildon_touch_selector_append_text (selector, "30");
		hildon_touch_selector_append_text (selector, "45");
		hildon_touch_selector_append_text (selector, "60");

		// update interval
		self->update_picker = hildon_picker_button_new (HILDON_SIZE_FINGER_HEIGHT,
			HILDON_BUTTON_ARRANGEMENT_VERTICAL);
		hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (self->update_picker), selector);
		// install callback function to validate input
		g_signal_connect (G_OBJECT (self->update_picker), "value-changed",
			G_CALLBACK (update_picker_changed_value_cb), NULL);
		// TODO: this doesn't look good, really use HildonCaption
		hildon_button_set_title(HILDON_BUTTON (self->update_picker), _("Update interval (minutes):"));
		// align left, center vertically
		gtk_button_set_alignment(GTK_BUTTON(self->update_picker), 0, 0.5);
		gtk_box_pack_start (GTK_BOX (hbox), self->update_picker, TRUE, TRUE, 0);

		hbox = gtk_hbox_new (FALSE, HILDON_MARGIN_DEFAULT);
		gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
    
		// banner notifications
		self->notify_button = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
		// TODO: this doesn't look good, really use HildonCaption
		gtk_button_set_label(GTK_BUTTON (self->notify_button), _("Notify on state change"));
		// align left, center vertically
		gtk_button_set_alignment(GTK_BUTTON(self->notify_button), 0, 0.5);
		gtk_box_pack_start (GTK_BOX (hbox), self->notify_button, TRUE, TRUE, 0);

		hbox = gtk_hbox_new (FALSE, HILDON_MARGIN_DEFAULT);
		gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

		// monitor internet connection, only update when online
		self->connection_button = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
		gtk_button_set_label(GTK_BUTTON (self->connection_button), _("Only update when online"));
		gtk_button_set_alignment(GTK_BUTTON(self->connection_button), 0, 0.5);
		gtk_box_pack_start (GTK_BOX (hbox), self->connection_button, TRUE, TRUE, 0);

		hbox = gtk_hbox_new (FALSE, HILDON_MARGIN_DEFAULT);
		gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

		// button forces connection
		self->connforce_button = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
		gtk_button_set_label(GTK_BUTTON (self->connforce_button), _("Button requests internet connection"));
		gtk_button_set_alignment(GTK_BUTTON(self->connforce_button), 0, 0.5);
		gtk_box_pack_start (GTK_BOX (hbox), self->connforce_button, TRUE, TRUE, 0);
	}

	// set current values
	gtk_entry_set_text(GTK_ENTRY(self->url_entry), self->url);

	// no default selection
	// TODO: find the selector nearest to the current value and activate it
	hildon_picker_button_set_active (HILDON_PICKER_BUTTON (self->update_picker), -1);
	char *value_text = g_strdup_printf ("%d", self->update);
	hildon_button_set_value(HILDON_BUTTON (self->update_picker), value_text);
	g_free(value_text);

	hildon_check_button_set_active(HILDON_CHECK_BUTTON(self->notify_button), self->notify);
	hildon_check_button_set_active(HILDON_CHECK_BUTTON(self->connection_button), self->use_connection);
	hildon_check_button_set_active(HILDON_CHECK_BUTTON(self->connforce_button), self->force_connection);

	gtk_widget_show_all (self->settings_window);
}


static void
ccchh_dooris_desktop_widget_realize (GtkWidget *widget)
{
	// g_debug("ccchh-dooris-widget: ccchh_dooris_desktop_widget_realize()");

	/* Use An RGBA colormap rather than RGB, 
	 * so we can use transparency in our expose_event() implementation.
	 */
	GdkScreen *screen = gtk_widget_get_screen (widget);
	gtk_widget_set_colormap (widget,
		gdk_screen_get_rgba_colormap (screen));

	gtk_widget_set_app_paintable (widget, TRUE);

	/* Call the base class's implementation: */
	GTK_WIDGET_CLASS (ccchh_dooris_desktop_widget_parent_class)->realize (widget);
}


// TODO: this function gets called twice. why?
static void
ccchh_dooris_desktop_widget_dispose (GObject *object)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (object);
 
	// g_debug("ccchh-dooris-widget: ccchh_dooris_desktop_widget_dispose()");

	// remove settings window if it is still open
	if (self->settings_window) {
		gtk_widget_destroy (self->settings_window);
		self->settings_window = NULL;
		self->url_entry = NULL;
	}
   
	/* Call the base class's implementation: */
	G_OBJECT_CLASS (ccchh_dooris_desktop_widget_parent_class)->dispose (object);
}


static void
ccchh_dooris_desktop_widget_finalize (GObject *object)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (object);

	// g_debug("ccchh-dooris-widget: ccchh_dooris_desktop_widget_finalize()");
 
	stop_update_timer(self);
  
	remove_connection_handler(self);

	// destroy icon surface if we still hold one
	if (self->image_current && cairo_surface_get_reference_count(self->image_current) > 0) {
		cairo_surface_destroy(self->image_current);
	}

	// shut down gnome vfs library
	// TODO: is this wise? what if other widgets use it?
//	gnome_vfs_shutdown();
 
	if (self->url) {
//		g_free(self->url);
		free(self->url);
		self->url = NULL;
	}

	/* Call the base class's implementation: */
	G_OBJECT_CLASS (ccchh_dooris_desktop_widget_parent_class)->finalize (object);
}


static void
ccchh_dooris_desktop_widget_class_init (CCCHHDoorIsDesktopWidgetClass *klass)
{

	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);

	// g_debug("ccchh-dooris-widget: ccchh_dooris_desktop_widget_class_init()");

	object_class->dispose = ccchh_dooris_desktop_widget_dispose;
	object_class->finalize = ccchh_dooris_desktop_widget_finalize;
 
	widget_class->realize = ccchh_dooris_desktop_widget_realize;
	widget_class->expose_event = ccchh_dooris_desktop_widget_expose_event;
}


static void
ccchh_dooris_desktop_widget_class_finalize (CCCHHDoorIsDesktopWidgetClass *klass G_GNUC_UNUSED)
{
	// g_debug("ccchh-dooris-widget: ccchh_dooris_desktop_widget_class_finalize()");
}


static void
ccchh_dooris_desktop_widget_init (CCCHHDoorIsDesktopWidget *self)
{
	// g_debug("ccchh-dooris-widget: ccchh_dooris_desktop_widget_init()");

	// initialize GnomeVFS library
	if (!gnome_vfs_init()) {
		// FIXME: crashes the whole desktop process
		g_error("ccchh-dooris-widget: could not initialize GnomeVFS library");
		abort();
	}
 
	// default settings
	self->notify = TRUE;
	self->update = 23;
	// TODO: this should be moved to turing.hamburg.ccc.de
	self->url = gnome_vfs_make_uri_from_input("http://dooris.koalo.de/door.txt");
//	self->url = g_strdup("http://dooris.koalo.de/door.txt");
//	self->url = gnome_vfs_make_uri_from_input("http://formularfetischisten.de/~packbart/lustitsch/whale.mov");
//	self->url = g_strdup("http://formularfetischisten.de/~packbart/temp/door.txt");
//	self->url = gnome_vfs_make_uri_from_input("http://formularfetischisten.de/~packbart/temp/door.txt");
	self->use_connection = TRUE;
	self->force_connection = TRUE;

	// initialize state
	// this combination ensures that update() sets an image on first call
	// TODO: is there a better way?
	self->current_state = DOORIS_FNORD;
	self->last_state = -1;
	self->last_known_state = DOORIS_FNORD;

	self->last_change_ts = 0L;
	self->last_report_ts = 0L;

	// set image to NULL, first expose event will then call update()
	self->image_current = NULL;
	self->current_alpha = IMAGE_ALPHA_NORMAL;

	self->updating = FALSE;
	self->force_banner = FALSE;

	// set widget size
	// TODO: is this the right way?
	gtk_widget_set_size_request (GTK_WIDGET (self), IMAGE_WIDGET_WIDTH, IMAGE_WIDGET_HEIGHT);
	gtk_window_resize (GTK_WINDOW (self), IMAGE_WIDGET_WIDTH, IMAGE_WIDGET_HEIGHT);

	/* Allow this widget to handle button-press events:
	 * Note that gtk_widget_add_events would only work after the widget is realized:
	 */
	gint mask = gtk_widget_get_events (GTK_WIDGET (self)) | GDK_BUTTON_PRESS_MASK;
	gtk_widget_set_events (GTK_WIDGET (self), mask);
     
	/* Handle a mouse button press: */
	g_signal_connect (self, "button-press-event", 
		G_CALLBACK (ccchh_dooris_desktop_widget_on_button_press_event), NULL);

	/* Specify that a settings button should be shown in layout mode, 
	 * and handle a request to configure the settings:
	 */
	hd_home_plugin_item_set_settings (HD_HOME_PLUGIN_ITEM (self), TRUE);
	g_signal_connect (self, "show-settings", 
		G_CALLBACK (ccchh_dooris_desktop_widget_on_show_settings), NULL);   

	/* Create connection object */
	self->connection = con_ic_connection_new ();
	self->connection_handler = 0;

	if (self->use_connection) {
		// connection event will eventually
		// set self->connected and start timer
		install_connection_handler(self);
	} else {
		self->connected = TRUE;
		start_update_timer(self);
	}
	// g_debug("ccchh-dooris-widget: initial URL: %s\n", self->url);
}


// this function is never called, is it?
CCCHHDoorIsDesktopWidget*
ccchh_dooris_desktop_widget_new (void)
{
	// g_debug("ccchh-dooris-widget: ccchh_dooris_desktop_widget_new()");
	return g_object_new (TYPE_CCCHH_DOORIS_DESKTOP_WIDGET, NULL);
}
