/* ccchh_dooris_desktop_widget.c */

/**
 * CCCHH DoorIs Fnord Desktop Widget
 *
 * Desktop widget that indicates the main entrance door lock state at the
 * CCC in Hamburg.
 *
 * A switch in the door frame at the CCC Hamburg registers locking / unlocking
 * events. The current state is exported to a webserver.
 *
 * This widget shows an icon on the desktop indicating whether the door is
 * open or locked and optionally notifies the user upon state changes.
 *
 *
 * Copyright (c) 2010 Hauke Lampe
 * This code is licensed under a MIT-style License, see file COPYING
 *
 **/

/**
  TODO:

   * maybe important, requires deeper code changes:
   - download another file containing the DHCP/WLAN client counters
     - better get them merged into the same file
     - download/processing function evolved from synchronous
       call, not suited for multiple URLs
     - use pango to draw numbers over icon

   * maybe good, requires reading more docs:
   - check settings dialog for object/memory leaks
     - which parts are freed automatically, which are not?
     - is g_unref() the right tool?
   - check return code from g_timeout_add_seconds/g_signal_connect
     - for now, we wouldn't know if start_timer()/install_handler() failed
     - what to do in case of error?
   - does g_source_remove destroy the source or does it leak?
     - what data structures does a source use?

   * could be useful, requires more thinking:
   - aggregate calls to force_redraw()
     - for now, we call force_redraw() on any possible change
   - skip draw if not on active desktop
     - there's an event for that
     - don't skip update or banner
   - move constants from .h to configuration settings or autoconf options

   * style issues, don't affect features:
   - use more HildonCaption in settings dialog?
     - checkbox buttons look fine without Caption
     - PickerButton could use a larger font

   * nice to have:
   - draw nice icons
     - winni already did some work on new icons
   - support multiple instances
     - to support future extensions
     - not much use now

 **/




#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>

#include <glib.h>
#include <glib/gi18n-lib.h> /* Declares _() alias for gettext() */

#include <gtk/gtk.h>

#include <gconf/gconf-client.h>

#include <hildon/hildon.h>
#include <libhildondesktop/libhildondesktop.h>

#include <conic.h> /* libconic - Internet Connectivity library */

#include <libgnomevfs/gnome-vfs.h>

#include "ccchh-dooris-widget.h"


/** This is like G_DEFINE_TYPE(), but it also 
 * implements standard functions such as hd_plugin_module_load(), 
 * which hildon-desktop expects to be able to call on this library.
 */
HD_DEFINE_PLUGIN_MODULE (CCCHHDoorIsDesktopWidget, ccchh_dooris_desktop_widget,
  HD_TYPE_HOME_PLUGIN_ITEM)




// #define DEBUG

#ifdef DEBUG

// redefine g_debug() to g_message()
// TODO: find out how to get debug logs from hildon-desktop
#define g_debug g_message

#else

// check g_debug statements for syntax errors but don't emit code
// (compiler should optimize it away)
#define g_debug(fmt...) do { if (0) { g_message(fmt); } } while (0)

#endif // DEBUG




// start update timer
// returns TRUE if timer was started
// returns FALSE if timer was already running or auto_update disabled
// TODO: check return code from g_timeout_add_seconds
static inline gboolean
start_update_timer(CCCHHDoorIsDesktopWidget *self)
{
	g_assert(IS_CCCHH_DOORIS_DESKTOP_WIDGET(self));

	if (!self->auto_update)
		return FALSE;

	if (!self->timeout_handler) {
		g_debug(WIDGET_NAME": starting update timer");
		if (self->update_interval < UPDATE_INTERVAL_MIN) {
			g_warning(WIDGET_NAME": self->update_interval < UPDATE_INTERVAL_MIN, maybe a bug?");
			self->update_interval = UPDATE_INTERVAL_MIN;
		} else if (self->update_interval > UPDATE_INTERVAL_MAX) {
			g_warning(WIDGET_NAME": self->update_interval > UPDATE_INTERVAL_MAX, maybe a bug?");
			self->update_interval = UPDATE_INTERVAL_MAX;
		}
		self->timeout_handler = g_timeout_add_seconds(self->update_interval*60, ccchh_dooris_desktop_widget_on_timeout, self);
		return TRUE;
	}
	return FALSE;
}


// stop update timer
// returns TRUE if timer was stopped
// returns FALSE if no timer was running
static inline gboolean
stop_update_timer(CCCHHDoorIsDesktopWidget *self)
{
	g_assert(IS_CCCHH_DOORIS_DESKTOP_WIDGET(self));

	// stop timer if it was running
	if (self->timeout_handler) {
		g_debug(WIDGET_NAME": stopping update timer");
		g_source_remove (self->timeout_handler);
		self->timeout_handler = 0;
		return TRUE;
	}
	return FALSE;
}


// install libconic connection event handler
// returns TRUE if handler was installed
// returns FALSE if handler was already set
// TODO: check return code from g_signal_connect
static inline gboolean
install_connection_handler(CCCHHDoorIsDesktopWidget *self)
{
	g_assert(IS_CCCHH_DOORIS_DESKTOP_WIDGET(self));

	if (!self->connection_handler) {
		// set connected to FALSE, initial event may set it to TRUE
		self->connected = FALSE;
		// Connect signal to receive connection events
		self->connection_handler = g_signal_connect(self->connection, "connection-event",
			G_CALLBACK (ccchh_dooris_desktop_widget_connection_event), self);
		// Enable automatic events
		g_object_set(self->connection, "automatic-connection-events", TRUE, NULL);
		g_debug(WIDGET_NAME": connection monitoring enabled");
		return TRUE;
	}

	return FALSE;
}


// remove libconic connection event handler
// returns TRUE if handler was removed
// returns FALSE if no handler was set
static inline gboolean
remove_connection_handler(CCCHHDoorIsDesktopWidget *self)
{
	g_assert(IS_CCCHH_DOORIS_DESKTOP_WIDGET(self));

	if (self->connection_handler) {
		g_object_set(self->connection, "automatic-connection-events", FALSE, NULL);
		g_signal_handler_disconnect(self->connection, self->connection_handler);
		self->connection_handler = 0;
		g_debug(WIDGET_NAME": connection monitoring disabled");
		return TRUE;
	}

	return FALSE;
}




// load settings from gconf
// TODO: use gconf_client_get_without_default()?
//       initial load could return FALSE for booleans
//       if keys are not set yet
static void
load_settings(CCCHHDoorIsDesktopWidget *self)
{
	g_assert(IS_CCCHH_DOORIS_DESKTOP_WIDGET(self));

	GError *error;

	error = NULL;

	g_debug(WIDGET_NAME": load_settings()");

	gchar *strval = gconf_client_get_string(self->gconf_client,
		GCONF_DIR"/update_url", &error);

	if (error) {
		g_warning(WIDGET_NAME": failed to get configuration value update_url: %s",
			error->message);
		g_error_free(error);
		error = NULL;
	} else if (strval) {
		if (self->update_url) {
			free(self->update_url);
			self->update_url = NULL;
		}

		self->update_url = gnome_vfs_make_uri_from_input(strval);
		g_debug(WIDGET_NAME": load_settings: url = %s", self->update_url);

		g_free(strval);
		strval = NULL;
	}

	gint intval = gconf_client_get_int(self->gconf_client,
		GCONF_DIR"/update_interval", &error);

	if (error) {
		g_warning(WIDGET_NAME": failed to get configuration value update_interval: %s",
			error->message);
		g_error_free(error);
		error = NULL;
	} else if (intval) {
		self->update_interval = intval;
		if (self->update_interval < UPDATE_INTERVAL_MIN) {
			g_warning(WIDGET_NAME": self->update_interval < UPDATE_INTERVAL_MIN, maybe a bug?");
			self->update_interval = UPDATE_INTERVAL_MIN;
		} else if (self->update_interval > UPDATE_INTERVAL_MAX) {
			g_warning(WIDGET_NAME": self->update_interval > UPDATE_INTERVAL_MAX, maybe a bug?");
			self->update_interval = UPDATE_INTERVAL_MAX;
		}
		g_debug(WIDGET_NAME": load_settings: update = %ld", (long)self->update_interval);
	}

	gboolean boolval = gconf_client_get_bool(self->gconf_client,
		GCONF_DIR"/notify_enabled", &error);

	if (error) {
		g_warning(WIDGET_NAME": failed to get configuration value notify_enabled: %s",
			error->message);
		g_error_free(error);
		error = NULL;
	} else {
		self->notify_enabled = boolval;
		g_debug(WIDGET_NAME": load_settings: notify = %d", self->notify_enabled);
	}


	boolval = gconf_client_get_bool(self->gconf_client,
		GCONF_DIR"/use_connection", &error);

	if (error) {
		g_warning(WIDGET_NAME": failed to get configuration value use_connection: %s",
			error->message);
		g_error_free(error);
		error = NULL;
	} else {
		self->use_connection = boolval;
		g_debug(WIDGET_NAME": load_settings: use_connection = %d", self->use_connection);
	}

	boolval = gconf_client_get_bool(self->gconf_client,
		GCONF_DIR"/force_connection", &error);

	if (error) {
		g_warning(WIDGET_NAME": failed to get configuration value force_connection: %s",
			error->message);
		g_error_free(error);
		error = NULL;
	} else {
		self->force_connection = boolval;
		g_debug(WIDGET_NAME": load_settings: force_connection = %d", self->force_connection);
	}

	boolval = gconf_client_get_bool(self->gconf_client,
		GCONF_DIR"/conn_wlan_only", &error);

	if (error) {
		g_warning(WIDGET_NAME": failed to get configuration value conn_wlan_only: %s",
			error->message);
		g_error_free(error);
		error = NULL;
	} else {
		self->conn_wlan_only = boolval;
		g_debug(WIDGET_NAME": load_settings: conn_wlan_only = %d", self->conn_wlan_only);
	}

	// auto_update: new since 0.1.10
	// client_get_bool would return FALSE if key unset but default configuration is TRUE
	GConfValue *confval = gconf_client_get_without_default(self->gconf_client,
		GCONF_DIR"/auto_update", &error);

	if (error) {
		g_warning(WIDGET_NAME": failed to get configuration value auto_update: %s",
			error->message);
		g_error_free(error);
		error = NULL;
	} else if (confval) {
		self->auto_update = gconf_value_get_bool(confval);
		g_debug(WIDGET_NAME": load_settings: auto_update = %d", self->auto_update);
		gconf_value_free(confval);
	}
}


// save settings into gconf
static void
save_settings(CCCHHDoorIsDesktopWidget *self)
{
	g_assert(IS_CCCHH_DOORIS_DESKTOP_WIDGET(self));

	GError *error;

	error = NULL;

	g_debug(WIDGET_NAME": save_settings()");

	if (!gconf_client_set_string(self->gconf_client,
		GCONF_DIR"/update_url", self->update_url, &error))
	{
		g_warning(WIDGET_NAME": failed to set configuration value update_url: %s",
			error->message);
		g_error_free(error);
		error = NULL;
	}

	if (!gconf_client_set_int(self->gconf_client,
		GCONF_DIR"/update_interval", self->update_interval, &error))
	{
		g_warning(WIDGET_NAME": failed to set configuration value update_interval: %s",
			error->message);
		g_error_free(error);
		error = NULL;
	}

	if (!gconf_client_set_bool(self->gconf_client,
		GCONF_DIR"/notify_enabled", self->notify_enabled, &error))
	{
		g_warning(WIDGET_NAME": failed to set configuration value notify_enabled: %s",
			error->message);
		g_error_free(error);
		error = NULL;
	}

	if (!gconf_client_set_bool(self->gconf_client,
		GCONF_DIR"/use_connection", self->use_connection, &error))
	{
		g_warning(WIDGET_NAME": failed to set configuration value use_connection: %s",
			error->message);
		g_error_free(error);
		error = NULL;
	}

	if (!gconf_client_set_bool(self->gconf_client,
		GCONF_DIR"/force_connection", self->force_connection, &error))
	{
		g_warning(WIDGET_NAME": failed to set configuration value force_connection: %s",
			error->message);
		g_error_free(error);
		error = NULL;
	}

	if (!gconf_client_set_bool(self->gconf_client,
		GCONF_DIR"/conn_wlan_only", self->conn_wlan_only, &error))
	{
		g_warning(WIDGET_NAME": failed to set configuration value conn_wlan_only: %s",
			error->message);
		g_error_free(error);
		error = NULL;
	}

	if (!gconf_client_set_bool(self->gconf_client,
		GCONF_DIR"/auto_update", self->auto_update, &error))
	{
		g_warning(WIDGET_NAME": failed to set configuration value auto_update: %s",
			error->message);
		g_error_free(error);
		error = NULL;
	}
}


// TODO: skip draw if not on active desktop
static void
draw (CCCHHDoorIsDesktopWidget *self, cairo_t *cr)
{
	g_assert(IS_CCCHH_DOORIS_DESKTOP_WIDGET(self));

	g_debug(WIDGET_NAME": draw()");

	if (!self->image_current) {
		g_debug(WIDGET_NAME": draw() without image");
		self->image_current = cairo_image_surface_create_from_png(IMAGE_DOORIS_FNORD);
	}

	cairo_save(cr);
	
	// transparent background
	cairo_set_source_rgba(cr, IMAGE_COLOR_BG, IMAGE_ALPHA_BG);
	cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
	cairo_paint(cr);

	// paint icon
	cairo_set_source_surface(cr, self->image_current, 5, 5);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	cairo_paint_with_alpha(cr, self->current_alpha);

	// if not connected, overlay icon with "disconnected" image
	if (!self->connected) {
		g_debug(WIDGET_NAME": not connected, overlay image");
		cairo_surface_t *image_overlay = cairo_image_surface_create_from_png(IMAGE_DISCONNECTED);
		cairo_set_source_surface(cr, image_overlay, 5, 5);
		cairo_paint_with_alpha(cr, self->current_alpha * IMAGE_DISCONNECTED_ALPHA);
		cairo_surface_destroy(image_overlay);
	}

	// if button pressed, highlight with white mask
	if (self->highlighted) {
		cairo_set_source_rgba(cr, IMAGE_COLOR_HIGHLIGHT, IMAGE_ALPHA_HIGHLIGHT);
		cairo_mask_surface(cr, self->image_current, 5, 5);
	}

	// dim icon if timer stopped
	if (!self->timeout_handler) {
		cairo_set_source_rgba(cr, IMAGE_COLOR_DIM, IMAGE_ALPHA_DIM);
		cairo_mask_surface(cr, self->image_current, 5, 5);
	}

	cairo_restore(cr);
}


static void
show_status_banner (CCCHHDoorIsDesktopWidget *self)
{
	g_assert(IS_CCCHH_DOORIS_DESKTOP_WIDGET(self));

	time_t timenow, timediff;
	gchar *text, *ts_text;
	const gchar *state_text;

	g_debug(WIDGET_NAME": show_status_banner()");

	time (&timenow);

	// format the time elapsed since last state change
	// TODO: is it correct and portable to cast time_t to ulong?

	if (timenow != -1 && self->last_change_ts) {
		timediff = timenow - self->last_change_ts;

		if (timediff < 60) {
			ts_text = g_strdup_printf(_("(%lds ago)"),
				(ulong)timediff);
		} else if (timediff < 3600) {
			ts_text = g_strdup_printf(_("(%ldm %lds ago)"),
				(ulong)timediff / 60,
				(ulong)timediff % 60);
		} else if (timediff < 86400) {
			ts_text = g_strdup_printf(_("(%ldh %ldm %lds ago)"),
				(ulong)timediff / 3600,
				(ulong)(timediff % 3600) / 60,
				(ulong)timediff % 60);
		} else {
			ts_text = g_strdup_printf(_("(%ldd %ldh %ldm %lds ago)"),
				(ulong)timediff / 86400,
				(ulong)(timediff % 86400) / 3600,
				(ulong)(timediff % 3600) / 60,
				(ulong)timediff % 60);
		}
	} else {
		ts_text = g_strdup("");
	}

	switch (self->current_state) {
		case DOORIS_OPEN:
			state_text = _("CCCHH: Door <b>unlocked</b>"); break;
		case DOORIS_CLOSED:
			state_text = _("CCCHH: Door locked"); break;
		default:
			state_text = _("CCCHH: Unknown");
			// repurpose ts_text to display self->too_old
			g_free (ts_text);
			ts_text = g_strdup((self->too_old) ? _("(data too old)") : (""));
			break;
	}

	// assemble string to display
	text = g_strdup_printf("%s %s %s", state_text, ts_text,
			(self->connected) ? ("") : _("(offline)"));

	// display banner on screen
	hildon_banner_show_information_with_markup (GTK_WIDGET (self),
		NULL, text);

	g_free (text);
	g_free (ts_text);
}


static void
force_redraw (CCCHHDoorIsDesktopWidget *self)
{
	g_assert(IS_CCCHH_DOORIS_DESKTOP_WIDGET(self));

	g_debug(WIDGET_NAME": force_redraw()");

	// Invalidate the drawing
	GtkWidget *widget = GTK_WIDGET (self);

	if (!widget->window)
		return;

	// Redraw the cairo canvas completely by exposing it
	GdkRegion *region = gdk_drawable_get_clip_region (widget->window);
	if (!region)
		return;
	gdk_window_invalidate_region (widget->window, region, TRUE);

	// process expose event
	gdk_window_process_updates (widget->window, TRUE);

	gdk_region_destroy (region);
}




// async update with callbacks

// lock fetch mutex, redraw transparent button
static inline gboolean
fetch_update_lock (CCCHHDoorIsDesktopWidget *self)
{
	g_assert(IS_CCCHH_DOORIS_DESKTOP_WIDGET(self));

	if (!g_static_mutex_trylock(&(self->fetch_lock))) {
		return FALSE;
	}

	// indicate update in progress, make icon transparent
	self->current_alpha = IMAGE_ALPHA_PROGRESS;
	force_redraw(self);

	return TRUE;
}


// unlock fetch mutex, redraw opaque button
static inline void
fetch_update_unlock (CCCHHDoorIsDesktopWidget *self)
{
	g_assert(IS_CCCHH_DOORIS_DESKTOP_WIDGET(self));

	// reset icon transparency to normal
	self->current_alpha = IMAGE_ALPHA_NORMAL;
	force_redraw(self);

	g_static_mutex_unlock(&(self->fetch_lock));
}


// initialize async update from server, lock mutex
static void
fetch_update_start (CCCHHDoorIsDesktopWidget *self)
{
	g_assert(IS_CCCHH_DOORIS_DESKTOP_WIDGET(self));

	g_debug(WIDGET_NAME": fetch_update_start()");

	// conn_wlan_only: update anyway if force_banner is set,
	// indicating a button press, even if !connected.
	// self->connected does not always represent real
	// connection state if conn_wlan_only is set.
	// connected_bearer should be NULL if network is down
	if (!self->connected &&
			!(self->conn_wlan_only && self->force_banner &&
			  self->connected_bearer)) {
		g_debug(WIDGET_NAME": not connected, not fetching new state");
		return;
	}

	if (!fetch_update_lock(self)) {
		g_message(WIDGET_NAME": fetch_lock set, not updating");
		return;
	}

	g_debug(WIDGET_NAME": begin update from URL: %s", self->update_url);

	// abort update if timer expires
	self->gvfs_timeout_handler = g_timeout_add_seconds(TIMEOUT_CONNECT, fetch_update_cancel_cb, self);

	// begin update
	// next callback: either fetch_update_async_open_cb() or fetch_update_cancel_cb()
	gnome_vfs_async_open(&self->gvfs_handle, self->update_url, GNOME_VFS_OPEN_READ,
		GNOME_VFS_PRIORITY_MIN, fetch_update_async_open_cb, self);
}


static gboolean
fetch_update_cancel_cb (gpointer data)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (data);
  
	g_debug(WIDGET_NAME": fetch_update_cancel_cb()");

	if (g_source_is_destroyed(g_main_current_source())) {
		// this shouldn't happen
		g_warning(WIDGET_NAME": update_cancel: source destroyed");
		return FALSE;
	}

	// cancel gvfs operation
	g_warning(WIDGET_NAME": I/O timeout, cancelling update");
	gnome_vfs_async_cancel(self->gvfs_handle);

	// set internal state to unknown and unlock mutex
	fetch_update_process(self, DOORIS_FNORD, 0L, 0L);

	return FALSE;
}

	
// handle async open callback, begin read
static void
fetch_update_async_open_cb (GnomeVFSAsyncHandle *handle, GnomeVFSResult result,
    gpointer data)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (data);

	g_debug(WIDGET_NAME": fetch_update_async_open_cb()");

	if (self->gvfs_timeout_handler) {
		g_source_remove (self->gvfs_timeout_handler);
		self->gvfs_timeout_handler = 0;
	}

	if (result != GNOME_VFS_OK) {
		g_warning(WIDGET_NAME": error opening URL: %s", gnome_vfs_result_to_string (result));
		fetch_update_process(self, DOORIS_FNORD, 0L, 0L);
		return;
	}

	// reuse input buffer, ensure it has at least MAX_HTTP_SIZE+1
	if (!(self->gvfs_input_buffer = g_try_realloc(self->gvfs_input_buffer, MAX_HTTP_SIZE+1))) {
		g_warning(WIDGET_NAME": failed to allocate input buffer");
		fetch_update_process(self, DOORIS_FNORD, 0L, 0L);
		return;
	}
	// initialize input buffer with zeroes, so strtok won't read beyond its end
	memset(self->gvfs_input_buffer, 0, MAX_HTTP_SIZE+1);

	self->gvfs_timeout_handler = g_timeout_add_seconds(TIMEOUT_READ, fetch_update_cancel_cb, self);

	gnome_vfs_async_read(handle, self->gvfs_input_buffer, MAX_HTTP_SIZE,
		fetch_update_async_read_cb, self);
}


// handle async read callback, begin close, parse input
static void
fetch_update_async_read_cb (GnomeVFSAsyncHandle *handle, GnomeVFSResult result,
	gpointer buffer, GnomeVFSFileSize bytes_requested, GnomeVFSFileSize bytes_read,
	gpointer data)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (data);

	g_debug(WIDGET_NAME": fetch_update_async_read_cb()");

	if (self->gvfs_timeout_handler) {
		g_source_remove (self->gvfs_timeout_handler);
		self->gvfs_timeout_handler = 0;
	}

	if (result != GNOME_VFS_OK) {
		g_warning(WIDGET_NAME": read error: %s", gnome_vfs_result_to_string (result));
		fetch_update_process(self, DOORIS_FNORD, 0L, 0L);
		return;
	}

	g_debug(WIDGET_NAME": read %"GNOME_VFS_SIZE_FORMAT_STR" bytes", bytes_read);

	if (!self->gvfs_input_buffer) {
		g_warning(WIDGET_NAME": no input buffer after read");
		fetch_update_process(self, DOORIS_FNORD, 0L, 0L);
		return;
	}

	// ensure buffer has a zero byte at MAX_HTTP_SIZE for strtok
	if (!(self->gvfs_input_buffer = g_try_realloc(self->gvfs_input_buffer, MAX_HTTP_SIZE+1))) {
		g_warning(WIDGET_NAME": failed to allocate input buffer");
		fetch_update_process(self, DOORIS_FNORD, 0L, 0L);
		return;
	}
	self->gvfs_input_buffer[MAX_HTTP_SIZE] = '\x0';

	// TODO: no timeout on close for now,
	// cancel_cb would call fetch_update_process() again
	//self->gvfs_timeout_handler = g_timeout_add_seconds(TIMEOUT_CLOSE, fetch_update_cancel_cb, self);

	// close file
	gnome_vfs_async_close(handle, fetch_update_async_close_cb, self);

	// meanwhile, parse input buffer

	gulong ts, new_change_ts, new_report_ts;
	dooris_state_t new_state;
	char *line, *tokctx;

	new_state = DOORIS_FNORD;
	new_change_ts = 0L;
	new_report_ts = 0L;
	tokctx = NULL;

	// fixed-format text response

	// first line of body is either "locked" or "unlocked"

	if (!(line = strtok_r(self->gvfs_input_buffer, "\n", &tokctx))) {
		g_warning(WIDGET_NAME": first line of response not found");
		fetch_update_process(self, DOORIS_FNORD, 0L, 0L);
		return;
	}

	if (!strcmp(line, "locked")) {
		new_state = DOORIS_CLOSED;
		g_debug(WIDGET_NAME": new state: closed");
	} else if (!strcmp(line, "unlocked")) {
		new_state = DOORIS_OPEN;
		g_debug(WIDGET_NAME": new state: open");
	} else {
		g_warning(WIDGET_NAME": could not parse first line of response");
		fetch_update_process(self, DOORIS_FNORD, 0L, 0L);
		return;
	}

	// second line of body is last sensor report timestamp

	if ((line = strtok_r(NULL, "\n", &tokctx))) {
		errno = 0;
		ts = strtoul(line, (char **) NULL, 10);
		if (ts && !errno) {
			new_report_ts = ts;
			g_debug(WIDGET_NAME": last_report_ts: %lu, errno: %i\n", ts, errno);
		} else {
			g_warning(WIDGET_NAME": could not parse last_report_ts");
		}

		// third line of body is last state change timestamp

		if ((line = strtok_r(NULL, "\n", &tokctx))) {
			errno = 0;
			ts = strtoul(line, (char **) NULL, 10);
			if (ts && !errno) {
				new_change_ts = ts;
				g_debug(WIDGET_NAME": last_change_ts: %lu, errno: %i\n", ts, errno);
			} else {
				g_warning(WIDGET_NAME": could not parse last_change_ts");
			}
		}
	}

	fetch_update_process(self, new_state, new_change_ts, new_report_ts);
}


// handle async close callback
static void
fetch_update_async_close_cb (GnomeVFSAsyncHandle *handle, GnomeVFSResult result,
	gpointer data)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (data);

	g_debug(WIDGET_NAME": fetch_update_async_close_cb()");

	if (self->gvfs_timeout_handler) {
		g_source_remove (self->gvfs_timeout_handler);
		self->gvfs_timeout_handler = 0;
	}

	if (result != GNOME_VFS_OK) {
		g_debug(WIDGET_NAME": error on close: %s", gnome_vfs_result_to_string (result));
		// Nothing else to do here
	}
}


// update internal state, unlock mutex
static void
fetch_update_process (CCCHHDoorIsDesktopWidget *self, dooris_state_t new_state,
	gulong new_change_ts, gulong new_report_ts)
{
	g_assert(IS_CCCHH_DOORIS_DESKTOP_WIDGET(self));

	time_t timenow;

	g_debug(WIDGET_NAME": fetch_update_process()");

	self->current_state = new_state;
	self->last_change_ts = new_change_ts;
	self->last_report_ts = new_report_ts;
	self->too_old = FALSE;

	time (&timenow);
	if (timenow != -1 && self->last_report_ts && (timenow - self->last_report_ts) > REPORT_TIMEOUT) {
		g_message(WIDGET_NAME": fetch_update_process: last_report_ts too old, forcing state to DOORIS_FNORD");
		self->too_old = TRUE;
		self->current_state = DOORIS_FNORD;
	}

	if (self->current_state == self->last_state) {
		g_debug(WIDGET_NAME": state unchanged");

		// button sets force_banner, display current state even if unchanged
		if (self->force_banner) {
			self->force_banner = FALSE;
			show_status_banner(self);
		}

		fetch_update_unlock(self);
		return;
	}

	// state has changed since last update

	// release old icon surface
	if (self->image_current) {
		cairo_surface_destroy(self->image_current);
		self->image_current = NULL;
	}

	// set new icon surface
	switch (self->current_state) {
		case DOORIS_OPEN:
			self->image_current = cairo_image_surface_create_from_png(IMAGE_DOORIS_OPEN);
			g_debug(WIDGET_NAME": image_current set to open");
			break;
		case DOORIS_CLOSED:
			self->image_current = cairo_image_surface_create_from_png(IMAGE_DOORIS_CLOSED);
			g_debug(WIDGET_NAME": image_current set to closed");
			break;
		default:
			self->image_current = cairo_image_surface_create_from_png(IMAGE_DOORIS_FNORD);
			g_debug(WIDGET_NAME": image_current set to fnord");
			break;
	}

	self->last_state = self->current_state;

	// state is valid and has changed since last known state
	if (self->current_state != DOORIS_FNORD && self->current_state != self->last_known_state) {

		self->last_known_state = self->current_state;
		g_debug(WIDGET_NAME": state changed");

		// state changed: redraw icon
		force_redraw(self);

		// flash banner if notifications are enabled
		if (self->notify_enabled) {
			self->force_banner = TRUE;
		}
	}

	// set by button or state change
	if (self->force_banner) {
		self->force_banner = FALSE;
		show_status_banner(self);
	}

	fetch_update_unlock(self);
}




// event handlers

// timer event - trigger update
static gboolean
ccchh_dooris_desktop_widget_on_timeout (gpointer data)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (data);
  
	g_debug(WIDGET_NAME": ccchh_dooris_desktop_widget_on_timeout()");

	if (g_source_is_destroyed(g_main_current_source())) {
		// this shouldn't happen
		g_warning(WIDGET_NAME": on_timeout: source destroyed");
		return FALSE;
	}

	// stop timer when offline
	// connection event starts it again
	if (!self->connected) {
		g_debug(WIDGET_NAME": not connected, stopping update timer");
		// TODO: do we need remove_source() here?
		self->timeout_handler = 0;
		force_redraw(self);

		return FALSE;
	}

	fetch_update_start(self);

	return TRUE; /* keep running this event */
}


// expose event - draw icon
static gboolean
ccchh_dooris_desktop_widget_expose_event (GtkWidget *widget, GdkEventExpose *event)
{    
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (widget);

	g_debug(WIDGET_NAME": ccchh_dooris_desktop_widget_expose_event()");

	if (!self->image_current) {
		g_debug(WIDGET_NAME": expose event without image");
		self->image_current = cairo_image_surface_create_from_png(IMAGE_DOORIS_FNORD);
		// no state yet, begin update if enabled
		if (self->auto_update)
			fetch_update_start(self);
	}

	cairo_t *cr  = gdk_cairo_create (widget->window);

	/* Clip only the exposed area of the cairo context,  
	 * to potentially avoid unnecessary drawing:
	 */
	cairo_rectangle (cr,
		event->area.x, event->area.y,
		event->area.width, event->area.height);
	cairo_clip (cr);

	draw (CCCHH_DOORIS_DESKTOP_WIDGET (widget), cr);

	cairo_destroy (cr);

	return FALSE;
}


// button press event - highlight icon
static gboolean 
ccchh_dooris_desktop_widget_on_button_press_event (GtkWidget *widget, 
  GdkEventButton * event G_GNUC_UNUSED, gpointer user_data G_GNUC_UNUSED)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (widget);

	g_debug(WIDGET_NAME": ccchh_dooris_desktop_widget_on_button_press_event()");
 
	// TODO: do we really need a button lock? fetch_update locks itself
	if (!g_static_mutex_trylock(&(self->button_lock))) {
		g_debug(WIDGET_NAME": button_lock set, not updating");
		return TRUE;
	}

	self->highlighted = TRUE;
	force_redraw(self);

	return TRUE; /* Prevent further handling of this signal. */
}


// button leave notify event - remove highlight
// TODO: should check mutex and return immediately if not locked,
//       i.e. no previous button-press
static gboolean 
ccchh_dooris_desktop_widget_on_button_leave_event (GtkWidget *widget, 
  GdkEventButton * event G_GNUC_UNUSED, gpointer user_data G_GNUC_UNUSED)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (widget);

	g_debug(WIDGET_NAME": ccchh_dooris_desktop_widget_on_button_leave_event()");

	self->highlighted = FALSE;
	force_redraw(self);

	g_static_mutex_unlock(&(self->button_lock));

	return TRUE; /* Prevent further handling of this signal. */
}


// button release event - begin update, remove highlight
static gboolean 
ccchh_dooris_desktop_widget_on_button_release_event (GtkWidget *widget, 
  GdkEventButton * event G_GNUC_UNUSED, gpointer user_data G_GNUC_UNUSED)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (widget);

	g_debug(WIDGET_NAME": ccchh_dooris_desktop_widget_on_button_release_event()");
 
	// request connection from libconic, if offline and
	// force_connection is enabled
	// TODO: this does not work immediately, the call to update()
	//       will still see !connected. maybe wait a bit or use a timer event?
	// TODO: conn_wlan_only: force libconic to use only WLAN IAPs.
	//       for now, do nothing if conn_wlan_only is set
	if (!self->connected && self->force_connection && !self->conn_wlan_only) {
		g_debug(WIDGET_NAME": not connected, requesting connection");
		/* Request connection and check for the result */
		gboolean success = con_ic_connection_connect(self->connection,
			CON_IC_CONNECT_FLAG_NONE);
		if (!success) 
			g_warning(WIDGET_NAME": request for connection failed");
	}

	self->force_banner = TRUE;
	fetch_update_start(self);

	// if a timer was running, reset it
	if (stop_update_timer(self)) {
		start_update_timer(self);
	}

	self->highlighted = FALSE;
	force_redraw(self);

	g_static_mutex_unlock(&(self->button_lock));

	return TRUE; /* Prevent further handling of this signal. */
}


// libconic connection event
static void
ccchh_dooris_desktop_widget_connection_event(ConIcConnection *connection,
  ConIcConnectionEvent *event, gpointer user_data)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (user_data);

	g_debug(WIDGET_NAME": ccchh_dooris_desktop_widget_connection_event()");

	// libconic signalled a connection state change

	ConIcConnectionStatus status = con_ic_connection_event_get_status(event);
	const gchar *iap_id = con_ic_event_get_iap_id(CON_IC_EVENT(event));
	const gchar *bearer_type = con_ic_event_get_bearer_type(CON_IC_EVENT(event));

	// Problem: When switching from GPRS to/from WLAN
	// we receive STATUS_CONNECTED for the new IAP first,
	// then STATUS_DISCONNECTED for the previous IAP

	// in STATUS_DISCONNECTING, save disconnecting IAP ID and
	// set self->connected to FALSE
	// STATUS_CONNECTED sets self->connected to TRUE
	// if STATUS_DISCONNECTED is called with the same IAP ID
	// and self->connected has reverted to TRUE, ignore the event

	switch (status) {
		case CON_IC_STATUS_DISCONNECTING:

			g_debug(WIDGET_NAME": connection event: disconnecting iap: %s", iap_id);

			if (self->disconnecting_iap) {
				g_free(self->disconnecting_iap);
				self->disconnecting_iap = NULL;
			}

			// in disconnected state, connected_bearer should be NULL
			if (self->connected_bearer) {
				g_free(self->connected_bearer);
				self->connected_bearer = NULL;
			}

			// save disconnecting IAP ID
			if (iap_id)
				self->disconnecting_iap = g_strdup(iap_id);

			// TODO: "disconnecting_iap" logic loses previous connection
			// state, forces unnecessary redraws and timer resets when roaming
			self->connected = FALSE;

			break;

		case CON_IC_STATUS_CONNECTED:

			g_debug(WIDGET_NAME": connection event: connected bearer: %s", bearer_type);

			// we are online now

			if (self->connected_bearer) {
				g_free(self->connected_bearer);
				self->connected_bearer = NULL;
			}

			// save connected bearer
			if (bearer_type)
				self->connected_bearer = g_strdup(bearer_type);

			// ignore online event if bearer type string does not begin
			// with "WLAN_" and conn_wlan_only option is set
			if (self->conn_wlan_only &&
					strncmp(self->connected_bearer, "WLAN_", 5)) {
				g_debug(WIDGET_NAME": ignoring online event from non-WLAN IAP: conn_wlan_only set");

			} else if (!self->connected) {
				self->connected = TRUE;
				g_debug(WIDGET_NAME": new connection state: online");

				// start timer
				start_update_timer(self);

				// force redraw to remove disconnection overlay
				force_redraw(self);
			}

			break;

		case CON_IC_STATUS_DISCONNECTED:

			if (iap_id && self->disconnecting_iap && self->connected &&
					!strcmp(iap_id, self->disconnecting_iap))
			{
				g_debug(WIDGET_NAME": ignoring offline event from disconnecting IAP: %s", iap_id);

			} else {

				g_debug(WIDGET_NAME": connection event: disconnected IAP: %s", iap_id);

				// we are offline now

				self->connected = FALSE;
				g_debug(WIDGET_NAME": new connection state: offline");

				// don't stop timer here
				// it will deactivate itsef if !connected
				// frequent stop/start would prevent it from firing

				// force redraw to add disconnection overlay
				force_redraw(self);
			}

			if (self->disconnecting_iap) {
				g_free(self->disconnecting_iap);
				self->disconnecting_iap = NULL;
			}

			break;

		case CON_IC_STATUS_NETWORK_UP:

			g_debug(WIDGET_NAME": connection event: network up");

			// nothing to do here
			break;
	}
}


// configuration dialog response event - save or discard changes
static void 
ccchh_dooris_desktop_widget_on_settings_dialog_response (GtkDialog *dialog,
  gint response_id, gpointer user_data)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (user_data);
	g_assert(self->settings_window);
	g_assert(self->settings_window == GTK_WIDGET (dialog));
  
	g_debug(WIDGET_NAME": ccchh_dooris_desktop_widget_on_settings_dialog_response()");

	// only use new settings if user pressed the OK button
	switch (response_id) {

		// this is the "Defaults" button response
		case GTK_RESPONSE_NO:
			g_debug(WIDGET_NAME": restoring default configuration");

			gtk_entry_set_text(GTK_ENTRY(self->url_entry), DEFAULT_URL);

			hildon_check_button_set_active(HILDON_CHECK_BUTTON(self->notify_button), TRUE);
			hildon_check_button_set_active(HILDON_CHECK_BUTTON(self->connforce_button), TRUE);
			hildon_check_button_set_active(HILDON_CHECK_BUTTON(self->connwlan_button), FALSE);
			hildon_check_button_set_active(HILDON_CHECK_BUTTON(self->connection_button), TRUE);

			// TODO: avoid index constant
			hildon_touch_selector_set_active(
				hildon_picker_button_get_selector(HILDON_PICKER_BUTTON(self->update_picker)),
				0, 4);

			hildon_button_set_value(HILDON_BUTTON (self->update_picker), "23");

			// keep dialog window open

			break;

		// apply and save changes
		case GTK_RESPONSE_ACCEPT:
			g_debug(WIDGET_NAME": response accept");

			// status URL
			if (self->update_url) {
				free(self->update_url);
				self->update_url = NULL;
			}

			// try to build a valid URI from user input
			self->update_url = gnome_vfs_make_uri_from_input(gtk_entry_get_text(GTK_ENTRY (self->url_entry)));
			g_debug(WIDGET_NAME": new URL: %s\n", self->update_url);

			// update interval
			const gchar *choice = hildon_button_get_value(HILDON_BUTTON (self->update_picker));
			if (strcmp(choice, _("disabled"))) {
				// automatic updates enabled
				self->auto_update = TRUE;
				self->update_interval = CLAMP(g_ascii_strtod (choice, NULL), UPDATE_INTERVAL_MIN, UPDATE_INTERVAL_MAX);
				g_debug(WIDGET_NAME": new update interval: %ld\n", (long)self->update_interval);

				// reset timer
				stop_update_timer(self);
				start_update_timer(self);
			} else {
				// automatic updates disabled
				self->auto_update = FALSE;

				g_debug(WIDGET_NAME": automatic updates disabled");

				// if a timer was running, stop it
				stop_update_timer(self);
			}

			// banner notifications
			self->notify_enabled = hildon_check_button_get_active(HILDON_CHECK_BUTTON(self->notify_button));
			if (self->notify_enabled) {
				g_debug(WIDGET_NAME": notification enabled");
			} else {
				g_debug(WIDGET_NAME": notification disabled");
			}

			// monitor internet connection state, update only if online
			self->use_connection = hildon_check_button_get_active(HILDON_CHECK_BUTTON(self->connection_button));

			if (self->use_connection) {
				install_connection_handler(self);
			} else {
				remove_connection_handler(self);

				// force connection state to online
				// redraw if necessary to remove disconnection overlay
				if (!self->connected) {
					self->connected = TRUE;
					force_redraw(self);
				}

				// start timer (if auto_update enabled, too)
				start_update_timer(self);
			}

			// button forces internet connection
			self->force_connection = hildon_check_button_get_active(HILDON_CHECK_BUTTON(self->connforce_button));
			if (self->force_connection) {
				g_debug(WIDGET_NAME": button forces connection");
			} else {
				g_debug(WIDGET_NAME": button does not force connection");
			}

			// use only WLAN connections
			self->conn_wlan_only = hildon_check_button_get_active(HILDON_CHECK_BUTTON(self->connwlan_button));
			if (self->conn_wlan_only) {
				g_debug(WIDGET_NAME": use only WLAN networks");
			} else {
				g_debug(WIDGET_NAME": use any network");
			}

			save_settings(self);

			// fall through to dialog window disposal

		// if the user cancelled the dialog, do nothing
		default:
			g_debug(WIDGET_NAME": response ID was %i", response_id);

			gtk_widget_destroy (self->settings_window);
			// TODO: does this leak any object references such as GtkSizeGroup
			// or HildonCaption?
			// isn't there a better way to this? array+memset?
			self->settings_window = NULL;
			self->url_entry = NULL;
			self->update_picker = NULL;
			self->notify_button = NULL;
			self->connection_button = NULL;
			self->connforce_button = NULL;
			self->connwlan_button = NULL;
	}
}


// update interval picker callback - restrict value to UPDATE_INTERVAL_MIN - UPDATE_INTERVAL_MAX
static void
update_picker_changed_value_cb (HildonPickerButton *picker, gpointer data)
{
	gdouble number = 0;

	g_debug(WIDGET_NAME": update_picker_changed_value_cb()");

	const gchar *choice = hildon_button_get_value(HILDON_BUTTON (picker));
	if (strcmp(choice, _("disabled"))) {
		number = CLAMP(g_ascii_strtod (choice, NULL), UPDATE_INTERVAL_MIN, UPDATE_INTERVAL_MAX);
		gchar *value_text = g_strdup_printf ("%d", (int) number);
		hildon_button_set_value(HILDON_BUTTON (picker), value_text);
		g_free(value_text);
	}
}


// use_connection button callback - set dependent buttons (in-)sensitive
static void
connection_button_toggled_cb (HildonCheckButton *button, gpointer data)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET(data);
	g_debug(WIDGET_NAME": connection_button_toggled_cb()");
	gboolean active = hildon_check_button_get_active(button);
	gtk_widget_set_sensitive(self->connforce_button, active);
	gtk_widget_set_sensitive(self->connwlan_button, active);
}


// open configuration dialog
static void 
ccchh_dooris_desktop_widget_on_show_settings (GtkWidget *widget G_GNUC_UNUSED, 
  gpointer user_data G_GNUC_UNUSED)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (widget);
 
	g_debug(WIDGET_NAME": ccchh_dooris_desktop_widget_on_show_settings()");

	// could this be called when the settings window is already open?
	if (!self->settings_window) {
		/* The Hildon HIG says "Dialogs should be used for small tasks" so 
		 * we use GtkDialog instead of HildonWindow: */
		self->settings_window = gtk_dialog_new_with_buttons(
			_("CCCHH DoorIs Widget Settings"), 
			NULL /* parent window */, 
			GTK_DIALOG_NO_SEPARATOR /* flags */, 
			_("Defaults"), GTK_RESPONSE_NO,
			GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
			NULL);
		// call dialog_response() on close
		g_signal_connect (self->settings_window, "response", 
			G_CALLBACK (ccchh_dooris_desktop_widget_on_settings_dialog_response), 
			self);

		// set dialog size
		// TODO: is this the right way to do in Hildon? which w/h are best?
		gtk_window_set_default_size(GTK_WINDOW(self->settings_window), 400, 400);

		// arrange the inputs
		// we put controls and labels into hbox-es
		// and stack them into a vbox

		GtkWidget *vbox = gtk_vbox_new (FALSE, HILDON_MARGIN_DEFAULT);

		// pack the filled vbox into a scrollable area
		GtkWidget *pan = hildon_pannable_area_new ();
		hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA (pan), vbox);

		// add the area into the window
		gtk_box_pack_start (GTK_BOX(GTK_DIALOG(self->settings_window)->vbox), GTK_WIDGET(pan), TRUE, TRUE, 0);

		// define the controls

		GtkWidget *hbox = gtk_hbox_new (FALSE, HILDON_MARGIN_DEFAULT);
		gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);

		GtkSizeGroup *sizegroup = gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL);

		// status url
		self->url_entry = hildon_entry_new (HILDON_SIZE_FINGER_HEIGHT);
		GtkWidget *caption = hildon_caption_new(NULL, _("Status URL:"),
			self->url_entry, NULL, HILDON_CAPTION_MANDATORY);
		gtk_box_pack_start (GTK_BOX (hbox), caption, TRUE, TRUE, 0);

		hbox = gtk_hbox_new (FALSE, HILDON_MARGIN_DEFAULT);
		gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);

		gint selidx = 0;
		gint idx = 1;

		// predefined choices for update interval selector
		HildonTouchSelector *selector = HILDON_TOUCH_SELECTOR (hildon_touch_selector_entry_new_text());
		hildon_touch_selector_append_text (selector, _("disabled"));
		if (!selidx && !self->auto_update)
			selidx = idx;
		idx++;
		hildon_touch_selector_append_text (selector, "5");
		if (!selidx && self->update_interval <= 5)
			selidx = idx;
		idx++;
		hildon_touch_selector_append_text (selector, "10");
		if (!selidx && self->update_interval <= 10)
			selidx = idx;
		idx++;
		hildon_touch_selector_append_text (selector, "15");
		if (!selidx && self->update_interval <= 15)
			selidx = idx;
		idx++;
		hildon_touch_selector_append_text (selector, "20");
		if (!selidx && self->update_interval <= 20)
			selidx = idx;
		idx++;
		hildon_touch_selector_append_text (selector, "25");
		if (!selidx && self->update_interval <= 25)
			selidx = idx;
		idx++;
		hildon_touch_selector_append_text (selector, "30");
		if (!selidx && self->update_interval <= 30)
			selidx = idx;
		idx++;
		hildon_touch_selector_append_text (selector, "45");
		if (!selidx && self->update_interval <= 45)
			selidx = idx;
		idx++;
		hildon_touch_selector_append_text (selector, "60");
		if (!selidx && self->update_interval <= 60)
			selidx = idx;
		idx++;
		hildon_touch_selector_set_active (selector, 0, selidx-1);

		// update interval
		self->update_picker = hildon_picker_button_new (HILDON_SIZE_FINGER_HEIGHT,
			HILDON_BUTTON_ARRANGEMENT_VERTICAL);
		hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (self->update_picker), selector);

		// install callback function to validate input
		g_signal_connect (G_OBJECT (self->update_picker), "value-changed",
			G_CALLBACK (update_picker_changed_value_cb), NULL);

		// align left, center vertically
		gtk_button_set_alignment(GTK_BUTTON(self->update_picker), 0, 0.5);
		caption = hildon_caption_new(sizegroup, _("Update interval (minutes):"),
			self->update_picker, NULL, HILDON_CAPTION_MANDATORY);
		gtk_box_pack_start (GTK_BOX (hbox), caption, TRUE, TRUE, 0);

		hbox = gtk_hbox_new (FALSE, HILDON_MARGIN_DEFAULT);
		gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);
    
		// banner notifications
		self->notify_button = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
		gtk_button_set_label(GTK_BUTTON (self->notify_button), _("Notify on state change"));
		// align left, center vertically
		gtk_button_set_alignment(GTK_BUTTON(self->notify_button), 0, 0.5);
		gtk_box_pack_start (GTK_BOX (hbox), self->notify_button, TRUE, TRUE, 0);

		hbox = gtk_hbox_new (FALSE, HILDON_MARGIN_DEFAULT);
		gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);

		// monitor internet connection, only update when online
		self->connection_button = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
		gtk_button_set_label(GTK_BUTTON (self->connection_button), _("Only update when online"));
		gtk_button_set_alignment(GTK_BUTTON(self->connection_button), 0, 0.5);
		gtk_box_pack_start (GTK_BOX (hbox), self->connection_button, TRUE, TRUE, 0);

		// install signal handler to toggle sensitivity of connforce/wlan_button
		g_signal_connect (self->connection_button, "toggled", G_CALLBACK(connection_button_toggled_cb), self);

		hbox = gtk_hbox_new (FALSE, HILDON_MARGIN_DEFAULT);
		gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);

		// button forces connection
		self->connforce_button = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
		gtk_button_set_label(GTK_BUTTON (self->connforce_button), _("Button requests internet connection"));
		gtk_button_set_alignment(GTK_BUTTON(self->connforce_button), 0, 0.5);
		gtk_box_pack_start (GTK_BOX (hbox), self->connforce_button, TRUE, TRUE, 0);

		hbox = gtk_hbox_new (FALSE, HILDON_MARGIN_DEFAULT);
		gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);

		// use only WLAN
		self->connwlan_button = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
		gtk_button_set_label(GTK_BUTTON (self->connwlan_button), _("Use only WLAN connections"));
		gtk_button_set_alignment(GTK_BUTTON(self->connwlan_button), 0, 0.5);
		gtk_box_pack_start (GTK_BOX (hbox), self->connwlan_button, TRUE, TRUE, 0);

		g_object_unref(sizegroup);
	}

	// set current values
	gtk_entry_set_text(GTK_ENTRY(self->url_entry), self->update_url);

	gchar *value_text;
	if (self->auto_update) {
		value_text = g_strdup_printf ("%d", self->update_interval);
	} else {
		value_text = g_strdup(_("disabled"));
	}
	hildon_button_set_value(HILDON_BUTTON (self->update_picker), value_text);
	g_free(value_text);

	hildon_check_button_set_active(HILDON_CHECK_BUTTON(self->notify_button), self->notify_enabled);
	hildon_check_button_set_active(HILDON_CHECK_BUTTON(self->connection_button), self->use_connection);
	hildon_check_button_set_active(HILDON_CHECK_BUTTON(self->connforce_button), self->force_connection);
	hildon_check_button_set_active(HILDON_CHECK_BUTTON(self->connwlan_button), self->conn_wlan_only);
	gtk_widget_set_sensitive(self->connforce_button, self->use_connection);
	gtk_widget_set_sensitive(self->connwlan_button, self->use_connection);

	gtk_widget_show_all(self->settings_window);
}




static void
ccchh_dooris_desktop_widget_realize (GtkWidget *widget)
{
	g_debug(WIDGET_NAME": ccchh_dooris_desktop_widget_realize()");

	/* Use An RGBA colormap rather than RGB, 
	 * so we can use transparency in our expose_event() implementation.
	 */
	GdkScreen *screen = gtk_widget_get_screen (widget);
	gtk_widget_set_colormap (widget,
		gdk_screen_get_rgba_colormap (screen));

	gtk_widget_set_app_paintable (widget, TRUE);

	/* Call the base class's implementation: */
	GTK_WIDGET_CLASS (ccchh_dooris_desktop_widget_parent_class)->realize (widget);
}


// TODO: this function is called twice. why?
static void
ccchh_dooris_desktop_widget_dispose (GObject *object)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (object);
 
	g_debug(WIDGET_NAME": ccchh_dooris_desktop_widget_dispose()");

	// remove settings window if it is still open
	if (self->settings_window) {
		gtk_widget_destroy (self->settings_window);
		self->settings_window = NULL;
		self->url_entry = NULL;
	}
   
	/* Call the base class's implementation: */
	G_OBJECT_CLASS (ccchh_dooris_desktop_widget_parent_class)->dispose (object);
}


static void
ccchh_dooris_desktop_widget_finalize (GObject *object)
{
	CCCHHDoorIsDesktopWidget *self = CCCHH_DOORIS_DESKTOP_WIDGET (object);

	g_debug(WIDGET_NAME": ccchh_dooris_desktop_widget_finalize()");
 
	stop_update_timer(self);
  
	remove_connection_handler(self);

	if (self->gvfs_timeout_handler) {
		g_source_remove (self->gvfs_timeout_handler);
		self->gvfs_timeout_handler = 0;
	}

	// destroy icon surface if we still hold one
	if (self->image_current && cairo_surface_get_reference_count(self->image_current) > 0) {
		cairo_surface_destroy(self->image_current);
	}

	if (self->update_url) {
		free(self->update_url);
		self->update_url = NULL;
	}

	g_static_mutex_free(&(self->button_lock));
	g_static_mutex_free(&(self->fetch_lock));

/*
	if (self->gvfs_handle) {
		g_object_unref(self->gvfs_handle);
		self->gvfs_handle = NULL;
	}
*/

	if (self->gvfs_input_buffer) {
		g_free(self->gvfs_input_buffer);
		self->gvfs_input_buffer = NULL;
	}

	if (self->disconnecting_iap) {
		g_free(self->disconnecting_iap);
		self->disconnecting_iap = NULL;
	}

	if (self->connected_bearer) {
		g_free(self->connected_bearer);
		self->connected_bearer = NULL;
	}

	if (self->gconf_client) {
		g_object_unref(self->gconf_client);
		self->gconf_client = NULL;
	}

	/* Call the base class's implementation: */
	G_OBJECT_CLASS (ccchh_dooris_desktop_widget_parent_class)->finalize (object);
}


static void
ccchh_dooris_desktop_widget_class_init (CCCHHDoorIsDesktopWidgetClass *klass)
{

	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);

	g_debug(WIDGET_NAME": ccchh_dooris_desktop_widget_class_init()");

	object_class->dispose = ccchh_dooris_desktop_widget_dispose;
	object_class->finalize = ccchh_dooris_desktop_widget_finalize;
 
	widget_class->realize = ccchh_dooris_desktop_widget_realize;
	widget_class->expose_event = ccchh_dooris_desktop_widget_expose_event;
}


static void
ccchh_dooris_desktop_widget_class_finalize (CCCHHDoorIsDesktopWidgetClass *klass G_GNUC_UNUSED)
{
	g_debug(WIDGET_NAME": ccchh_dooris_desktop_widget_class_finalize()");
}


static void
ccchh_dooris_desktop_widget_init (CCCHHDoorIsDesktopWidget *self)
{
	g_assert(IS_CCCHH_DOORIS_DESKTOP_WIDGET(self));

	g_debug(WIDGET_NAME": ccchh_dooris_desktop_widget_init()");

/*
	// initialize GnomeVFS library
	if (!gnome_vfs_init()) {
		// FIXME: crashes the whole desktop process
		g_error(WIDGET_NAME": could not initialize GnomeVFS library");
		abort();
	}
*/
 
	// default settings
	self->notify_enabled = TRUE;
	self->update_interval = 23;
	self->update_url = gnome_vfs_make_uri_from_input(DEFAULT_URL);
	self->use_connection = TRUE;
	self->force_connection = TRUE;
	self->auto_update = TRUE;
	self->conn_wlan_only = FALSE;

	// initialize state
	// this combination ensures that update() sets an image on first call
	// TODO: is there a better way?
	self->current_state = DOORIS_FNORD;
	self->last_state = -1;
	self->last_known_state = DOORIS_FNORD;

	self->last_change_ts = 0L;
	self->last_report_ts = 0L;
	self->too_old = FALSE;

	// set image to NULL, first expose event will then call update()
	self->image_current = NULL;
	self->current_alpha = IMAGE_ALPHA_NORMAL;

	g_static_mutex_init(&(self->button_lock));
	g_static_mutex_init(&(self->fetch_lock));

	self->gconf_client = gconf_client_get_default();

	self->force_banner = FALSE;

	self->disconnecting_iap = NULL;
	self->connected_bearer = NULL;

	self->gvfs_handle = NULL;
	self->gvfs_input_buffer = NULL;
	self->gvfs_timeout_handler = 0;

	// set widget size
	// TODO: is this the right way?
	gtk_widget_set_size_request (GTK_WIDGET (self), IMAGE_WIDGET_WIDTH, IMAGE_WIDGET_HEIGHT);
	gtk_window_resize (GTK_WINDOW (self), IMAGE_WIDGET_WIDTH, IMAGE_WIDGET_HEIGHT);

	/* Allow this widget to handle button-press events:
	 * Note that gtk_widget_add_events would only work after the widget is realized:
	 */
	gint mask = gtk_widget_get_events (GTK_WIDGET (self)) | GDK_BUTTON_PRESS_MASK |
		GDK_BUTTON_RELEASE_MASK | GDK_LEAVE_NOTIFY_MASK;
	gtk_widget_set_events (GTK_WIDGET (self), mask);
     
	/* Handle mouse button events: */
	g_signal_connect (self, "button-press-event", 
		G_CALLBACK (ccchh_dooris_desktop_widget_on_button_press_event), NULL);

	g_signal_connect (self, "button-release-event", 
		G_CALLBACK (ccchh_dooris_desktop_widget_on_button_release_event), NULL);

	g_signal_connect (self, "leave-notify-event", 
		G_CALLBACK (ccchh_dooris_desktop_widget_on_button_leave_event), NULL);

	/* Specify that a settings button should be shown in layout mode, 
	 * and handle a request to configure the settings:
	 */
	hd_home_plugin_item_set_settings (HD_HOME_PLUGIN_ITEM (self), TRUE);
	g_signal_connect (self, "show-settings", 
		G_CALLBACK (ccchh_dooris_desktop_widget_on_show_settings), NULL);   

	if (gconf_client_dir_exists(self->gconf_client, GCONF_DIR, NULL)) {
		load_settings(self);
	}

	/* Create connection object */
	self->connection = con_ic_connection_new ();
	self->connection_handler = 0;

	if (self->use_connection) {
		// connection event will eventually
		// set self->connected and start timer
		install_connection_handler(self);
	} else {
		self->connected = TRUE;
		start_update_timer(self);
	}

	g_debug(WIDGET_NAME": initial URL: %s\n", self->update_url);
}


// this function is never called, is it?
CCCHHDoorIsDesktopWidget*
ccchh_dooris_desktop_widget_new (void)
{
	g_debug(WIDGET_NAME": ccchh_dooris_desktop_widget_new()");
	return g_object_new (TYPE_CCCHH_DOORIS_DESKTOP_WIDGET, NULL);
}
