/*
 *  global.c - auxiliary routines, mainly gp screen management and config
 *
 *  Copyright (C) 1997-2004 John Coppens (john@jcoppens.com)
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifdef HAVE_CONFIG_H                                                            
#  include <config.h>                                                           
#endif                                                                          

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>

#include "gtkimageview.h"
#include "support.h"
#include "interface.h"
#include "interface2.h"
#include "global.h"
#include "conf.h"
#include "main.h"
#include "garchivereader.h"
#include "garchivereadercommandline.h"
#include "garchivereaderunzip.h"
#include "imagecache.h"

#ifdef USE_HILDON
#    include <hildon/hildon-program.h>
#    include <hildon/hildon-note.h>
#    include <hildon/hildon-banner.h>
#    include <hildon/hildon-defines.h>
#endif

enum FitSettings {
  FitSettings_NONE = 0,
  FitSettings_PAGE = 1,
  FitSettings_WIDTH = 2
};

#define ZONE_SIZE 20

int		 page_nr = 0,
  timer_id = 0;
#if USE_HILDON
#define FIT_WIDTH_MARGIN 40
#else
#define FIT_WIDTH_MARGIN 20
#endif

GtkImageView    *book_canvas = NULL;
double		 total_w = 0.0, total_h = 0.0;
gboolean	 file_list_expanded = FALSE;
int              fullscreen = 0;
int              rotated = 0;
enum FitSettings zoom_fit = FitSettings_PAGE;
gdouble          start_x = -1, start_y = -1;

magic_sign zip_magic = { 0, 4, 0, "PK\003\004" };
magic_sign rar_magic = { 0, 4, 1, "Rar!" };
magic_sign sz_magic = { 0, 6, 53, "7z\xbc\xaf\x27\x1C" };

gchar* zip_list_command[] = { "unzip", "-Z", "-1", "%a", NULL };
gchar* zip_decompress_command[] = { "unzip", "-p", "-C", "%a", "%F", NULL };

gchar* rar_list_command[] = { "unrar", "v", "%a", NULL };
gchar* rar_decompress_command[] = { "unrar", "p", "-ierr", "-c-", "--", "%a", "%f", NULL };

gchar* sz_list_command[] = { "7z", "l", "%a", NULL };
gchar* sz_decompress_command[] = { "7z", "e", "-so", "%a", "%F", NULL };

#define NUM_READERS 3
GArchiveReader* readers[NUM_READERS];
GArchiveReader* reader;

guint timeout_id = 0;

guchar* image_buffer = NULL;

typedef void (*accel_callback)(void);
GHashTable* accel_table = NULL;

#ifdef USE_HILDON
int
ok_dialog(char* ttl, char* msg)
{
  GtkWidget* dlg;
  int i;
  dlg = hildon_note_new_information(GTK_WINDOW(MainWindow), msg);
  i = gtk_dialog_run(GTK_DIALOG(dlg));
  gtk_widget_destroy(GTK_WIDGET(dlg));
  return i;
}

void
info_dialog(char* msg)
{
  GtkWidget* dlg;
  dlg = hildon_banner_show_information(MainWindow,
				       NULL,
				       msg);
}
#else
int
ok_dialog(char *ttl, char *msg)
{
  GtkWidget *dlg;
  gint response;

  dlg = gtk_message_dialog_new(GTK_WINDOW(MainWindow), 
			       GTK_DIALOG_MODAL,
			       GTK_MESSAGE_INFO,
			       GTK_BUTTONS_OK,
			       "%s",
			       msg);
  gtk_window_set_title(GTK_WINDOW(dlg), ttl);
  gtk_window_set_position(GTK_WINDOW(dlg), GTK_WIN_POS_CENTER);
  response = gtk_dialog_run(GTK_DIALOG(dlg));
  gtk_widget_destroy(dlg);
  return response;
}

int
info_dialog(char* msg)
{
  /* nothing to do for now */
  /* TODO: create a window normally hidden that will display the message
     and disappear after some time */
}
#endif


void
initial_setup(void)
{
  book_canvas = GTK_IMAGE_VIEW(lookup_widget(MainWindow, "main_canvas"));
  image_buffer = g_malloc(pref.buffer_size);

  readers[0] = g_archive_reader_unzip_new(image_buffer, 
					  pref.buffer_size, 
					  zip_magic);
  /* Note: if you have problems with the in-memory unzipper, use this
     one instead */
  /*readers[0] = g_archive_reader_command_line_new(image_buffer, 
						 pref.buffer_size,
						 zip_magic,
						 zip_list_command,
						 zip_decompress_command);*/
  readers[1] = g_archive_reader_command_line_new(image_buffer,
						 pref.buffer_size,
						 rar_magic,
						 rar_list_command,
						 rar_decompress_command);
  readers[2] = g_archive_reader_command_line_new(image_buffer,
						 pref.buffer_size,
						 sz_magic,
						 sz_list_command,
						 sz_decompress_command);

  set_sort_strategy();

  accel_table = g_hash_table_new(g_str_hash,
				 g_str_equal);
  g_hash_table_insert(accel_table,
		      "toggle_keymode",
		      toggle_keymode);
  g_hash_table_insert(accel_table,
		      "scroll_down",
		      scroll_down);
  g_hash_table_insert(accel_table,
		      "scroll_up",
		      scroll_up);
  g_hash_table_insert(accel_table,
		      "scroll_right",
		      scroll_right);
  g_hash_table_insert(accel_table,
		      "scroll_left",
		      scroll_left);
  g_hash_table_insert(accel_table,
		      "page_down",
		      page_down);
  g_hash_table_insert(accel_table,
		      "page_up",
		      page_up);
  g_hash_table_insert(accel_table,
		      "goto_page",
		      goto_page);
}


void
clean_up(void)
{
  int i;
  if(accel_table) {
    g_hash_table_unref(accel_table);
    accel_table = NULL;
  }
  image_cache_free();
  reader = NULL;
  for(i = 0; i < NUM_READERS; ++i) {
    if(readers[i])
      g_object_unref(readers[i]);
    readers[i] = NULL;
  }
  if(image_buffer)
    g_free(image_buffer);
  image_buffer = NULL;
}

void
set_sort_strategy(void)
{
  /* re-sort and re-request the current page */ 
  image_cache_resort();
  show_page(page_nr);
}

void
start_show(void)
{
  char *p = NULL;
  int result = 0;
  GArchiveReader* tmpreader = NULL;
  int i;

  if (debug) printf("%s\n", pref.lastbook);

  for(i = 0; i < NUM_READERS; ++i) {
    result = g_archive_reader_register_path(readers[i], pref.lastbook);
    if(result == -1) {
      p = g_strdup_printf(_("File %s does not exist"), pref.lastbook);
      ok_dialog(_("File error"), p);
      g_free(p);
      return;
    } else if(result == 1) {
      tmpreader = readers[i];
      break;
    }
  }
  if(!tmpreader) {
    p = g_strdup_printf(_("Unknown file type:\n%s\n"), pref.lastbook);
    ok_dialog(_("File error"), p);
    g_free(p);
    return;
  }

  reader = tmpreader;
  image_cache_load(reader);
  if(image_cache_length() == 0) {
    gchar* last_error = g_archive_reader_get_last_error(reader);
    if(last_error) {
      printf("%s\n", last_error);
      ok_dialog(_("File error"), last_error);
      g_free(last_error);
    }
  }
  
  show_page(pref.pagenumber);
  page_nr = pref.pagenumber;		/* Thanks James Reeves jreeves[at]monkeyengines.co.uk */
}


static void
debug_scroll_adj(GtkAdjustment* adj)
{
  if (debug)
    printf(_("[vscroll] val: %.2f max: %.2f min: %.2f\n"),
	   adj->value, adj->upper, adj->lower);
}

static gboolean
clear_indicators(gpointer data) {
  if(GTK_IS_WIDGET(data)) {
    GtkWidget* canvas = GTK_WIDGET(data);
    gdk_window_invalidate_rect(canvas->window, NULL, TRUE);
  }
  timeout_id = 0;
  return FALSE;
}

static void
draw_indicators(GtkWidget* canvas, int which)
{
  GdkWindow* window = canvas->window;

  /* invert which we show if we're in reverse rotation */
  if(rotated && pref.rotate_ccw)
    which = -which;

  switch(which) {
  case 0:
    if(page_nr != 0)
      draw_indicators(canvas, -1);
    if(page_nr + pref.nrpages < image_cache_length())
      draw_indicators(canvas, 1);
    /* set up a timer to remove the arrows after a time */
    if(timeout_id != 0) {
      g_source_remove(timeout_id);
    }
    timeout_id = g_timeout_add(5000, clear_indicators, book_canvas);
    break;
  case 1:
  case -1:
    {
      gint left, top;
      GdkPoint points[3];
      gint arrow_point_x, arrow_point_y;
      gint arrow_end1_x, arrow_end1_y;
      gint arrow_end2_x, arrow_end2_y;
      gint rect_line_width = MAX(pref.zone_size / 6, 1);

      GdkGC* rect_gc = gdk_gc_new(GDK_DRAWABLE(window));
      GdkGC* line_gc = gdk_gc_new(GDK_DRAWABLE(window));

      gdk_gc_copy(rect_gc, canvas->style->black_gc);
      /* use thick, round lines */
      gdk_gc_set_line_attributes(rect_gc, 
				 rect_line_width,
				 GDK_LINE_SOLID,
				 GDK_CAP_ROUND,
				 GDK_JOIN_ROUND);
      gdk_gc_set_fill(rect_gc, GDK_SOLID);
      
      /* use thick, non-round lines */
      gdk_gc_copy(line_gc, canvas->style->white_gc);
      gdk_gc_set_line_attributes(line_gc,
				 MAX(pref.zone_size / 6, 1),
				 GDK_LINE_SOLID,
				 GDK_CAP_PROJECTING,
				 GDK_JOIN_MITER);

      /* draw the rectangle */
      if(rotated) {
	left = (canvas->allocation.width - pref.zone_size) / 2;
	if(which == -1)
	  top = 0;
	else
	  top = canvas->allocation.height - pref.zone_size;
      } else {
	top = (canvas->allocation.height - pref.zone_size) / 2;
	if(which == -1)
	  left = 0;
	else
	  left = canvas->allocation.width - pref.zone_size;
      }
      gdk_draw_line(GDK_DRAWABLE(window), rect_gc,
		    left + rect_line_width/2,
		    top + rect_line_width/2,
		    left + pref.zone_size - 1 - rect_line_width/2,
		    top + rect_line_width/2);
      gdk_draw_line(GDK_DRAWABLE(window), rect_gc,
		    left + pref.zone_size - 1 - rect_line_width/2,
		    top + rect_line_width/2,
		    left + pref.zone_size - 1 - rect_line_width/2,
		    top + pref.zone_size - 1 - rect_line_width/2);
      gdk_draw_line(GDK_DRAWABLE(window), rect_gc,
		    left + rect_line_width/2,
		    top + pref.zone_size - 1 - rect_line_width/2,
		    left + pref.zone_size - 1 - rect_line_width/2,
		    top + pref.zone_size - 1 - rect_line_width/2);
      gdk_draw_line(GDK_DRAWABLE(window), rect_gc,
		    left + rect_line_width/2,
		    top + rect_line_width/2,
		    left + rect_line_width/2,
		    top + pref.zone_size - 1 - rect_line_width/2);
      gdk_draw_rectangle(GDK_DRAWABLE(window), rect_gc, TRUE, 
			 left + rect_line_width, 
			 top + rect_line_width, 
			 pref.zone_size - rect_line_width * 2 - 1, 
			 pref.zone_size - rect_line_width * 2 - 1);
      
      /* draw the lines */
      if(rotated) {
	arrow_point_x = left + pref.zone_size / 2;
	arrow_end1_x = left + pref.zone_size * 3 / 4;
	arrow_end2_x = left + pref.zone_size / 4;
	if(which == -1) {
	  arrow_point_y = top + pref.zone_size / 3;
	  arrow_end1_y = arrow_end2_y = top + pref.zone_size * 2 / 3;
	} else {
	  arrow_point_y = top + pref.zone_size * 2 / 3;
	  arrow_end1_y = arrow_end2_y = top + pref.zone_size / 3;
	}
      } else {
	arrow_point_y = top + pref.zone_size / 2;
	arrow_end1_y = top + pref.zone_size / 4;
	arrow_end2_y = top + pref.zone_size * 3 / 4;
	if(which == -1) {
	  arrow_point_x = left + pref.zone_size / 3;
	  arrow_end1_x = arrow_end2_x = left + pref.zone_size * 2 / 3;
	} else {
	  arrow_point_x = left + pref.zone_size * 2 / 3;
	  arrow_end1_x = arrow_end2_x = left + pref.zone_size / 3;
	}
      }
      points[0].x = arrow_end1_x; points[0].y = arrow_end1_y;
      points[1].x = arrow_point_x; points[1].y = arrow_point_y;
      points[2].x = arrow_end2_x; points[2].y = arrow_end2_y;
      gdk_draw_lines(GDK_DRAWABLE(window), line_gc, points, 3);
      break;
    }
  }
}

static gboolean
draw_indicators_on_idle(gpointer data) {
  if(GTK_IS_WIDGET(data)) {
    GtkWidget* canvas = GTK_WIDGET(data);
    draw_indicators(canvas, 0);
  }
  return FALSE;
}

static void
show_hide_scrollbars_when_fullscreen(gboolean fullscreen) 
{
  GtkWidget* scroller = lookup_widget(MainWindow, "main_scroller");
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroller),
				 fullscreen && !pref.scrollbars_fullscreen ? GTK_POLICY_NEVER : GTK_POLICY_AUTOMATIC,
				 fullscreen && !pref.scrollbars_fullscreen ? GTK_POLICY_NEVER : GTK_POLICY_AUTOMATIC);
}

#if USE_HILDON
static void
set_fullscreen_ui(gboolean fullscreen)
{
  static timeout = 0;
  if(fullscreen) {
    gtk_widget_hide(nav_toolbar);
    timeout = g_timeout_add(FOLLOW_TIMER, draw_indicators_on_idle, book_canvas);
  } else {
    gtk_widget_show(nav_toolbar);
    if(timeout != 0)
      g_source_remove(timeout);
    timeout = 0;
  }
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(fullscreen_item), 
				 fullscreen);
  show_hide_scrollbars_when_fullscreen(fullscreen);
}
#else
static void
set_fullscreen_ui(gboolean fullscreen)
{
  static timeout = 0;
  if(fullscreen) {
    timeout = g_timeout_add(FOLLOW_TIMER, draw_indicators_on_idle, book_canvas);
  } else {
    if(timeout != 0)
      g_source_remove(timeout);
    timeout = 0;
  }
  show_hide_scrollbars_when_fullscreen(fullscreen);
}
#endif

void
scroll_to(double where)
{
  GtkWidget *wdg;
  GtkAdjustment *adj;

  wdg = lookup_widget(MainWindow, "main_scroller");
  if(rotated) {
    adj = gtk_scrolled_window_get_hadjustment(GTK_SCROLLED_WINDOW(wdg));
    if(!pref.rotate_ccw)
      where = 1.0 - where;
    gtk_adjustment_set_value(GTK_ADJUSTMENT(adj),
			     where*(adj->upper - adj->page_size));
    gtk_scrolled_window_set_hadjustment(GTK_SCROLLED_WINDOW(wdg), adj);
  }
  else {
    adj = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(wdg));
    gtk_adjustment_set_value(GTK_ADJUSTMENT(adj),
			     where*(adj->upper - adj->page_size));
    gtk_scrolled_window_set_vadjustment(GTK_SCROLLED_WINDOW(wdg), adj);
  }
  debug_scroll_adj(adj);
}


void
show_page(int nr)
{
  GdkPixbuf *final = NULL;
  gchar *bff;
  const gchar* page_title;
  int nr_pages;

  nr_pages = image_cache_length();

  if(!nr_pages) {
    info_dialog(_("No pages loaded\n"));
    return;
  }

  final = image_cache_request_page(nr, 
				   pref.nrpages == 2, 
				   pref.manga_mode,
				   &total_w,
				   &total_h,
				   &page_title);

#if USE_HILDON
  bff = g_strdup_printf(_("%s"),
			page_title);
#else
  bff = g_strdup_printf(_("cbrPager - %s"),
			page_title);
#endif
  gtk_window_set_title(GTK_WINDOW(MainWindow), bff);
  g_free(bff);

  if(pref.auto_rotate && 
     ((total_w >= total_h && rotated) ||
      (total_w < total_h && !rotated)))
    rotate(FALSE);

  gtk_image_view_set_pixbuf(book_canvas, final, zoom_fit == FitSettings_PAGE ? TRUE : FALSE);
  reset_fit_settings();

  /* save last requested page */
  pref.pagenumber = nr;
  save_config();
}

void
reset_fit_settings(void)
{
  if(zoom_fit == FitSettings_WIDTH)
    set_zoom_width();
  else if(zoom_fit == FitSettings_NONE)
    set_zoom_factor(zoom_factor);
}

void
fit_page(int w, int h)
{
  gtk_image_view_set_fitting(book_canvas, TRUE);
  zoom_fit = FitSettings_PAGE;
}


void
fit_width(int w)
{
  double ratio = (double)w / total_w;
  set_zoom_factor(ratio);
  /* need to reset zoom_fit because it gets clobbered by set_zoom_factor */
  zoom_fit = FitSettings_WIDTH;
}


void
next_page(int adv, gboolean show_indicator)
{
  int by = adv, nr_pages = image_cache_length();

  if (by == 0) by = pref.nrpages;
  if (page_nr == (nr_pages - by)) {
    info_dialog(_("End of archive"));
    return;
  }
  page_nr += by;

  if(show_indicator)
    draw_indicators(GTK_WIDGET(book_canvas), 1);
  show_page(page_nr);
  switch (pref.nextpage) {
  case 0: scroll_to(0.0); break;
  case 1: scroll_to(1.0); break;
  }
}


void
previous_page(int prv, gboolean show_indicator)
{
  int by = prv;
  
  if (by == 0) by = pref.nrpages;

  if(page_nr == 0) {
    info_dialog(_("Beginning of archive"));
    return;
  }
  
  page_nr -= by;
  if (page_nr < 0) {
    page_nr = 0;
  }
  
  if(show_indicator)
    draw_indicators(GTK_WIDGET(book_canvas), -1);
  show_page(page_nr);
  switch (pref.prevpage) {
  case 0: scroll_to(0.0); break;
  case 1: scroll_to(1.0); break;
  }
}


void
goto_begin(void)
{
  page_nr = 0;
  show_page(page_nr);
}


void
goto_end(void)
{
  page_nr = image_cache_length() - pref.nrpages;
  if(page_nr > 0)
    show_page(page_nr);
}


void
file_list_expand(gboolean expand)
{
  GtkWidget *nav = lookup_widget(nav_window, "nav_toolbar"),
    *paned;
  GtkTreeView *tview;

  if (expand) {
    tview = GTK_TREE_VIEW(gtk_tree_view_new());
    g_object_ref(G_OBJECT(nav));
    gtk_container_remove(GTK_CONTAINER(nav_window), nav);

    if (!pref.nav_hor) {
      paned = gtk_hpaned_new();
      gtk_widget_show(paned);
      gtk_container_add(GTK_CONTAINER(nav_window), paned);
      switch (pref.nav_pos) {
      case NAV_NW:
      case NAV_SW:
	gtk_paned_pack1(GTK_PANED(paned), nav, TRUE, FALSE);
	gtk_paned_pack2(GTK_PANED(paned), GTK_WIDGET(tview), TRUE, FALSE);
	break;
      case NAV_NE:
      case NAV_SE:
	gtk_paned_pack2(GTK_PANED(paned), nav, TRUE, FALSE);
	gtk_paned_pack1(GTK_PANED(paned), GTK_WIDGET(tview), TRUE, FALSE);
	break;
      }
    } else {
      paned = gtk_vpaned_new();
      switch (pref.nav_pos) {
      case NAV_NW:
      case NAV_NE:
      case NAV_SW:
      case NAV_SE:
	break;
      }
    }
    g_object_unref(G_OBJECT(nav));
    gtk_widget_show(GTK_WIDGET(tview));
  } else {
  }
  file_list_expanded = expand;
  printf(_("Expand %d\n"), expand);
}


void
file_list_tbtn_clicked(void)
{
  file_list_expand(!file_list_expanded);
}


void
update_nav_position(void)
{
  int x, y, w, h, nh, nw;

  gtk_window_get_position(GTK_WINDOW(MainWindow), &x, &y);

  gtk_window_set_policy(GTK_WINDOW(nav_window), FALSE, FALSE, TRUE);
  gtk_toolbar_set_orientation(
			      GTK_TOOLBAR(lookup_widget(nav_window, "nav_toolbar")),
			      pref.nav_hor ? GTK_ORIENTATION_HORIZONTAL : GTK_ORIENTATION_VERTICAL);

  gtk_window_get_size(GTK_WINDOW(MainWindow), &w, &h);
  gtk_window_get_size(GTK_WINDOW(nav_window), &nw, &nh);

  switch (pref.nav_pos) {
  case NAV_NW:
    gtk_window_move(GTK_WINDOW(nav_window), x+5, y+30);
    break;
  case NAV_NE:
    gtk_window_move(GTK_WINDOW(nav_window), x+w-nw-3, y+30);
    break;
  case NAV_SE:
    if (pref.nav_hor)
      gtk_window_move(GTK_WINDOW(nav_window), x+w-nw-3, y+h-12);
    else
      gtk_window_move(GTK_WINDOW(nav_window), x+w-nw-3, y+h-nh);
    break;
  case NAV_SW:
    if (pref.nav_hor)
      gtk_window_move(GTK_WINDOW(nav_window), x+5, y+h-12); // -nh+10);
    else
      gtk_window_move(GTK_WINDOW(nav_window), x+5, y+h-nh);
    break;
  }
}


gboolean
follow_timer(gpointer data)
{
  update_nav_position();
  return TRUE;
}


void
install_timer(void)
{
  timer_id = gtk_timeout_add(FOLLOW_TIMER, follow_timer, NULL);
}


void
set_zoom_factor(double factor)
{
  if(!gtk_image_view_get_fitting(book_canvas) &&
     zoom_factor == factor)
    return;
  gtk_image_view_set_fitting(book_canvas, FALSE);
  gtk_image_view_set_zoom(book_canvas, factor);
  zoom_fit = FitSettings_NONE;
  zoom_factor = factor;
}

void
rotate(gboolean reset_fit)
{
  if(rotated)
    gtk_image_view_rotate_cancel(book_canvas);
  else
    gtk_image_view_rotate(book_canvas,
			  pref.rotate_ccw ? 90 : 270);
  rotated = !rotated;
  if(reset_fit)
    reset_fit_settings();
}

void
toggle_fullscreen(void)
{
  /* we need to prevent recursion to avoid problems with the
     menu re-activating the fullscreen feature when pressing
     the fullscreen button.  Note that I could just have the
     button mess through the menu item, but it was easier just
     to patch here and move on.  I'll clean this up someday
     -- BGE */
  /* TODO: clean this up */
  static int prevent_recursion = 0;

  if(prevent_recursion)
    return;
  
  prevent_recursion = 1;

  if(fullscreen) {
    gtk_window_unfullscreen(GTK_WINDOW(MainWindow));
  } else {
    gtk_window_fullscreen(GTK_WINDOW(MainWindow));
  }
  fullscreen = !fullscreen;
  set_fullscreen_ui(fullscreen);

  prevent_recursion = 0;
}

void
set_zoom_fit(void)
{
  int w, h;
  
  gtk_window_get_size(GTK_WINDOW(MainWindow), &w, &h);
  fit_page(w, h);
}


static gint
find_scrollbar_width(GtkWidget* scroller, GtkWidget* scrollbar, gint default_value)
{
  gint adj = default_value;
  gint temp;

  if(scrollbar) {
    adj = 0;
    temp = 0;
    /* find the fit to width margin using scrollbar width */
    gtk_widget_style_get(scrollbar, "slider-width", &temp, NULL);
    adj += temp;
    
    temp = 0;
    gtk_widget_style_get(scrollbar, "trough-border", &temp, NULL);
    adj += temp * 2;
    
    temp = 0;
    gtk_widget_style_get(scroller, "scrollbar-spacing", &temp, NULL);
    adj += temp;
  }
  
  return adj;
}

static gint
get_scrollbar_adjustment(int rotated, const char* scroller_name) {
  static int scrollbar_width_h = -1, scrollbar_width_v = -1;
  GtkWidget* scroller;
  GtkWidget* scrollbar_h;
  GtkWidget* scrollbar_v;
  
  if(scrollbar_width_h == -1) {
    scroller = lookup_widget(MainWindow, scroller_name);
    if(scroller) {
      scrollbar_h = gtk_scrolled_window_get_hscrollbar(GTK_SCROLLED_WINDOW(scroller));
      scrollbar_v = gtk_scrolled_window_get_vscrollbar(GTK_SCROLLED_WINDOW(scroller));
    }

    scrollbar_width_h = find_scrollbar_width(scroller, scrollbar_h, FIT_WIDTH_MARGIN);
    scrollbar_width_v = find_scrollbar_width(scroller, scrollbar_v, FIT_WIDTH_MARGIN);
  }
  return rotated ? scrollbar_width_h : scrollbar_width_v;
}

void
set_zoom_width(void)
{
  int w, h, adj;
  adj = get_scrollbar_adjustment(rotated, "main_scroller");
  gtk_window_get_size(GTK_WINDOW(MainWindow), &w, &h);

  fit_width((rotated ? h : w) - adj);
}

void
main_canvas_keyevent(GdkEventKey *event, gboolean is_pressed)
{
  const char* accel = NULL;
  GtkWidget* widget = NULL;
  accel_callback callback = NULL;
#if USE_HILDON
  if(event->keyval == HILDON_HARDKEY_MENU) {
    return;
  } else {
    /* workaround Maemo bug--dpad does not return key events properly,
     * so use the key up event instead of key down.  Same problem with
     * escape key (this would sometimes crash the whole program!)
     */
    switch(event->keyval) {
    case HILDON_HARDKEY_LEFT:
    case HILDON_HARDKEY_RIGHT:
    case HILDON_HARDKEY_UP:
    case HILDON_HARDKEY_DOWN:
    case HILDON_HARDKEY_ESC:
      if(is_pressed)
	return;
      break;
    default:
      if(!is_pressed)
	return;
    }
  }
#else
  if(!is_pressed) {
    return;
  }
#endif
  
  accel = find_accelerator(event->keyval, event->state);
  if(accel == NULL)
    return;
  else if(debug)
    fprintf(stderr, _("Accel: %s\n"), accel);
  /* lookup in function table */
  callback = (accel_callback)g_hash_table_lookup(accel_table,
						 accel);
  if(callback)
    (*callback)();
  else {
    /* try to find it in the widgets */
    widget = lookup_widget(nav_window, accel);
    if(widget)
      g_signal_emit_by_name(widget, "clicked", NULL);
    else
      fprintf(stderr, _("Unbound event: %s\n"), accel);
  }
}

static gboolean
is_in_left_zone(gint x, gint y, const GtkAllocation alloc)
{
  if(rotated) {
    if(pref.rotate_ccw)
      return y >= alloc.height - pref.zone_size && y < alloc.height;
    else
      return y >= 0 && y < pref.zone_size;
  } else
    return x >= 0 && x < pref.zone_size;
}

static gboolean
is_in_right_zone(gint x, gint y, const GtkAllocation alloc)
{
  if(rotated) {
    if(pref.rotate_ccw)
      return y >= 0 && y < pref.zone_size;
    else
      return y >= alloc.height - pref.zone_size && y < alloc.height;
  } else
    return x >= alloc.width - pref.zone_size && x < alloc.width;
}


void
main_canvas_buttonpressed(GdkEventButton *event)
{
  GtkWidget *w;
  gint x, y;
  int processed = 0;
  int nr_pages;

  if (event->type != GDK_BUTTON_PRESS) return;

  nr_pages = image_cache_length();

  switch (event->button) {
  case 1:				// Left button
    /* check if we're in one of the zones */
    if(fullscreen) {
      x = event->x;
      y = event->y;
      if(is_in_left_zone(x, y, GTK_WIDGET(book_canvas)->allocation)) {
	previous_page(0, TRUE);
	processed = 1;
      } else if(is_in_right_zone(x, y, GTK_WIDGET(book_canvas)->allocation)) {
	next_page(0, TRUE);
	processed = 1;
      }
    }
    if(!processed && !gtk_image_view_get_fitting(book_canvas)) {
      start_x = event->x;
      start_y = event->y;
    }
    break;
  case 3:
    if(nr_pages > 0) {
      w = create_help_popup();
      gtk_widget_show_all(w);
      gtk_menu_popup(GTK_MENU(w), NULL, NULL, NULL, NULL,
		     event->button, event->time);
    }
    break;
  }
}

void
main_canvas_buttonreleased(GdkEventButton* event)
{
  start_x = -1.0;
  start_y = -1.0;
}

void
main_canvas_motion(GdkEventMotion* event)
{
  gdouble delta_x, delta_y;
  GdkRectangle viewport;
  int offset_x, offset_y, dummy_x, dummy_y;
  GdkModifierType dummy_mods;
  gdouble current_zoom;
  if(start_x == -1.0 || (event->state & GDK_BUTTON1_MASK) == 0)
    goto out;
  current_zoom = gtk_image_view_get_zoom(book_canvas);
  if(current_zoom <= 0.0)
    goto out;
  current_zoom = 1.0;
  
  delta_x = (event->x - start_x) / current_zoom;
  delta_y = (event->y - start_y) / current_zoom;
  if(fabs(delta_x) < 2 && fabs(delta_y) < 2)
    goto out;
  gtk_image_view_get_viewport(book_canvas, &viewport);
  offset_x = viewport.x - delta_x;
  offset_y = viewport.y - delta_y;
  
  gtk_image_view_set_offset (book_canvas, offset_x, offset_y, FALSE);
  start_x = event->x;
  start_y = event->y;
  
 out:
  gdk_window_get_pointer(GTK_WIDGET(book_canvas)->window, 
			 &dummy_x, 
			 &dummy_y, 
			 &dummy_mods);
}

void
about_box(GtkWidget* main_window)
{
  static char* author_keys[] = {
    N_("John Coppens (cbrPager code) [http://www.jcoppens.com/]"),
    N_("Benoit Goudreault-Emond (Maemo port) <bgoudreaultemond@gmail.com>"),
    N_("Bjorn Lindqvist (gtkimageview code) <bjourne@gmail.com>"),
    N_("Ricardo Veguilla <vegilla@hpcf.upr.edu>"),
    N_("roy wood <roy.wood@gmail.com>"),
    N_("Emmanuel Seyman <seyman@wanadoo.fr>"),
    N_("Bryan Brown <cmptrsmrt@comcast.net>"),
    N_("James Reeves <jreeves@monkeyengines.co.uk>"),
    N_("Seth Cohn <sethcohn@gmail.com>"),
    N_("John Paul Lorenti <jplorenti@users.sourceforge.net>"),
    N_("Jason Oliveira <mushroomblue@elazulspad.net>"),
    N_("Kristian Andre Gallis <kristaga@student.geo.uio.no>"),
    N_("Alex <aderkach@gmail.com>"),
    N_("Christian Troger <ctroeger@gmx.net>"),
    N_("Richard Burton <richardaburton@hotmail.com>"),
    N_("Leif Sandstede <leif@Sandstede.de>"),
    N_("Ilja Pyykkonen <ilja@pyykkonen.net>"),
    N_("Avuton Olrich <avuton@gmail.com>"),
    N_("Mamoru Tasaka <mtasaka@ioa.s.u-tokyo.ac.jp>"),
    NULL
  };
  static char** authors = NULL;
  static const char* license = NULL;
  if(!license) {
    license = _("cbrPager is free software; you can redistribute it and/or "
		"modify it under the terms of the GNU General Public Licence as "
		"published by the Free Software Foundation; either version 2 of the "
		"Licence, or (at your option) any later version.\n"
		"\n"
		"cbrPager is distributed in the hope that it will be useful, "
		"but WITHOUT ANY WARRANTY; without even the implied warranty of "
		"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU "
		"General Public Licence for more details.\n"
		"\n"
		"You should have received a copy of the GNU General Public Licence "
		"along with cbrPager; if not, write to the Free Software "
		"Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, "
		"MA  02110-1301  USA");
  }
  if(!authors) {
    char** iterator = NULL;
    for(iterator = author_keys; *iterator; ++iterator) {
      *iterator = _(*iterator);
    }
    authors = author_keys;
  }
  gtk_show_about_dialog(GTK_WINDOW(main_window),
			"name", "cbrPager",
			"version", prog_version,
			"copyright", _("Copyright (c) 2008 John Coppens, Benoit Goudreault-Emond and others.\nDistributed under the GNU General Public License, version 2."),
			"comments", _("A fast comic book archive (cbr, cbz, cb7) viewer.  This is the Maemo port of cbrPager."),
			"license", license,
			"wrap-license", TRUE,
			"authors", authors,
			"website", "http://cbrpager.sourceforge.net/",
			NULL);
}

void
goto_page()
{
  int nr_pages = image_cache_length();
  int result = invoke_goto_page_dialog(MainWindow, 
				       page_nr, 
				       nr_pages - pref.nrpages);
  if(result >= 0) {
    page_nr = result;
    show_page(result);
  }
}

void
toggle_keymode(void)
{
  if(toggle_accelerator_mode())
    info_dialog(_("Special mode ON"));
  else
    info_dialog(_("Special mode OFF"));
}

void
toggle_rotation_direction(void)
{
  if(rotated) {
    gtk_image_view_rotate(book_canvas, 180);
  }
}

void
scroll_down(void)
{
  g_signal_emit_by_name(book_canvas,
			"scroll",
			GTK_SCROLL_NONE,
			GTK_SCROLL_STEP_DOWN);
}

void
scroll_up(void)
{
  g_signal_emit_by_name(book_canvas,
			"scroll",
			GTK_SCROLL_NONE,
			GTK_SCROLL_STEP_UP);
}

void
scroll_right(void)
{
  g_signal_emit_by_name(book_canvas,
			"scroll",
			GTK_SCROLL_STEP_RIGHT,
			GTK_SCROLL_NONE);
}

void
scroll_left(void)
{
  g_signal_emit_by_name(book_canvas,
			"scroll",
			GTK_SCROLL_STEP_LEFT,
			GTK_SCROLL_NONE);
}

void
page_down(void)
{
  GtkScrollType h, v;
  if(rotated) {
    if(pref.rotate_ccw)
      h = GTK_SCROLL_PAGE_RIGHT;
    else
      h = GTK_SCROLL_PAGE_LEFT;
    v = GTK_SCROLL_NONE;
  } else {
    h = GTK_SCROLL_NONE;
    v = GTK_SCROLL_PAGE_DOWN;
  }
  g_signal_emit_by_name(book_canvas,
			"scroll",
			h,
			v);
}

void 
page_up(void)
{
  GtkScrollType h, v;
  if(rotated) {
    if(pref.rotate_ccw)
      h = GTK_SCROLL_PAGE_LEFT;
    else
      h = GTK_SCROLL_PAGE_RIGHT;
    v = GTK_SCROLL_NONE;
  } else {
    h = GTK_SCROLL_NONE;
    v = GTK_SCROLL_PAGE_UP;
  }
  g_signal_emit_by_name(book_canvas,
			"scroll",
			h,
			v);
}
