// case - file manager for N900
// Copyright (C) 2010 Lukas Hrazky <lukkash@email.cz>
// 
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.


#include "style.h"

#include <QPainter>

#include "progressbar.h"

#include <iostream>


void Style::drawControl(ControlElement element,
    const QStyleOption *opt,
    QPainter *p,
    const QWidget *widget) const
{
    switch (element) {
    case CE_ProgressBarGroove:
        if (const QStyleOptionProgressBar *pb = qstyleoption_cast<const QStyleOptionProgressBar *>(opt)) {
            p->setPen(opt->palette.color(QPalette::ButtonText).darker());
            p->drawRoundedRect(pb->rect.adjusted(0, 0, -1, -1), 4, 4);
        }
        break;

    case CE_ProgressBarLabel:
        if (const QStyleOptionProgressBar *pb = qstyleoption_cast<const QStyleOptionProgressBar *>(opt)) {
            QRect rect = pb->rect;
            int maximum = pb->maximum;
            int minimum = pb->minimum;
            int value = pb->progress;

            const ProgressBar *w = qobject_cast<const ProgressBar*>(widget);

            float perc = (maximum - minimum) ?  static_cast<float>(value - minimum) / (maximum - minimum) : 0;

            rect.adjust(2, 2, -2, -2);
            QRect filledRect = rect, emptyRect = rect;
            filledRect.setWidth(filledRect.width() * perc);
            emptyRect.setX(emptyRect.x() + filledRect.width());

            rect.adjust(3, 0, -3, 0);
            QPoint iconPoint;
            QRect topRect = rect, mainTextRect, bottomLeftRect = rect, bottomRightRect;
            topRect.setHeight(topRect.height() / 2);
            bottomLeftRect.setY(bottomLeftRect.y() + topRect.height());

            int topRightWidth = p->fontMetrics().size(Qt::TextSingleLine, pb->text).width();
            QRect pauseRect1(topRect.x() + topRect.width() - topRightWidth,
                topRect.y() + topRect.height() / 2, 0, 0), pauseRect2;

            if (w->paused) {
                pauseRect1.adjust(-24, -7, -19, 7);
                pauseRect2 = pauseRect1;
                pauseRect2.adjust(10, 0, 10, 0);
            }

            if (w->fromText.size() || w->toText.size()) {
                iconPoint = bottomLeftRect.center() - QPoint(10, 10);

                mainTextRect = topRect;
                mainTextRect.setRight(pauseRect1.x() - 10);

                bottomRightRect = bottomLeftRect;
                bottomLeftRect.setWidth(bottomLeftRect.width() / 2 - 13);
                bottomRightRect.setX(bottomRightRect.x() + bottomLeftRect.width() + 26);
            } else {
                iconPoint = topRect.center() - QPoint(10, 8);

                mainTextRect = bottomLeftRect;
            }

            QString mainText = shortenTextPath(p, mainTextRect, w->mainText);
            QString bottomLeftText = shortenTextPath(p, bottomLeftRect, w->fromText);
            QString bottomRightText = shortenTextPath(p, bottomRightRect, w->toText);

            if (filledRect.width()) {
                p->setClipRect(filledRect);
                p->setPen(opt->palette.text().color());
                p->setBrush(opt->palette.text());

                p->drawPixmap(iconPoint, w->fgIcon);

                p->drawText(mainTextRect, Qt::AlignCenter | Qt::TextSingleLine, mainText);
                p->drawText(topRect, Qt::AlignRight | Qt::AlignVCenter | Qt::TextSingleLine, pb->text);
                p->drawText(bottomLeftRect, Qt::AlignRight | Qt::AlignVCenter | Qt::TextSingleLine, bottomLeftText);
                p->drawText(bottomRightRect, Qt::AlignLeft | Qt::AlignVCenter | Qt::TextSingleLine, bottomRightText);

                if (w->paused) {
                    p->drawRect(pauseRect1);
                    p->drawRect(pauseRect2);
                }
            }

            if (emptyRect.width()) {
                p->setClipRect(emptyRect);
                p->setPen(opt->palette.buttonText().color());
                p->setBrush(opt->palette.buttonText());

                p->drawPixmap(iconPoint, w->bgIcon);

                p->drawText(mainTextRect, Qt::AlignCenter | Qt::TextSingleLine, mainText);
                p->drawText(topRect, Qt::AlignRight | Qt::AlignVCenter | Qt::TextSingleLine, pb->text);
                p->drawText(bottomLeftRect, Qt::AlignRight | Qt::AlignVCenter | Qt::TextSingleLine, bottomLeftText);
                p->drawText(bottomRightRect, Qt::AlignLeft | Qt::AlignVCenter | Qt::TextSingleLine, bottomRightText);

                if (w->paused) {
                    p->drawRect(pauseRect1);
                    p->drawRect(pauseRect2);
                }
            }
        }
        break;

    case CE_ProgressBarContents:
        if (const QStyleOptionProgressBar *pb = qstyleoption_cast<const QStyleOptionProgressBar *>(opt)) {
            QRect rect = pb->rect;
            int maximum = pb->maximum;
            int minimum = pb->minimum;
            int value = pb->progress;

            QPalette pal = pb->palette;
            // Correct the highlight color if it is the same as the background
            if (pal.highlight() == pal.background())
                pal.setColor(QPalette::Highlight, pb->palette.color(QPalette::Active,
                                                                     QPalette::Highlight));

            float perc = (maximum - minimum) ?  static_cast<float>(value - minimum) / (maximum - minimum) : 0;

            QRect filledRect = rect;
            filledRect.adjust(2, 2, -2, -2);
            filledRect.setWidth(filledRect.width() * perc);

            QColor highlight = opt->palette.highlight().color();
            QColor lighter = highlight.lighter(120);
            QColor darker = highlight.darker(110);

            QGradientStops stops;
            stops << QGradientStop(0.00, lighter);
            stops << QGradientStop(0.15, darker);
            stops << QGradientStop(0.80, darker);
            stops << QGradientStop(1.00, lighter);
            QLinearGradient fillGrad(0, 0, 0, filledRect.height());
            fillGrad.setStops(stops);
            QBrush fillBrush(fillGrad);

            p->setPen(lighter);
            p->setBrush(fillBrush);
            p->drawRoundedRect(filledRect.adjusted(0, 0, -1, -1), 2, 2);
        }
        break;
        
    default:
        QMaemo5Style::drawControl(element, opt, p, widget);
    }
}


QString Style::shortenTextPath(QPainter *p, const QRect &r, QString s) const {
    QFontMetrics m = p->fontMetrics();
    int width = r.width();
    int start = -2, end = 1;

    while (m.size(Qt::TextSingleLine, s).width() > width && s.size() > 7) {
        if (end <= 0) {
            start = end + 7;
        } else {
            // first pass only
            if (start == -2) {
                start = s.lastIndexOf('/', -2);
                if (start <= 0) start += 4;
                end = s.lastIndexOf('/', start - 1);
            } else {
                start = end + 4;
                end = s.lastIndexOf('/', end - 1);
            }
        }

        s.replace(end + 1, start - end - 1, "...");
    }

    return s;
}
