// case - file manager for N900
// Copyright (C) 2010 Lukas Hrazky <lukkash@email.cz>
// 
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.


#include "progressbar.h"

#include <QtGui>

#include "fileoperator.h"

#include <iostream>


ProgressBar::ProgressBar(const QPixmap &icon, const QPixmap &inverseIcon) :
    bgIcon(icon),
    fgIcon(inverseIcon),
    paused(false),
    contextEvent(false),
    lastTimeUpdate(0),
    startTime(0)
{
    timeBuf[0] = 0;
    setMaximum(1);
    setValue(0);
    setMinimum(0);
    QFont barFont = font();
    barFont.setPointSize(12);
    setFont(barFont);
    setMinimumHeight(44);
    setFormat("");
    mainText = tr("gathering information...");
}


void ProgressBar::updateProgress(int val) {
    if (value() + val > maximum()) {
        std::cout << "WARNING: exceeding progressbar maximum (" << maximum() << ") by " << val << std::endl;
    }

    setValue(value() + val);

    time_t now = time(0);
    if (lastTimeUpdate < now) {
        lastTimeUpdate = now;

        time_t elapsed = now - startTime;
        time_t remaining = (time_t) ((float) elapsed / value() * (maximum() - value()));
        struct tm *ts = gmtime(&remaining);
        
        if (remaining == 0) {
            timeBuf[0] = 0;
        } else if (remaining < 60) {
            strftime(timeBuf, sizeof(timeBuf), "  %Ss", ts);
        } else if (remaining < 3600) {
            strftime(timeBuf, sizeof(timeBuf), "  %M:%S", ts);
        } else {
            strftime(timeBuf, sizeof(timeBuf), "  %H:%M:%S", ts);
        }
    }

    setFormat(QString("%p%") + timeBuf);
}


void ProgressBar::updateMainText(const QString &text) {
    mainText = text;
    if (fromText.size()) {
        mainText.remove(0, fromText.size() + 1);
    }
    repaint();
}


void ProgressBar::setBottomTexts(const QString &left, const QString &right) {
    fromText = left;
    toText = right;
}


void ProgressBar::setStartTime(time_t t) {
    startTime = t;
    lastTimeUpdate = time(0);
    updateProgress(0);
}


void ProgressBar::pause() {
    paused = true;
    repaint();
}


void ProgressBar::resume(time_t stallTime) {
    startTime += stallTime;
    paused = false;
    repaint();
}


void ProgressBar::showRemoveNotice() {
    toText = "<" + tr("deleting") + ">";
    repaint();
}


void ProgressBar::mouseReleaseEvent(QMouseEvent *) {
    if (!contextEvent) emit togglePauseOperation(this);
    contextEvent = false;
}


void ProgressBar::contextMenuEvent(QContextMenuEvent *) {
    contextEvent = true;
    emit abortOperation(this);
}
