// case - file manager for N900
// Copyright (C) 2010 Lukas Hrazky <lukkash@email.cz>
// 
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.


#ifndef FILEOPERATOR_H
#define FILEOPERATOR_H

#include <QWidget>
#include <QProgressBar>
#include <QFileInfo>
#include <QThread>
#include <QMutex>
#include <QWaitCondition>
#include <QDir>
#include <QMap>
#include <QSet>


class FileManipulatorThread;


class FileOperator : public QWidget {
    Q_OBJECT

public:
    // DONT_ASK_ONCE is a hackish way to avoid asking twice to overwrite the same directory when moving
    enum Response{NONE, ABORT, RETRY, IGNORE, KEEP, OVERWRITE, SKIP_DIR, ASK, DONT_ASK_ONCE};

    FileOperator(QWidget *parent = 0);

    static QString shortenPath(const QString &path);
    static QString unwindPath(const QString &path);

    void deleteFiles(const QFileInfoList &files);
    void copyFiles(const QFileInfoList &files, QDir &destination);
    void moveFiles(const QFileInfoList &files, QDir &destination);

public slots:
    void showErrorPrompt(FileManipulatorThread* manipulator,
        const QString &message,
        const QString &fileName,
        const int err);
    void showOverwritePrompt(FileManipulatorThread* manipulator,
        const QString &fileName,
        const bool dirOverDir);
    void showInputFilenamePrompt(FileManipulatorThread* manipulator,
        const QFileInfo &fileName,
        const bool dirOverDir);

    void remove(FileManipulatorThread* manipulator);
    void setBarSize(FileManipulatorThread* manipulator, unsigned int size);
    void updateProgress(FileManipulatorThread* manipulator, int value);

protected:
    void caterNewThread(FileManipulatorThread *thread);

    QList<FileManipulatorThread*> manipulatorList;
};


class FileManipulatorThread : public QThread {
    Q_OBJECT

public:
    explicit FileManipulatorThread(const QFileInfoList files, QDir dest = QDir());
    ~FileManipulatorThread();
    void setResponse(const FileOperator::Response response, const bool appyToAll = false, const int err = 0);

    void setText(int value);

    QProgressBar *progressBar;

    time_t startTime;
    QMutex mutex;
    QWaitCondition waitCond;
    // the new name entered from the overwrite dialog
    QString newNameFromDialog;

protected:
    void processFiles(const QFileInfoList &files);
    virtual void perform(const QFileInfo &file) = 0;

    bool remove(QString &fileName, const bool doUpdates = false);
    bool remove(const QFileInfoList &files, const bool doUpdates = false);
    bool remove(const QFileInfo &file, const bool doUpdates = false);

    void copy(const QFileInfo &file);

    unsigned int calculateFileSize(const QFileInfoList &files,
        const bool count = false,
        const bool addSize = false);

    QFileInfoList listDirFiles(const QString &dirPath);

    void setBarSize(unsigned int size);
    void updateProgress(int value);
    void updateFile(const QString &name);

    // files to process by the operation
    const QFileInfoList files;
    // destination for files - changes as the operation recurses into directories
    QDir dest;

    // responses from the dialog prompts (error and overwrite)
    FileOperator::Response response;
    FileOperator::Response overwriteAll;
    // a flag to abort the operation
    bool abort;
    // an array indicating whether to always ignore the error of index errno
    bool ignoreAll[256];

    // set of files that won't be deleted by the remove(...) functions
    // used when move(...) would not overwrite target file to ensure the source file doesn't get deleted
    QSet<QString> removeExcludeFiles;

    // A map of file paths to their size. Not the actual size, but what is calculated for the
    // purpose of the progressbar for the given operation. So either fileSize/BLOCK_SIZE or simply
    // 1 for a file and file count for dirs (or both for copy&delete)
    QMap<QString, qint64> fileSizeMap;

    // the name of the file thats being processed (for progressBar) and the text of the progressBar (the format)
    QString fileName, barText;
    // stamp of the last ETA recalculation - done every second
    time_t lastTimeUpdate;
    char timeBuf[10];
    // progress information of the bar and for the current file
    unsigned int barSize, barValue, fileSize, fileValue;

signals:
    void showErrorPrompt(FileManipulatorThread*, const QString&, const QString&, const int);
    void showOverwritePrompt(FileManipulatorThread*, const QString&, const bool);
    void showInputFilenamePrompt(FileManipulatorThread*, const QFileInfo&, const bool);
    void finished(FileManipulatorThread*);
    void setBarSize(FileManipulatorThread*, unsigned int);
    void updateProgress(FileManipulatorThread*, int);
};


class DeleteThread : public FileManipulatorThread {
    Q_OBJECT

public:
    explicit DeleteThread(const QFileInfoList &files);

protected:
    void run();
    virtual void perform(const QFileInfo &file);
};


class CopyThread : public FileManipulatorThread {
    Q_OBJECT

public:
    explicit CopyThread(const QFileInfoList &files, QDir &dest);

protected:
    void run();
    virtual void perform(const QFileInfo &file);
};


class MoveThread : public FileManipulatorThread {
    Q_OBJECT

public:
    explicit MoveThread(const QFileInfoList &files, QDir &dest);

protected:
    void run();
    virtual void perform(const QFileInfo &file);
    void rename(const QFileInfoList &files, const QDir &dest);
};


#endif // FILEOPERATOR_H
