// case - file manager for N900
// Copyright (C) 2010 Lukas Hrazky <lukkash@email.cz>
// 
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.


#include "fileoperator.h"

#include <QtGui>
#include <QDir>
#include <QMessageBox>
#include <QHBoxLayout>
#include <QChar>

#include <math.h>
#include <errno.h>
#include <iostream>


#define BLOCK_SIZE 524288


#define SHOW_ERROR_PROMPT(promptString, fileName)                                           \
    response = FileOperator::NONE;                                                          \
    if (ignoreAll[errno]) {                                                                 \
        response = FileOperator::IGNORE;                                                    \
    } else {                                                                                \
        char buf[255];                                                                      \
        char *realBuf = strerror_r(errno, buf, 255);                                        \
        emit showErrorPrompt(this, promptString + " " + realBuf + ".", fileName, errno);    \
        waitCond.wait(&mutex);                                                              \
    }


#define ERROR_PROMPT(operation, promptString, fileName)                                     \
{                                                                                           \
    response = FileOperator::NONE;                                                          \
    while (!abort && operation) {                                                           \
        SHOW_ERROR_PROMPT(promptString, fileName)                                           \
        if (response == FileOperator::IGNORE) {                                             \
            break;                                                                          \
        }                                                                                   \
    }                                                                                       \
}


#define ERROR_PROMPT_XP(operation, promptString, fileName, onIgnore, quitCmd)               \
{                                                                                           \
    ERROR_PROMPT(operation, promptString, fileName)                                         \
    if (abort || response == FileOperator::IGNORE) {                                        \
        if (!abort) onIgnore;                                                               \
        quitCmd;                                                                            \
    }                                                                                       \
}


#define OVERWRITE_PROMPT(file, newFile)                                                     \
{                                                                                           \
    response = FileOperator::NONE;                                                          \
                                                                                            \
    if (newFile.exists()) {                                                                 \
        if (overwriteAll != FileOperator::NONE) {                                           \
            response = overwriteAll;                                                        \
        } else {                                                                            \
            bool dirOverDir = false;                                                        \
            if (newFile.isDir() && file.isDir()) dirOverDir = true;                         \
            emit showOverwritePrompt(this, newFile.absoluteFilePath(), dirOverDir);         \
            waitCond.wait(&mutex);                                                          \
        }                                                                                   \
    }                                                                                       \
}


FileOperator::FileOperator(QWidget *parent) : QWidget(parent) {
    QHBoxLayout *layout = new QHBoxLayout;
    layout->setContentsMargins(0, 0, 0, 0);
    layout->setSpacing(0);
    setLayout(layout);
}


QString FileOperator::shortenPath(const QString &path) {
    QString homePath = QFSFileEngine::homePath();
    QString result = path;
    if (path.indexOf(homePath, 0) == 0) {
        result.replace(0, homePath.size(), "~");
    }

    return result;
}


void FileOperator::deleteFiles(const QFileInfoList &files) {
    QString title, desc;
    if (files.size() == 1) {
        title = tr("Delete file");
        desc = tr("Are you sure you want to delete %1?")
            .arg(FileOperator::shortenPath(files[0].absoluteFilePath()));
    } else {
        title = tr("Delete files");
        desc = tr("You are about to delete %1 files. Are you sure you want to continue?").arg(files.size());
    }

    int confirm = QMessageBox::warning(
        0,
        title,
        desc,
        QMessageBox::Yes,
        QMessageBox::No
    );

    if(confirm == QMessageBox::Yes) {
        caterNewThread(new DeleteThread(files));
    }
}


void FileOperator::copyFiles(const QFileInfoList &files, QDir &destination) {
    QString title, desc;
    if (files.size() == 1) {
        title = tr("Copy file");
        desc = tr("Are you sure you want to copy %1 to %2?")
            .arg(FileOperator::shortenPath(files[0].absoluteFilePath()))
            .arg(FileOperator::shortenPath(destination.absolutePath()));
    } else {
        title = tr("Copy files");
        desc = tr("You are about to copy %1 files to %2. Are you sure you want to continue?")
            .arg(files.size()).arg(FileOperator::shortenPath(destination.absolutePath()));
    }

    int confirm = QMessageBox::warning(
        0,
        title,
        desc,
        QMessageBox::Yes,
        QMessageBox::No
    );

    if(confirm == QMessageBox::Yes) {
        caterNewThread(new CopyThread(files, destination));
    }
}


void FileOperator::moveFiles(const QFileInfoList &files, QDir &destination) {
    // for move we don't wanna move to the same dir
    if (files[0].absolutePath() == destination.absolutePath()) return;

    QString title, desc;
    if (files.size() == 1) {
        title = tr("Move file");
        desc = tr("Are you sure you want to move %1 to %2?")
            .arg(FileOperator::shortenPath(files[0].absoluteFilePath()))
            .arg(FileOperator::shortenPath(destination.absolutePath()));
    } else {
        title = tr("Move files");
        desc = tr("You are about to move %1 files to %2. Are you sure you want to continue?")
            .arg(files.size()).arg(FileOperator::shortenPath(destination.absolutePath()));
    }

    int confirm = QMessageBox::warning(
        0,
        title,
        desc,
        QMessageBox::Yes,
        QMessageBox::No
    );

    if(confirm == QMessageBox::Yes) {
        caterNewThread(new MoveThread(files, destination));
    }
}


void FileOperator::showErrorPrompt(FileManipulatorThread* manipulator,
    const QString &message,
    const QString &fileName,
    const int err)
{
    QMessageBox msgBox;
    msgBox.addButton(QMessageBox::Cancel);
    QAbstractButton *abortButton = msgBox.addButton(tr("Abort"), QMessageBox::DestructiveRole);
    QAbstractButton *retryButton = msgBox.addButton(QMessageBox::Retry);
    QAbstractButton *ignoreButton = msgBox.addButton(QMessageBox::Ignore);
    QAbstractButton *ignoreAllButton = msgBox.addButton(tr("Ignore All"), QMessageBox::AcceptRole);
    msgBox.setText(message.arg(FileOperator::shortenPath(fileName)));

    msgBox.exec();

    if (msgBox.clickedButton() == abortButton) {
        manipulator->setResponse(ABORT);
    } else if (msgBox.clickedButton() == retryButton) {
        manipulator->setResponse(RETRY);
    } else if (msgBox.clickedButton() == ignoreButton) {
        manipulator->setResponse(IGNORE);
    } else if (msgBox.clickedButton() == ignoreAllButton) {
        manipulator->setResponse(IGNORE, true, err);
    }
}


void FileOperator::showOverwritePrompt(
    FileManipulatorThread* manipulator,
    const QString &fileName,
    const bool dirOverDir)
{
    QMessageBox msgBox;
    msgBox.addButton(QMessageBox::Cancel);
    QAbstractButton *yesButton = msgBox.addButton(QMessageBox::Yes);
    QAbstractButton *yesToAllButton = msgBox.addButton(QMessageBox::YesToAll);
    QAbstractButton *noButton = msgBox.addButton(QMessageBox::No);
    QAbstractButton *noToAllButton = msgBox.addButton(QMessageBox::NoToAll);
    QAbstractButton *abortButton = msgBox.addButton(tr("Abort"), QMessageBox::DestructiveRole);
    QAbstractButton *askButton = 0;

    if (dirOverDir) {
        msgBox.setText(tr("Directory %1 already exists. Overwrite the files inside?")
            .arg(FileOperator::shortenPath(fileName)));
        askButton = msgBox.addButton(tr("Ask"), QMessageBox::AcceptRole);
    } else {
        msgBox.setText(tr("File %1 already exists. Overwrite?").arg(FileOperator::shortenPath(fileName)));
    }

    msgBox.exec();

    if (msgBox.clickedButton() == abortButton) {
        manipulator->setResponse(ABORT);
    } else if (msgBox.clickedButton() == yesButton) {
        manipulator->setResponse(OVERWRITE);
    } else if (msgBox.clickedButton() == yesToAllButton) {
        manipulator->setResponse(OVERWRITE, true);
    } else if (msgBox.clickedButton() == noButton) {
        manipulator->setResponse(KEEP);
    } else if (msgBox.clickedButton() == noToAllButton) {
        manipulator->setResponse(KEEP, true);
    } else if (msgBox.clickedButton() == askButton) {
        manipulator->setResponse(NONE, true);
    }
}


void FileOperator::remove(FileManipulatorThread* manipulator) {
    manipulator->wait();
    layout()->removeWidget(manipulator->progressBar);
    manipulatorList.removeAll(manipulator);
    delete manipulator;
}


void FileOperator::setBarSize(FileManipulatorThread* manipulator, unsigned int size) {
    if (!manipulator->progressBar->maximum()) {
        manipulator->startTime = time(0);
    }
    manipulator->progressBar->setMinimum(0);
    manipulator->progressBar->setMaximum(size);
}


void FileOperator::updateProgress(FileManipulatorThread* manipulator, int value) {
    manipulator->setText(value);
}


void FileOperator::caterNewThread(FileManipulatorThread *thread) {
    manipulatorList.append(thread);

    connect(thread, SIGNAL(showErrorPrompt(FileManipulatorThread*, const QString&, const QString&, const int)),
        this, SLOT(showErrorPrompt(FileManipulatorThread*, const QString&, const QString&, const int)));
    connect(thread, SIGNAL(showOverwritePrompt(FileManipulatorThread*, const QString&, bool)),
        this, SLOT(showOverwritePrompt(FileManipulatorThread*, const QString&, bool)));
    connect(thread, SIGNAL(finished(FileManipulatorThread*)),
        this, SLOT(remove(FileManipulatorThread*)));
    connect(thread, SIGNAL(setBarSize(FileManipulatorThread*, unsigned int)),
        this, SLOT(setBarSize(FileManipulatorThread*, unsigned int)));
    connect(thread, SIGNAL(updateProgress(FileManipulatorThread*, int)),
        this, SLOT(updateProgress(FileManipulatorThread*, int)));

    thread->progressBar->setValue(0);

    layout()->addWidget(thread->progressBar);
    thread->start(QThread::LowestPriority);
}


FileManipulatorThread::FileManipulatorThread(const QFileInfoList files, QDir dest) :
    progressBar(new QProgressBar()),
    startTime(0),
    files(files),
    dest(dest),
    response(FileOperator::NONE),
    overwriteAll(FileOperator::NONE),
    abort(false),
    lastTimeUpdate(0),
    barSize(0),
    barValue(0),
    fileSize(0),
    fileValue(0)
{
    memset(ignoreAll, false, sizeof(ignoreAll));
    progressBar->setMaximum(0);
    QFont barFont = progressBar->font();
    barFont.setPointSize(12);
    progressBar->setFont(barFont);
    progressBar->setFormat(tr("Gathering information..."));
    progressBar->setMinimumHeight(44);
    progressBar->setStyle(new QPlastiqueStyle);
    //progressBar->setStyle(new QMotifStyle);
}


FileManipulatorThread::~FileManipulatorThread() {
    if (progressBar->value() < progressBar->maximum()) {
        std::cout << "WARNING: deleting a progressbar which's value " << progressBar->value() <<
            " has not reached maximum of " << progressBar->maximum() << std::endl;
    }
    delete progressBar;
}


void FileManipulatorThread::setResponse(
    const FileOperator::Response response,
    const bool applyToAll,
    const int err)
{
    mutex.lock();

    this->response = response;

    if (applyToAll) {
        if (response == FileOperator::KEEP
            || response == FileOperator::OVERWRITE
            || response == FileOperator::NONE)
        {
            overwriteAll = response;
        }

        if (response == FileOperator::IGNORE) {
            ignoreAll[err] = true;
        }
    }

    if (response == FileOperator::ABORT) abort = true;

    mutex.unlock();
    waitCond.wakeAll();
}


void FileManipulatorThread::processFiles(const QFileInfoList &files) {
    for (QFileInfoList::const_iterator it = files.begin(); it != files.end(); ++it) {
        perform(*it);
        if (abort) break;
    }
}


bool FileManipulatorThread::remove(QString &fileName, const bool doUpdates) {
    return remove(QFileInfo(fileName), doUpdates);
}


bool FileManipulatorThread::remove(const QFileInfoList &files, const bool doUpdates) {
    bool res = true;
    for (QFileInfoList::const_iterator it = files.begin(); it != files.end(); ++it) {
        if (!remove(*it, doUpdates)) res = false;
        if (abort) break;
    }
    return res;
}


bool FileManipulatorThread::remove(const QFileInfo &file, const bool doUpdates) {
    std::cout << "DELETING " << file.absoluteFilePath().toStdString() << std::endl;

    QString path = file.absoluteFilePath();
    QFSFileEngine engine(path);

    if (doUpdates) updateFile(path);

    if (file.isDir()) {
        if (!remove(listDirFiles(path), doUpdates)) return false;

        if (!listDirFiles(path).size()) {
            ERROR_PROMPT(!engine.rmdir(path, false), tr("Error deleting directory %1."), path)
        }
    } else {
        ERROR_PROMPT(!engine.remove(), tr("Error deleting file %1."), path)
    }

    if (!abort && doUpdates) updateProgress(1);

    if (abort || response == FileOperator::IGNORE) return false;
    return true;
}


void FileManipulatorThread::copy(const QFileInfo &file) {
    std::cout << "COPYING " << file.absoluteFilePath().toStdString()
        << " to " << dest.absolutePath().toStdString() << std::endl;

    QString path(file.absoluteFilePath());
    QString newPath(dest.absolutePath() + "/" + file.fileName());
    QFSFileEngine engine(path);
    QFSFileEngine newEngine(newPath);
    QFileInfo newFile(newPath);

    updateFile(path);

    // hack to prevent asking about the same file if we already asked in the rename(...) function
    if (overwriteAll == FileOperator::DONT_ASK_ONCE) {
        overwriteAll = FileOperator::NONE;
    } else {
        OVERWRITE_PROMPT(file, newFile)
    }

    if (abort) return;

    if (response == FileOperator::KEEP) {
        updateProgress(fileSizeMap[path]);
        return;
    }

    if (file.isDir()) {
        FileOperator::Response overwriteResponse = response;

        if (newFile.exists() && !newFile.isDir()) {
            if(!remove(newPath)) {
                updateProgress(fileSizeMap[path]);
                return;
            }
            newFile = QFileInfo(newPath);
        }

        if (!newFile.exists()) {
            ERROR_PROMPT_XP(!engine.mkdir(newPath, false),
                tr("Error creating directory %1."), newPath,
                updateProgress(fileSizeMap[path]),
                return)
        }

        updateProgress(1);
        
        QDir destBackup = dest;
        dest = newPath;

        FileOperator::Response tmpResp = overwriteAll;
        overwriteAll = overwriteResponse;

        processFiles(listDirFiles(path));

        overwriteAll = tmpResp;

        ERROR_PROMPT(!newEngine.setPermissions(file.permissions()),
            tr("Error setting permissions for directory %1."), newPath)

        if (abort) return;

        dest = destBackup;
    } else {
        ERROR_PROMPT_XP(engine.isSequential(),
            tr("Cannot copy sequential file %1."), path,
            updateProgress(fileSizeMap[path]),
            return)

        if (newFile.exists() && newFile.isDir()) {
            ERROR_PROMPT_XP(!remove(newPath),
                tr("Cannot replace directory %1 due to previous errors."), newPath,
                updateProgress(fileSizeMap[path]),
                return)
        }

        ERROR_PROMPT_XP(!engine.open(QIODevice::ReadOnly),
            tr("Error reading file %1."), path,
            updateProgress(fileSizeMap[path]),
            return)

        bool ignore = false;
        while (!abort && !ignore) {
            engine.seek(0);

            ERROR_PROMPT(!newEngine.open(QIODevice::WriteOnly | QIODevice::Truncate),
                tr("Error writing file %1."), newPath)

            if (abort || response == FileOperator::IGNORE) {
                if (response == FileOperator::IGNORE) {
                    updateProgress(fileSizeMap[path] - fileValue);
                    ignore = true;
                }
                break;
            }

            bool error = false;
            char block[BLOCK_SIZE];
            qint64 bytes;
            while ((bytes = engine.read(block, sizeof(block))) > 0) {
                if (bytes == -1 || bytes != newEngine.write(block, bytes)) {
                    if (bytes == -1) {
                        SHOW_ERROR_PROMPT(tr("Error while reading from file %1."), path);
                    } else {
                        SHOW_ERROR_PROMPT(tr("Error while writing to file %1."), newPath);
                    }

                    if (!abort) {
                        if (response == FileOperator::IGNORE) {
                            updateProgress(fileSizeMap[path] - fileValue);
                            ignore = true;
                        } else {
                            updateProgress(-fileValue);
                        }
                    }
                    error = true;
                    break;
                }

                updateProgress(1);
            }

            if (!error) break;
        }

        engine.close();
        newEngine.close();

        if (abort || ignore) {
            newEngine.remove();
        } else {
            ERROR_PROMPT(!newEngine.setPermissions(file.permissions()),
                tr("Error setting permissions for file %1."), newPath)
        }
    }
}


unsigned int FileManipulatorThread::calculateFileSize(const QFileInfoList &files,
    const bool count,
    const bool addSize)
{
    unsigned int res = 0;

    for (QFileInfoList::const_iterator it = files.begin(); it != files.end(); ++it) {
        unsigned int size = 0;

        if (it->isDir()) {
            size += calculateFileSize(listDirFiles(it->absoluteFilePath()), count, addSize);
        }

        if (addSize) {
            if (it->isDir()) {
                ++size;
            } else {
                size += ceil(static_cast<float>(it->size()) / BLOCK_SIZE);
            }
            fileSizeMap[it->absoluteFilePath()] = size;
        }

        if (count) {
            ++size;
        }

        res += size;
    }

    return res;
}


QFileInfoList FileManipulatorThread::listDirFiles(const QString &dirPath) {
    QDir dir = dirPath;
    return dir.entryInfoList(QDir::NoDotAndDotDot | QDir::AllEntries | QDir::System | QDir::Hidden);
}


void FileManipulatorThread::setBarSize(unsigned int size) {
    barSize = size;
    emit setBarSize(this, size);
}


void FileManipulatorThread::updateProgress(int value) {
    barValue += value;
    fileValue += value;
    emit updateProgress(this, value);
}


void FileManipulatorThread::updateFile(const QString &name) {
    fileValue = 0;
    fileName = FileOperator::shortenPath(name);
    emit updateProgress(this, 0);
}


void FileManipulatorThread::setText(int value) {
    if (progressBar->value() + value > progressBar->maximum()) {
        std::cout << "WARNING: exceeding progressbar maximum (" << progressBar->maximum()
            << ") by " << value << std::endl;
    }
 
    time_t now = time(0);
    if (lastTimeUpdate < now) {
        lastTimeUpdate = now;

        time_t elapsed = now - startTime;
        time_t remaining = (time_t) ((float) elapsed / barValue * (barSize - barValue));
        struct tm *ts = gmtime(&remaining);
        
        if (remaining < 60) {
            strftime(timeBuf, sizeof(timeBuf), "%Ss", ts);
        } else if (remaining < 3600) {
            strftime(timeBuf, sizeof(timeBuf), "%M:%S", ts);
        } else {
            strftime(timeBuf, sizeof(timeBuf), "%H:%M:%S", ts);
        }
    }

    progressBar->setFormat(barText.arg(fileName) + "\n%p%   ETA " + timeBuf);
    progressBar->setValue(progressBar->value() + value);
}


DeleteThread::DeleteThread(const QFileInfoList &files) : FileManipulatorThread(files) {
    barText = tr("deleting %1");
}


void DeleteThread::run() {
    mutex.lock();

    setBarSize(calculateFileSize(files, true));

    processFiles(files);

    sleep(0.5);
    emit finished(this);
}


void DeleteThread::perform(const QFileInfo &file) {
    remove(file, true);
}


CopyThread::CopyThread(const QFileInfoList &files, QDir &dest) : FileManipulatorThread(files, dest) {
    barText = tr("copying %1");
}


void CopyThread::run() {
    mutex.lock();

    setBarSize(calculateFileSize(files, false, true));

    processFiles(files);

    sleep(0.5);
    emit finished(this);
}


void CopyThread::perform(const QFileInfo &file) {
    copy(file);
}


MoveThread::MoveThread(const QFileInfoList &files, QDir &dest) : FileManipulatorThread(files, dest) {
    barText = tr("moving %1");
}


void MoveThread::run() {
    mutex.lock();

    rename(files, dest);

    sleep(0.5);
    emit finished(this);
}


void MoveThread::rename(const QFileInfoList &files, const QDir &dest) {
    setBarSize(barSize + files.size());

    for (int i = 0; i < files.size(); ++i) {
        QString path = files[i].absoluteFilePath();
        QFSFileEngine engine(path);
        QString newPath = dest.absolutePath() + "/" + files[i].fileName();

        updateFile(path);

        OVERWRITE_PROMPT(files[i], QFileInfo(newPath))

        if (response == FileOperator::KEEP) {
            // TODO lets not remove the source for now, I'm not sure what is correct behavior
            // remove(path);
            if (abort) break;
            updateProgress(1);
            continue;
        }

        while (!abort && !engine.rename(newPath)) {
            // source and target are on different partitions
            // this should happen on the first file, unless some are skipped by overwrite prompt
            // we calculate the actual file sizes, because from now on copy & remove takes over
            if (errno == EXDEV) {
                overwriteAll = response;
                // hack: we already checked the first file we are sending to processFiles(...)
                // so we don't want to ask about this one again
                if (overwriteAll == FileOperator::NONE) overwriteAll = FileOperator::DONT_ASK_ONCE;

                QFileInfoList remainingFiles = files.mid(i);

                setBarSize(barValue + calculateFileSize(remainingFiles, true, true));

                processFiles(remainingFiles);

                barText = tr("deleting %1");

                remove(remainingFiles, true);

                // just to quit the loops, we are done
                abort = true;
            // the target is nonempty dir. lets call this recursively and rename the contents one by one
            } else if (errno == ENOTEMPTY || errno == EEXIST) {
                FileOperator::Response tmpResp = overwriteAll;
                overwriteAll = response;

                rename(listDirFiles(path), QDir(newPath));
                if (abort) break;

                overwriteAll = tmpResp;

                ERROR_PROMPT(!engine.rmdir(path, false), tr("Error deleting directory %1."), path)

                break;
            // source and target are nonmatching types(file and dir)
            // remove the target and let it loop once again
            } else if (errno == ENOTDIR || errno == EISDIR) {
                if (!remove(newPath)) break;
            } else {
                SHOW_ERROR_PROMPT(tr("Error moving %1."), path)

                if (response == FileOperator::IGNORE) {
                    break;
                }
            }
        }
            
        if (abort) break;
        updateProgress(1);
    }
}


void MoveThread::perform(const QFileInfo &file) {
    copy(file);
}
