// case - file manager for N900
// Copyright (C) 2010 Lukas Hrazky <lukkash@email.cz>
// 
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.


#include "filelist.h"

#include <QProcess>
#include <QUrl>

#include <hildon-mime.h>
#include <dbus/dbus.h>


FileList::FileList(QWidget *parent) :
    QListView(parent),
    fileSystemModel(new QFileSystemModel)
{
    setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    //setSelectionMode(QAbstractItemView::SingleSelection);
    setSelectionMode(QAbstractItemView::MultiSelection);

    setModel(fileSystemModel);

    fileSystemModel->setFilter(fileSystemModel->filter() | QDir::System);
    setRootIndex(fileSystemModel->setRootPath(QDir::homePath()));

    connect(this, SIGNAL(doubleClicked(QModelIndex)), this, SLOT(activateItem(QModelIndex)));
}


const QFileInfoList FileList::selection() const {
    QFileInfoList files;
    QModelIndexList l = selectionModel()->selectedIndexes();
    for (int i = 0; i < l.size(); ++i) {
        files.append(fileSystemModel->fileInfo(l[i]));
    }
    return files;
}


const QString FileList::path() const {
    return fileSystemModel->rootPath();
}


bool FileList::changePath(QString path) {
    QDir dir(fileSystemModel->rootPath());
    if (dir.cd(path)) {
        setRootIndex(fileSystemModel->setRootPath(dir.absolutePath()));
        clearSelection();
        emit pathChanged(fileSystemModel->rootPath());
        return true;
    }

    return false;
}


bool FileList::goUp() {
    return changePath("..");
}


void FileList::toggleShowHiddenFiles() {
    clearSelection();
    scrollToTop();
    fileSystemModel->setFilter(fileSystemModel->filter() ^ QDir::Hidden);
}


void FileList::activateItem(QModelIndex index) {
    const QFileInfo &file = fileSystemModel->fileInfo(index);

    if(file.isDir()) {
        changePath(file.absoluteFilePath());
        // hack: we reset it to MultiSelection again in the mousePressEvent
        // without this, the item under the cursor gets selected right after changing the directory
        setSelectionMode(QAbstractItemView::NoSelection);
    } else if(file.isExecutable()) {
        QProcess::startDetached(file.absoluteFilePath());
    } else {
        // TODO: find better solution for this, maybe get fixed in Qt
        DBusConnection* conn;
        conn = dbus_bus_get(DBUS_BUS_SESSION, 0);
        hildon_mime_open_file(conn, QUrl::fromLocalFile(file.absoluteFilePath()).toEncoded().constData());

        // Not working with maemo5. Uses hildon_uri_open function from
        // libhildonmime which should work, but all files opened in browser.
        //QDesktopServices::openUrl(QUrl::fromLocalFile(file.absoluteFilePath()));
    }
}


void FileList::mousePressEvent(QMouseEvent *event) {
    emit mousePressed();
    QListView::mousePressEvent(event);
    // need to reset the selection mode in case it was set to NoSelection in activateItem(...)
    setSelectionMode(QAbstractItemView::MultiSelection);
}
