# reposet.py - Set of repositories
#
#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{RepoBase} and L{RepositoriesSet}.
"""

import os, sys, math, glob

from common.carlog import DEBUG, ERROR
from common.carmanconfig import CarmanConfig

class RepoBase(object):
    """
    Represents a map repository used to download maptiles. This class contains
    all information about map repository.

    @cvar NAME: repository's name.
    @cvar CACHE_NAME: repository's cache name.
    @cvar TILE_WIDTH: map tile width used by the repository.
    @cvar TILE_HEIGHT: map tile height used by the repository.
    @cvar URI: repository's URI.
    @cvar ZOOM: zoom range supported by the repository.
    @cvar DIMENSION_OFFSET: repository's offset used on map tile location.
    @cvar WORLD_SIZE: repository's world size.
    @cvar WORLD_SIZE_360: C{WORLD_SIZE / 360}.
    @cvar EXTENSION: map tile images extension used by the repository.
    @cvar URBAN_ZOOM: zoom factor used when maps zoom is in 'urban zoom' type.
    @cvar CITY_ZOOM: tile size when maps zoom is in 'city zoom' type.
    @cvar STATE_ZOOM: tile size when maps zoom is in 'state zoom' type.
    @cvar AVERAGE_SIZE: average tile size.
    """
    NAME = "Dummy"
    CACHE_NAME = ".dummy"
    TILE_WIDTH = 256
    TILE_HEIGHT = 256
    URI = ""
    ZOOM = range(0, 18)
    DIMENSION_OFFSET = 0
    WORLD_SIZE = (1 << (ZOOM[-1] + DIMENSION_OFFSET)) * TILE_WIDTH
    WORLD_SIZE_360 = WORLD_SIZE / 360.0
    EXTENSION = '.png'
    URBAN_ZOOM = 2
    CITY_ZOOM = 6
    STATE_ZOOM = 12
    AVERAGE_SIZE = 10

    def build_uri(self, tile_x, tile_y, zoom):
        """
        Builds image tile address. This abstract method shall be implemented
        by each map service classes which heritages from this class.

        @type   tile_x: number
        @param  tile_x: Map tile M{X} coordinate.
        @type   tile_y: number
        @param  tile_y: Map tile M{Y} coordinate.
        @type   zoom: number
        @param  zoom: Map zoom.
        """
        pass

    def get_downloads_zoom(self):
        """
        Returns all download zoom types.

        @rtype: tuple
        @return: donwload zoom types: URBAN_ZOOM, CITY_ZOOM or STATE_ZOOM.
        """
        return self.URBAN_ZOOM, self.CITY_ZOOM, self.STATE_ZOOM

    def get_name(self):
        """
        Returns repository's name.

        @rtype: string
        @return: repository's name.
        """
        return self.NAME

    def get_cache_name(self):
        """
        Returns repository's cache name.

        @rtype: string
        @return: repository's cache name.
        """
        return self.CACHE_NAME

    def get_extension(self):
        """
        Returns repository's extension used for map tile images.

        @rtype: string
        @return: repository images extension.
        """
        return self.EXTENSION

    def get_tile_width(self):
        """
        Returns map tile width.

        @rtype: number
        @return: map tile width used.
        """
        return self.TILE_WIDTH

    def get_tile_height(self):
        """
        Returns tile height.

        @rtype: number
        @return: map tile height used.
        """
        return self.TILE_HEIGHT

    def get_zoom(self):
        """
        Returns repository zoom.

        @rtype: number
        @return: repository's zoom.
        """
        return self.ZOOM

    def get_dimension_offset(self):
        """
        Returns dimension offset used by the repository.

        @rtype: number
        @return: dimension offset.
        """
        return self.DIMENSION_OFFSET

    def get_average_size(self):
        """
        Returns map tile average size used by the repository.

        @rtype: number
        @return: map tile average size.
        """
        return self.AVERAGE_SIZE

    def xy_to_latlon(self, x, y):
        """
        Converts (x,y) coordinates from Evas views to latitude and longitude used
        by the repository.

        @type x: number
        @param x: x coordinate.
        @type y: number
        @param y: y coodinate.
        @rtype: tuple
        @return: tuple C{(lat,lon)}.
        """
        lon = x / self.WORLD_SIZE_360 - 180
        y = y - self.WORLD_SIZE / 2
        radius = self.WORLD_SIZE / (2 * math.pi)
        lat = (math.pi / 2) - 2 * math.atan(math.exp(-1.0 * y / radius))
        return -math.degrees(lat), lon

    def latlon_to_xy(self, lat, lon):
        """
        Converts (lat,lon) coordinates from map repository to (x,y) coordinates
        used by Carman to positionate the map.
        """
        if lon >= 180.0:
            x = self.WORLD_SIZE
        elif lon <= -180.0:
            x = 0
        else:
            x = int(round(self.WORLD_SIZE_360 * (lon + 180)))
        if lat >= 90.0:
            y = 0
        elif lat <= -90.0:
            y = self.WORLD_SIZE
        else:
            tmp = math.sin(math.radians(lat))
            tmp = 0.5 * math.log((1 + tmp) / (1 - tmp))
            tmp = (1 - tmp / math.pi) / 2.0
            y = int(round(tmp * self.WORLD_SIZE))
        return x, y


class RepositoriesSet(object):
    """
    Handles maps repositories plugins. This class is used to load/unload maps
    repositories services. Maps services are used to change the type of Map
    repository which Carman will use to download the map tiles.

    @ivar config: CarmanConfig object.
    @ivar path: Map repository dir path.
    """
    def __init__(self):
        self.path = []
        config = CarmanConfig()
        self.path.append(config.get_repository_plugins_path())
        self.path.append(config.get_user_repository_plugins_path())
        for path in self.path:
            path = os.path.dirname(path)
            if path not in sys.path:
                sys.path.append(path)

    def load_repository(self, name):
        """
        Load a map repository.

        @type name: string
        @param name: python module name for the repository.
        @raise e: if python module is not valid.
        """
        try:
            module = __import__(name)
            repo = module.get_repository()
            if isinstance(repo, RepoBase):
                return repo
            else:
                DEBUG("repository %s invalid for maps" % name)
        except Exception, e:
            ERROR("error opening repository %s: %s" % (name, e))

    def load_plugins_description(self):
        """
        Load all repositories description presented in plugins dir path.

        @rtype: list
        @return: list containing all repositories descriptions.
        """
        list_aux = []
        description = []

        for path in self.path:
            list_aux.append(os.path.join(os.path.basename(path), "__init__"))
            for plugin in glob.glob(os.path.join(path, "*.py*")):
                name = os.path.join(os.path.basename(os.path.dirname(plugin)),
                    os.path.basename(plugin).rsplit('.', 1)[0])

                if name in list_aux:
                    continue
                list_aux.append(name)

                repo = self.load_repository(name)
                if repo is not None:
                    description.append((repo.get_name(), name))

        description.sort()
        return description
