#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{MapModel}.
"""

import os, ecore, shutil

from common.singleton import Singleton
from common.carmanconfig import CarmanConfig
from models.dbusmodel import CarmandDbusModel
from models.tile_downloader import TileDownloader

class MapModel(Singleton):
    """
    Handles all map-related features (map tile downloads, map service updates,
    map positioning calculations, and others).

    @ivar   downloader: Instance of L{TileDownloader}.
    @ivar   repo:       Map service download repository.
    @ivar   fdh:        Ecore file download handler.
    """

    def __init__(self):
        Singleton.__init__(self)

        self.__map_service_updated_cbs = []

        self.downloader = TileDownloader()
        self.repo = None

        self.choose_repository(CarmanConfig().get_repository_default_name())

        self.fdh = ecore.FdHandler(self.downloader.initialize(), \
            ecore.ECORE_FD_READ, self.__process_download)
        self.fdh.active_set(ecore.ECORE_FD_READ)

    def __process_download(self, obj, *args, **kargs):
        """
        Processes L{downloader} results.

        @type   obj: object
        @param  obj: Not used.
        @type   args: object
        @param  args: Not used.
        @type   kargs: object
        @param  kargs: Not used.
        @rtype: boolean
        @return: Always returns C{True}, in order to keep ecore Timer calling
        it periodically.
        """
        self.downloader.process_results()
        return True

    def add_map_service_updated(self, cb):
        """
        Adds a callback to the list of callbacks to be called when map service
        is updated.

        @type   cb: callback
        @param  cb: Callback to be added.
        """
        if callable(cb) and cb not in self.__map_service_updated_cbs:
            self.__map_service_updated_cbs.append(cb)

    def del_map_service_updated(self, cb):
        """
        Removes a callback from the list of callbacks to be called when map
        service is updated.

        @type   cb: callback
        @param  cb: Callback to be removed.
        """
        if cb in self.__map_service_updated_cbs:
            self.__map_service_updated_cbs.remove(cb)

    def cancel_download(self, x, y, zoom):
        """
        Cancel download of a specific map region.

        @type   x: number
        @param  x: Cartesian map M{X} position.
        @type   y: number
        @param  y: Cartesian map M{Y} position.
        @type   zoom: number
        @param  zoom: Map zoom.
        """
        self.downloader.cancel_download(x, y, zoom)

    def cancel_fg_downloads(self):
        """
        Cancel all foreground map tile downloads.
        """
        self.downloader.clear_fg_queue()

    def cancel_bg_downloads(self):
        """
        Cancel all background map tile downloads.
        """
        self.downloader.clear_bg_queue()

    def cancel_all_downloads(self):
        """
        Cancel all map tile downloads.
        """
        self.downloader.clear_queue()

    def choose_repository(self, repository):
        """
        Chooses given repository as current.

        @type   repository: class
        @param  repository: Map service instance (eg. L{OSMRepo}).
        """
        if self.repo:
            self.downloader.clear_queue()
        self.repo = self.downloader.choose_repo(repository)
        self.downloader.set_cache_dir(CarmandDbusModel().GetMapsFolder())
        self.zoom_range = self.repo.get_zoom()
        self.dim_offset = self.repo.get_dimension_offset()
        for cb in self.__map_service_updated_cbs:
            cb(self.repo)

    def clear_repository_files(self):
        """
        Clear all map service repository files.
        """
        self.cancel_all_downloads()
        cache_dir = self.downloader.get_cache_dir()
        if os.path.exists(cache_dir):
            shutil.rmtree(cache_dir)
            os.makedirs(cache_dir)
        for cb in self.__map_service_updated_cbs:
            cb(self.repo)

    def get_downloads_zoom(self):
        """
        Get current download zoom.
        @rtype: number
        @return: Repository map zoom.
        """
        return self.repo.get_downloads_zoom()

    def get_repository(self):
        """
        Returns current map service.

        @rtype: class
        @return: Map service instance (e.g. L{OSMRepo}).
        """
        return self.repo

    def get_tile_filename(self, x, y, zoom):
        """
        Returns tile filename.

        @type   x: number
        @param  x: Cartesian map M{X} position.
        @type   y: number
        @param  y: Cartesian map M{Y} position.
        @type   zoom: number
        @param  zoom: Map zoom.
        @rtype: string
        @return: Tile filename.
        """
        return self.downloader.get_tile_filename(x, y, zoom)

    def get_tile_file_size_avg(self):
        """
        Returns map service average tile file size.

        @rtype: number
        @return: Current map service average tile file size.
        """
        return self.repo.get_average_size()

    def get_tile_size(self):
        """
        Returns map service default tile size.

        @rtype: tuple
        @return: Map service tile size C{(width, height)}.
        """
        return self.repo.get_tile_width(), self.repo.get_tile_height()

    def get_world_size(self, zoom):
        """
        Returns world size, which is calculated using given zoom.

        @type   zoom: number
        @param  zoom: Map zoom.
        @rtype: number
        @return: World size according to the given zoom.
        """
        return 256 << (self.zoom_range[-1] - zoom + self.dim_offset)

    def has_fg_downloads(self):
        """
        Verifies if L{TileDownloader} has foreground downloads.

        @rtype: boolean
        @return: C{True} if L{TileDownloader} has foreground downloads,
                 C{False} otherwise.
        """
        return self.downloader.has_fg_downloads()

    def has_bg_downloads(self):
        """
        Verifies if L{TileDownloader} has background downloads.

        @rtype: boolean
        @return: C{True} if L{TileDownloader} has background downloads,
                 C{False} otherwise.
        """
        return self.downloader.has_bg_downloads()

    def has_downloads(self):
        """
        Verifies if L{TileDownloader} has downloads.

        @rtype: boolean
        @return: C{True} if L{TileDownloader} has downloads, C{False}
                 otherwise.
        """
        return self.downloader.has_downloads()

    def is_valid_tile(self, x, y, zoom):
        """
        Verifies if the given tile coordinates are valid.

        @type   x: number
        @param  x: Cartesian map M{X} position.
        @type   y: number
        @param  y: Cartesian map M{Y} position.
        @type   zoom: number
        @param  zoom: Map zoom.
        @rtype: boolean
        @return: C{True} if tile coordinates are valid, C{False} otherwise.
        """
        if x < 0 or y < 0 or zoom < self.zoom_range[0] or \
          zoom > self.zoom_range[-1]:
            return False
        dim = 1 << (self.zoom_range[-1] - zoom + self.dim_offset)
        return x < dim and y < dim

    def latlon_to_xy(self, lat, lon):
        """
        Transformates latitude and longitudes coordinates in cartesian map
        M{X, Y} positions.

        @rtype: tuple
        @return: Cartesian map positions C{(x, y)}.
        """
        return self.repo.latlon_to_xy(lat, lon)

    def xy_to_latlon(self, x, y, zoom):
        """
        Transformates cartesian map M{X, Y} position in latitude and longitude
        coordinates.

        @rtype: tuple
        @return: Latitude and longitude coordinates C{(lat, lon)}.
        """
        return self.repo.xy_to_latlon(x << zoom, y << zoom)

    def request_tile(self, x, y, zoom, priority, cb):
        """
        Requests a map tile to be downloaded using given coordinates
        (C{x, y, zoom}) and priority (foreground or background). The
        given callback is called when the download finishes.

        @type   x: number
        @param  x: Cartesian map M{X} position.
        @type   y: number
        @param  y: Cartesian map M{Y} position.
        @type   zoom: number
        @param  zoom: Map zoom.
        @type   priority: boolean
        @param  priority: C{True} if foreground, C{False} if background.
        @type   cb: callback
        @param  cb: Callback to be called when map tile download finishes.
        @rtype: string
        @return: Map tile filename, if map tile file already exists.
        """
        return self.downloader.request_tile(x, y, zoom, priority, cb)

    def finalize(self):
        """
        Closes L{TileDownloader} instance.
        """
        self.downloader.close()
