#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{KeyBoardModel}.
"""

from main.mainview import MainView
from common.singleton import Singleton

class KeyBoardModel(Singleton):
    """
    Implements functionalities to associate callbacks to keyboard keys.
    """

    def __init__(self):
        Singleton.__init__(self)

        self._key_down_cbs = []
        self._default_key_down_cbs = {}

        self.kb = MainView().get_evas_canvas().Rectangle()
        self.kb.on_key_down_add(self.__key_down_cb)
        self.kb.focus_set(True)

    def set_focus(self):
        """
        Enable the keyboard usage.This method sets focus on evas Rectangle so it can accept I/O signals.
        """
        self.kb.focus_set(True)

    def __key_down_cb(self, obj, event):
        """
        Execute a callback associated with a key.

        @param obj: Object that generate the event. After instantiated, this
        object contains a evas.c_evas.Rectangle instance.
        @type obj: object
        @param event: The event generated by a evas.c_evas.EventKeyDown.
        @type event: object
        """
        ret = False
        if self._key_down_cbs:
            ret = self._key_down_cbs[-1](obj, event)

        if not ret:
            if event.key in self._default_key_down_cbs:
                self._default_key_down_cbs[event.key]()

    def add_key_down_cb(self, cb):
        """
        Adds a callback to the list of callbacks to be called when a key is pressed down.

        @param cb: Callback function.
        @type cb: callback
        """
        self.kb.focus_set(True)
        if callable(cb) and cb not in self._key_down_cbs:
            self._key_down_cbs.append(cb)

    def del_key_down_cb(self, cb):
        """
        Removes a callback to the list of callbacks to be called when a key is pressed down.

        @param cb: Callback function.
        @type cb: callback
        """
        if cb in self._key_down_cbs:
            self._key_down_cbs.remove(cb)

    def set_default_key_down_cb(self, key, cb):
        """
        Associates a default callback to a key.

        @param key: Key to execute callback.
        @type key: string
        @param cb: Callback function.
        @type cb: callback
        """
        if callable(cb):
            self._default_key_down_cbs[key] = cb
