#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{InfoSharingModel}.

@var GREETING_PATTERN: Greeting message regular expression pattern.
@var ACCEPTED_PATTERN: Accepted connection message regular expression pattern.
@var REJECTED_PATTERN: Rejected connection message regular expression pattern.
@var CLOSE_CONN_PATTERN: Close connection message regular expression pattern.
@var DATA_PATTERN:   Data message regular expression pattern.
@var GREETING_FMT: Greeting message format.
@var ACCEPTED_FMT: Accepted message format.
@var REJECTED_FMT: Rejected message format.
@var CLOSE_CONN_FMT: Close connection message format.
@var DATA_FMT: Data message format.
@var GREETING_MESSAGE: Greeting message. Notice that L{GREETING_FMT} is
                       encapsulated here before it is sent.
@var NO_IM_MESSAGE: This message is shown to a non-Carman buddy which sends to
                    user a common message. A L{CLOSE_CONN_FMT} is encapsulated
                    here before it is sent to prevent erroneous connections.
@var __DATA_REPORT_TIME__: Default data report timer value.
"""

import ecore
import re

from common.carmanconfig import CarmanConfig
from common.singleton import Singleton
from main.messagedlgctrl import MessageDlgController
from models.buddymodel import BuddyModel
from models.connmodel import ConnectionModel
from models.dbusmodel import CarmandDbusModel
from models.gpsmodel import GPSModel
from models.obdmodel import OBDModel

GREETING_PATTERN   = "^.*cds;ccs;cde;$"
ACCEPTED_PATTERN   = "^.*cds;cca;cde;$"
REJECTED_PATTERN   = "^.*cds;ccr;cde;$"
CLOSE_CONN_PATTERN = "^.*cds;cce;cde;$"
DATA_PATTERN       = "^.*cds;dds;(([\\-]*[0-9]+\\.[0-9]+;){6})(([A-Z][a-z]+;){2})dde;cde;$"

GREETING_FMT       = "cds;ccs;cde;"
ACCEPTED_FMT       = "cds;cca;cde;"
REJECTED_FMT       = "cds;ccr;cde;"
CLOSE_CONN_FMT     = "cds;cce;cde;"
DATA_FMT           = "cds;dds;%f;%f;%f;%f;%f;%f;%s;%s;dde;cde;"

GREETING_MESSAGE   = "I'm trying to share my Carman information with you, but it appears that you're not using Carman as your IM client. If you don't want to share any information, please ignore this message.<br>For more information about Carman, please visit: http://openbossa.indt.org/carman<br><span style='color: #ffffff'>%s</span>"

NO_IM_MESSAGE      = "You are trying to communicate with me, but I'm currently using Carman and only accepting Carman-like messages. For more information about Carman, please visit: http://openbossa.indt.org/carman<br><span style='color: #ffffff'>%s</span>"

__DATA_REPORT_TIME__ = 3.0


class InfoSharingModel(Singleton):
    """
    Handles InfoShare-related features (instant messaging account connection,
    sent/received messages, connection status).

    This model uses a service to handle account connection (eg.
    L{ISService}).

    @cvar DISCONNECTED: Infoshare account status flag (account disconnected).
    @cvar CONNECTING: Infoshare account status flag (account connecting).
    @cvar CONNECTED: Infoshare account status flag (account connected).
    @ivar __data: Infoshare data C{(lat, lon, alt, speed, track, rpm)}.
    @ivar __is_service: Private variable which represents the service this
                        model uses to handle account connection (other models
                        should use this using L{service} property).
    """

    DISCONNECTED = "Disconnected"
    CONNECTING   = "Connecting"
    CONNECTED    = "Connected"

    def __init__(self):
        Singleton.__init__(self)

        self.__account_status = self.DISCONNECTED
        self.__config = CarmanConfig()
        self.__connection_error = False
        self.__data = [float(-1)] * 6
        self.__data_timer = None
        self.__is_service = None
        self.__gps_has_data = False

        self.__network_error_cbs = []
        self.__signed_off_cbs = []
        self.__signed_on_cbs = []
        self.__status_changed_cbs = []

        self.carmand_model = CarmandDbusModel()

        self.buddy_model = BuddyModel()
        self.buddy_model.add_buddy_connect_cb(self.__buddy_connected_cb)
        self.buddy_model.add_buddy_disconnect_cb(self.__buddy_disconnected_cb)

        self.conn_model = ConnectionModel()
        self.conn_model.add_connection_lost_cb(self.__connection_lost_cb)

        self.gps_model = GPSModel()
        self.gps_model.add_data_available_cb(self.__gps_data_available_cb)

        self.obd_model = OBDModel()
        self.obd_model.add_data_available_cb((("0C", 1, 0), ), \
                                             self.__obd_rpm_available_cb)
        self.obd_model.add_data_available_cb((("0D", 1, 0), ), \
                                             self.__obd_speed_available_cb)

    def instantiate_isservice(self):
        """
        Instantiates L{ISService}. This method causes L{ISService} to launch
        C{infosharingd}, which runs on a separate process and communicates
        with Carman using DBus.
        """
        from models.isservice import ISService
        self.__is_service = ISService()

        # Adds the proper callbacks to InfoShare service
        service = self.__is_service
        service.add_callback('buddy-signed-on', self.__on_buddy_signed_on)
        service.add_callback('buddy-signed-off', self.__on_buddy_signed_off)
        service.add_callback('connection-error', self.__on_connection_error)
        service.add_callback('received-msg', self.__on_received_msg)
        service.add_callback('request-add-buddy', self.__on_request_add_buddy)
        service.add_callback('request-authorize-buddy', \
            self.__on_request_authorize_buddy)
        service.add_callback('signed-on', self.__on_signed_on)
        service.add_callback('signed-off', self.__on_signed_off)

    def __get_service(self):
        """
        Returns InfoShare service.

        @rtype: class
        """
        return self.__is_service

    service = property(__get_service)

    def __get_account_status(self):
        """
        Returns account status (L{DISCONNECTED}, L{CONNECTING} or
        L{CONNECTED}).

        @rtype: string
        """
        return self.__account_status

    def __set_account_status(self, status):
        """
        Updates account status.

        @type   status: string
        @param  status: Account status (L{DISCONNECTED}, L{CONNECTING} or
                        L{CONNECTED}).
        """
        if status in [self.CONNECTED, self.CONNECTING, self.DISCONNECTED]:
            for cb in self.__status_changed_cbs:
                cb(status)
            self.__account_status = status

    account_status = property(__get_account_status, __set_account_status)

    def __gps_data_available_cb(self, *data):
        """
        Updates L{data} instance variable with available GPS data.

        @type   data:   tuple
        @param  data:   GPS data C{(lat, lon, alt, speed, track)}.
        """
        # when GPS is stopped, data items can get a 'nan'
        if all([item == item for item in data]):
            if self.obd_model.Status() != OBDModel.CONNECTED:
                self.__data[3] = float(data[3]) # speed

            if self.__data[3] > 0 or not self.__gps_has_data: # car is moving
                self.__gps_has_data = True
                self.__data[0] = float(data[0]) # lat
                self.__data[1] = float(data[1]) # lon
                self.__data[2] = float(data[2]) # alt
                self.__data[4] = float(data[4]) # track

    def __obd_rpm_available_cb(self, model, pid, *data):
        """
        Updates L{data} instance variable with available OBD's RPM data.

        @type   model:  class
        @param  model:  L{OBDModel} instance which generated signal.
        @type   pid:    number
        @param  pid:    Handle representing OBD's RPM data.
        @type   data:   list
        @param  data:   List of latest RPM data from OBD.
        """
        self.__data[5] = float(data[0]) # rpm

    def __obd_speed_available_cb(self, model, pid, *data):
        """
        Updates L{data} instance variable with available OBD's speed data.

        @type   model:  class
        @param  model:  L{OBDModel} instance which generated signal.
        @type   pid:    number
        @param  pid:    Handle representing OBD's speed data.
        @type   data:   list
        @param  data:   List of latest speed data from OBD.
        """
        self.__data[3] = float(data[0]) # speed

    def __buddy_connected_cb(self, name):
        """
        Creates a timer which periodically calls L{__send_data_cb} whenever one
        or more buddies are connected.

        @type   name:   string
        @param  name:   Buddy name.
        """
        self.__gps_has_data = False
        if not self.__data_timer:
            self.__data_timer = ecore.Timer(__DATA_REPORT_TIME__, \
                self.__send_data_cb)

    def __buddy_disconnected_cb(self, name, reason=None, connected=None):
        """
        Sends close connection message to given buddy. Also removes
        L{__send_data_cb} timer when there are no more connected buddies left.

        @type   name:   string
        @param  name:   Buddy name.
        @type   reason: string
        @param  reason: Buddy disconnection reason.
        @type   connected:  boolean
        @param  connected:  Not used.
        """
        self.send_close_connection(name)
        if not self.buddy_model.has_buddies() and self.__data_timer:
            self.__data_timer.delete()
            self.__data_timer = None

    def __connection_lost_cb(self):
        """
        Disconnects from all connecting/connected buddies when account
        connection is lost.
        """
        self.buddy_model.disconnect_all_buddies()

    def __on_buddy_signed_on(self, name, alias):
        """
        Callback which is called when a buddy signs on. Currently it is not
        used by Carman.

        @type   name:   string
        @param  name:   Buddy name (eg. C{average.joe@mail.com}).
        @type   alias:  string
        @param  alias:  Buddy alias (eg. C{Joe, the average guy}).
        """
        pass

    def __on_buddy_signed_off(self, name, alias):
        """
        Disconnects buddy from L{BuddyModel}, if buddy was connected.

        @type   name:   string
        @param  name:   Buddy name (eg. C{average.joe@mail.com}).
        @type   alias:  string
        @param  alias:  Buddy alias (eg. C{Joe, the average guy}).
        """
        if self.buddy_model.is_buddy_connected(name):
            self.buddy_model.disconnect_buddy(name, reason="Connection lost")

    def __on_connection_error(self, username, protocol_id, short_desc, desc=None):
        """
        Displays a message to the user warning about account connection error.

        @type   username: string
        @param  username: Account username (eg. C{john.smith@mail.com}).
        @type   protocol_id: string
        @param  protocol_id: Account protocol ID (eg. C{prpl-jabber}).
        @type   short_desc: string
        @param  short_desc: Connection error reason (short description).
        @type   desc: string
        @param  desc: Connection error reason (detailed description).
        """
        self.buddy_model.disconnect_all_buddies()

        short_desc = short_desc.split("\n")[0]
        def reconnect_account_cb():
            disconnect_account_cb()
            for cb in self.__network_error_cbs:
                cb()
            self.__set_account_status(self.CONNECTING)

        def disconnect_account_cb():
            self.__is_service.account_disconnect()
            self.__set_account_status(self.DISCONNECTED)

        self.__connection_error = not self.__connection_error
        if short_desc == "Network error":
            msg = MessageDlgController(confirm_cb=reconnect_account_cb,
                    cancel_cb=disconnect_account_cb)
            msg.show_message("%s<br>Reconnect account?" % \
                    short_desc, buttons=2, title="ACCOUNT STATUS")
        else:
            msg = MessageDlgController(confirm_cb=disconnect_account_cb)
            if desc:
                msg.show_message("Connection error:<br>%s<br>%s" % \
                        (short_desc, desc), title="ACCOUNT STATUS")
            else:
                msg.show_message("Connection error:<br>%s" % short_desc, \
                        title="ACCOUNT STATUS")

    def __on_received_msg(self, name, alias, stripped):
        """
        When a message is received from a buddy, this method verifies if the
        message is Carman-like. If yes, it redirects this to the proper
        callbacks (e.g. L{__received_greeting_msg} if the message is a greeting
        message). If not, it sends a message to the buddy notifying him/her
        that this client only accepts Carman-like messages.

        @type   name: string
        @param  name: Buddy name (eg. C{average.joe@mail.com}).
        @type   alias: string
        @param  alias: Buddy alias (eg. C{Joe, the average guy}).
        @type   stripped: string
        @param  stripped: HTML-stripped message sent from buddy.
        @rtype: boolean
        @return: C{True} if message is usable, C{False} otherwise.
        """
        greeting_pattern = re.compile(GREETING_PATTERN, re.MULTILINE)
        accepted_pattern = re.compile(ACCEPTED_PATTERN, re.MULTILINE)
        rejected_pattern = re.compile(REJECTED_PATTERN, re.MULTILINE)
        close_conn_pattern = re.compile(CLOSE_CONN_PATTERN, re.MULTILINE)
        data_pattern = re.compile(DATA_PATTERN, re.MULTILINE)

        # Fix buddy name/alias string encoding
        name = name.split("/")[0].encode("utf-8")
        alias = alias.encode("utf-8")

        if data_pattern.search(stripped):
            return self.buddy_model.received_data_msg(name, stripped)
        elif greeting_pattern.search(stripped):
            return self.__received_greeting_msg(name, alias)
        elif accepted_pattern.search(stripped):
            return self.__received_accepted_msg(name, alias)
        elif rejected_pattern.search(stripped):
            return self.__received_rejected_msg(name)
        elif close_conn_pattern.search(stripped):
            return self.buddy_model.received_close_conn_msg(name)
        else:
            # Someone not using Carman sent a message to you. Send a
            # human-readable message to this buddy.
            self.__is_service.send_message(name, NO_IM_MESSAGE % CLOSE_CONN_FMT)
            return False

    def __on_request_add_buddy(self, name, alias, msg):
        """
        Displays a message to the user notifying that a new buddy has added
        him/her.

        @type   name:   string
        @param  name:   Buddy name (eg. C{average.joe@mail.com}).
        @type   alias:  string
        @param  alias:  Buddy alias (eg. C{Joe, the average guy}).
        @type   msg:    string
        @param  msg:    Buddy request message, if any (not used).
        """
        def add_buddy_cb():
            self.__is_service.add_buddy(name, alias)

        if alias:
            message = "%s (%s)<br>added you.<br>Add buddy?" % (name, alias)
        else:
            message = "%s<br>Added you.<br>Add buddy?" % (name)

        msg = MessageDlgController(confirm_cb=add_buddy_cb)
        msg.show_message(message, buttons=2, title="BUDDY REQUEST")

    def __on_request_authorize_buddy(self, name, alias, on_list):
        """
        Displays a message to the user notifying that a new buddy has
        requested authorization from him/her to appear on his/her list.

        @type   name:   string
        @param  name:   Buddy name (eg. C{average.joe@mail.com}).
        @type   alias:  string
        @param  alias:  Buddy alias (eg. C{Joe, the average guy}).
        @type   on_list:    boolean
        @param  on_list:    C{True} if buddy appears on buddy list,
                            C{False} otherwise.
        """
        def authorize_cb():
            self.__is_service.authorize_buddy(name, True)
            if not on_list:
                self.__is_service.add_buddy(name, alias)

        def deny_cb():
            self.__is_service.authorize_buddy(name, False)
            if on_list:
                self.__is_service.remove_buddy(name)

        if alias:
            message = "%s (%s)<br>wants to add you.<br>" \
                    "Authorize and add buddy?" % (name, alias)
        else:
            message = "%s<br>wants to add you.<br>" \
                    "Authorize and add buddy?" % (name)

        msg = MessageDlgController(confirm_cb=authorize_cb, cancel_cb=deny_cb)
        msg.show_message(message, buttons=2, title="BUDDY REQUEST")

    def __on_signed_on(self, username, protocol_id):
        """
        Calls all registered callbacks for the case when an account has signed
        on. Also displays a message to the user notifying him/her that the
        account has signed on.

        @type   username: string
        @param  username: Account username (eg. C{john.smith@mail.com}).
        @type   protocol_id: string
        @param  protocol_id: Account protocol ID (eg. C{prpl-jabber}).
        """
        self.__set_account_status(self.CONNECTED)
        self.__config.last_account_status = 'online'

        protocols = self.service.get_protocols()
        msg = MessageDlgController()
        msg.show_message("You are now signed on<br>%s<br>(%s)" % \
            (username.split("/")[0], protocols[protocol_id][0]), \
            title="ACCOUNT STATUS")

        for cb in self.__signed_on_cbs:
            cb(username, protocol_id)

    def __on_signed_off(self, username, protocol_id):
        """
        Calls all registered callbacks for the case when an account has signed
        off. Also displays a message to user notifying him/her that the
        account has signed off.

        @type   username: string
        @param  username: Account username (eg. C{john.smith@mail.com}).
        @type   protocol_id: string
        @param  protocol_id: Account protocol ID (eg. C{prpl-jabber}).
        """
        self.__set_account_status(self.DISCONNECTED)
        self.__config.last_account_status = 'offline'

        if self.__connection_error:
            self.__connection_error = not self.__connection_error
        else:
            self.buddy_model.disconnect_all_buddies()
            protocols = self.service.get_protocols()
            msg = MessageDlgController()
            msg.show_message("You are now signed off<br>%s<br>(%s)" % \
                (username.split("/")[0], protocols[protocol_id][0]), \
                title="ACCOUNT STATUS")

        for cb in self.__signed_off_cbs:
            cb(username, protocol_id)

    def __received_accepted_msg(self, name, alias):
        """
        Handles an accepted connection message. It notifies L{BuddyModel} to
        add the buddy who accepted the connection.

        @type   name: string
        @param  name: Buddy name (eg. C{average.joe@mail.com}).
        @type   alias: string
        @param  alias: Buddy alias (eg. C{Joe, the average guy}).
        @rtype: boolean
        @return: C{True} if message is usable, C{False} otherwise.
        """
        if self.buddy_model.has_request(name):
            self.buddy_model.remove_request(name, accepted=True)

            if not self.buddy_model.is_buddy_connected(name):
                self.buddy_model.connect_buddy(name)
        return True

    def __received_greeting_msg(self, name, alias):
        """
        handles a greeting message. If all exceptions are verified, it calls
        L{BuddyModel} to add a request timer.

        @type   name: string
        @param  name: Buddy name (eg. C{average.joe@mail.com}).
        @type   alias: string
        @param  alias: Buddy alias (eg. C{Joe, the average guy}).
        @rtype: boolean
        @return: C{True} if message is usable, C{False} otherwise.
        """
        def user_accepted_cb():
            self.send_accept_conn(name)
            self.buddy_model.connect_buddy(name)

        def user_rejected_cb():
            self.send_reject_conn(name)

        if self.buddy_model.has_request(name):
            # Request already in progress
            pass
        elif self.buddy_model.is_buddy_connected(name):
            # Buddy already connected
            pass
        elif self.buddy_model.size >= BuddyModel.MAX_BUDDIES:
            # Reached maximum number of connected buddies
            self.send_reject_conn(name)
        else:
            self.buddy_model.add_received_request_timeout(name, alias, \
                user_accepted_cb, user_rejected_cb)

        return True

    def __received_rejected_msg(self, name):
        """
        Handles a rejected connection message. It notifies L{BuddyModel} to
        remove the request from the buddy who rejected connection.

        @type   name: string
        @param  name: Buddy name (eg. C{average.joe@mail.com}).
        @rtype: boolean
        @return: C{True} if message is usable, C{False} otherwise.
        """
        if self.buddy_model.has_request(name):
            self.buddy_model.remove_request(name)

            msg = MessageDlgController()
            msg.show_message("%s<br> rejected connection request" % name)
        return True

    def __send_data_cb(self):
        """
        Sends data message to connected buddies.

        This method picks up latest GPS/OBD data from L{__data} and status
        from both L{GPSModel} and L{OBDModel}.

        @rtype: boolean
        @return: Always returns C{True}, in order to keep ecore Timer calling
                 it periodically.
        """
        mode = self.carmand_model.GetMode()

        if (not self.__gps_has_data) or \
                (self.carmand_model.GetGPSDevice() == "none" and \
                mode == "normal"):
            gps_status = GPSModel.DISABLED
        else:
            gps_status = self.gps_model.Status()

        if self.carmand_model.GetOBDDevice() == "none" and mode == "normal":
            obd_status = OBDModel.DISABLED
        else:
            obd_status = self.obd_model.Status()

        message = DATA_FMT % (self.__data[0], self.__data[1], \
            self.__data[2], self.__data[3], self.__data[4], self.__data[5], \
            gps_status, obd_status)

        for buddy in self.buddy_model.get_connected_buddies():
            self.__is_service.send_message(buddy, message)

        return True

    def add_network_error_cb(self, cb):
        """
        Adds a callback to the list of callbacks to be called when a network
        error occurs.

        @type   cb: callback
        @param  cb: Callback to be added.
        """
        if callable(cb) and cb not in self.__network_error_cbs:
            self.__network_error_cbs.append(cb)

    def del_network_error_cb(self, cb):
        """
        Removes a callback from the list of callbacks to be called when a
        network error occurs.

        @type   cb: callback
        @param  cb: Callback to be removed.
        """
        if cb in self.__network_error_cbs:
            self.__network_error_cbs.remove(cb)

    def add_signed_on_cb(self, cb):
        """
        Adds a callback to the list of callbacks to be called when account
        has signed on.

        @type   cb: callback
        @param  cb: Callback to be added.
        """
        if callable(cb) and cb not in self.__signed_on_cbs:
            self.__signed_on_cbs.append(cb)

    def del_signed_on_cb(self, cb):
        """
        Removes a callback from the list of callbacks to be called when
        account has signed on.

        @type   cb: callback
        @param  cb: Callback to be removed.
        """
        if cb in self.__signed_on_cbs:
            self.__signed_on_cbs.remove(cb)

    def add_signed_off_cb(self, cb):
        """
        Adds a callback to the list of callbacks to be called when account
        has signed off.

        @type   cb: callback
        @param  cb: Callback to be added.
        """
        if callable(cb) and cb not in self.__signed_off_cbs:
            self.__signed_off_cbs.append(cb)

    def del_signed_off_cb(self, cb):
        """
        Removes a callback from the list of callbacks to be called when
        account has signed off.

        @type   cb: callback
        @param  cb: Callback to be removed.
        """
        if cb in self.__signed_off_cbs:
            self.__signed_off_cbs.remove(cb)

    def add_status_changed_cb(self, cb):
        """
        Adds a callback to the list of callbacks to be called when account
        status has changed.

        @type   cb: callback
        @param  cb: Callback to be added.
        """
        if callable(cb) and cb not in self.__status_changed_cbs:
            self.__status_changed_cbs.append(cb)

    def del_status_changed_cb(self, cb):
        """
        Removes a callback from the list of callbacks to be called when
        account status has changed.

        @type   cb: callback
        @param  cb: Callback to be removed.
        """
        if cb in self.__status_changed_cbs:
            self.__status_changed_cbs.remove(cb)

    def get_buddy_alias(self, name):
        """
        Returns buddy alias.

        @type   name: string
        @param  name: Buddy name (eg. C{average.joe@mail.com}).
        @rtype: string
        @return: Buddy alias (e.g. C{Joe, the average guy}).
        """
        if self.__is_service:
            return self.__is_service.get_buddy_alias(name)
        else:
            return None

    def send_accept_conn(self, name):
        """
        Sends an accepted connection message to the given buddy.

        @type   name: string
        @param  name: Buddy name (eg. C{average.joe@mail.com}).
        """
        self.__is_service.send_message(name, ACCEPTED_FMT)

    def send_close_connection(self, name):
        """
        Sends a close connection message to the given buddy.

        @type   name: string
        @param  name: Buddy name (eg. C{average.joe@mail.com}).
        """
        self.__is_service.send_message(name, CLOSE_CONN_FMT)

    def send_reject_conn(self, name):
        """
        Sends a rejected connection message to the given buddy.

        @type   name: string
        @param  name: Buddy name (eg. C{average.joe@mail.com}).
        """
        self.__is_service.send_message(name, REJECTED_FMT)

    def send_request_authorization(self, name):
        """
        Sends an authorization request message to the given buddy.

        @type   name: string
        @param  name: Buddy name (eg. C{average.joe@mail.com}).
        """
        if self.buddy_model.has_request(name):
            msg = MessageDlgController()
            msg.show_message("Connection request failed:<br>" \
                "Request already in progress", title="BUDDY REQUEST")
        elif self.buddy_model.is_buddy_connected(name):
            msg = MessageDlgController()
            msg.show_message("Connection request failed:<br>" \
                "Buddy already connected", title="BUDDY REQUEST")
        else:
            self.__is_service.send_message(name, \
                GREETING_MESSAGE % GREETING_FMT)
            self.buddy_model.add_sent_request_timeout(name)
