#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{DeleteBuddyController}.
"""

#from common.carlog import DEBUG
from main.listctrl import ListView
from main.messagedlgctrl import MessageDlgController
from models.buddymodel import BuddyModel
from models.infosharingmodel import InfoSharingModel

class DeleteBuddyController(object):
    """
    Delete Buddy dialog controller.

    @type   parent: L{ListView}
    @param  parent: Parent window.
    """
    def __init__(self, parent):
        self._parent = parent
        self.buddy_model = BuddyModel()
        self.is_service = InfoSharingModel().service
        self.view = ListView("list-normal", "list-item-normal", "DELETE BUDDY")

    def __buddy_selected_cb(self, obj, item, param):
        """
        Removes selected buddy from the buddy list. If the buddy
        is connected, user is disconnected from the buddy, then he/she
        is deleted.

        @type   obj: L{ListView}
        @param  obj: Not used.
        @type   item: L{ListItemView}
        @param  item: Not used.
        @type   param: object
        @param  param: Not used.
        """
        # Delete buddy from buddy list
        def delete_buddy():
            account = self.is_service.get_accounts_active()
            self.is_service.remove_buddy(param[0])
            obj.remove_item(item)
            if not obj.has_items():
                obj.hide(True)

        # Disconnect from an active buddy first
        def disconnect_buddy():
            self._parent.hide(True)
            self.buddy_model.disconnect_buddy(param[0])
            delete_buddy()

        # Get buddy name or alias
        if param[1]:
            buddy_name = param[1]
        else:
            buddy_name = param[0]

        # Check if the buddy is connected
        if self.buddy_model.is_buddy_connected(param[0]):
            msg = MessageDlgController(confirm_cb=disconnect_buddy)
            msg.show_message("Buddy is connected.<br>Disconnect and " \
                    "delete<br>%s?" % buddy_name, buttons=2, \
                    title="DELETE BUDDY")
        else:
            msg = MessageDlgController(confirm_cb=delete_buddy)
            msg.show_message("Do you really want to delete buddy <br>%s?" % \
                            buddy_name, buttons=2, title="DELETE BUDDY")

    def load_list(self):
        """
        Verifies if all exceptions are validated before displaying list of
        connectable buddies.

        @rtype: boolean
        @return: C{True} if all exceptions are validated, C{False} otherwise.
        """
        account = self.is_service.get_accounts_active()
        if not account:
            self.view.hide(True)
            msg = MessageDlgController()
            msg.show_message("No account currently active", \
                title="DELETE BUDDY")
            return False

        buddies = self.is_service.get_buddies_online()
        if not buddies:
            self.view.hide(True)
            msg = MessageDlgController()
            msg.show_message("No buddies online", title="DELETE BUDDY")
            return False

        for buddy in buddies:
            if buddy[1]:
                self.view.add_item(buddy[1], self.__buddy_selected_cb, \
                        param=buddy)
            else:
                self.view.add_item(buddy[0], self.__buddy_selected_cb, \
                        param=buddy)
        return True

    def show(self):
        """
        Shows L{ListView}.
        """
        self.view.show()
