#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{MessageDlgController}.
"""

from common.carlog import DEBUG
from models.kbmodel import KeyBoardModel
from main.messagedlgview import MessageDlgView

class MessageDlgController(object):
    """
    Controls the message dialog window, which acts as an agent asking user
    about a decision to make or simply notifying him/her about something
    that occured. This class controls a L{view} which has an instance of
    L{MessageDlgView}.

    When user clicks on C{Confirm} or C{Cancel} buttons (in some cases just
    the C{Confirm} button appears), a callback associated to that action is
    then called.

    @type   confirm_cb: callback
    @param  confirm_cb: Callback to be called when user clicks on C{Confirm}
                        or C{OK} button.
    @type   cancel_cb: callback
    @param  cancel_cb: Callback to be called when user clicks on C{Cancel}
                       button, if any.
    @ivar   __confirm_cb: Stores the c{confirm_cb}, if callable.
    @ivar   __cancel_cb: Stores the c{cancel_cb}, if callable.
    @ivar   view: Instance of L{MessageDlgView}.
    """

    def __init__(self, confirm_cb=None, cancel_cb=None):
        add_callback = lambda cb: cb if callable(cb) else None
        self.__confirm_cb = add_callback(confirm_cb)
        self.__cancel_cb = add_callback(cancel_cb)

        self.view = MessageDlgView()
        self.view.confirm_connect_cb(self.confirm_cb)
        self.view.cancel_connect_cb(self.cancel_cb)

        self.kb_model = KeyBoardModel()
        self.kb_model.add_key_down_cb(self.__key_down_cb)

    def __key_down_cb(self, obj, event):
        """
        Associates key presses with callbacks.

        @type   obj: class
        @param  obj: Not used.
        @type   event: class
        @param  event: Instance of L{ecore.x.EventKeyDown}.

        @rtype: boolean
        @return: C{True} if event is used, C{False} otherwise.
        """
        if event.key in ("Return", "space", "KP_Enter"):
            self.confirm_cb()
        elif event.key == "Escape":
            self.cancel_cb()
        else:
            return False
        return True

    def show_message(self, msg, buttons=1, title=None, \
            confirm_label=None, cancel_label=None):
        """
        Shows the L{MessageDlgView} according to the given parameters.

        @type   msg: string
        @param  msg: Message to be displayed to user.
        @type   buttons: number
        @param  buttons: Number of buttons to be displayed (1 or 2).
        @type   title: string
        @param  title: Message dialog window's title.
        @type   confirm_label: string
        @param  confirm_label: Label to be applied on the C{Confirm} or C{OK}
                               button, if any.
        @type   cancel_label: string
        @param  cancel_label: Label to be applied on the C{Cancel} button, if
                              any.
        """
        if buttons == 2:
            self.view.set_buttons()
        if title:
            self.view.set_title(title)
        self.view.show_message(msg, confirm_label, cancel_label)

    def confirm_cb(self, *data):
        """
        Called when user clicks on C{Confirm} or C{OK} button. Calls the
        callback contained in L{__confirm_cb}, if any.

        @type   data: tuple
        @param  data: Not used.
        """
        self.view.hide()
        self.kb_model.del_key_down_cb(self.__key_down_cb)

        if self.__confirm_cb:
            self.__confirm_cb()
        del self.__confirm_cb, self.__cancel_cb

    def cancel_cb(self, *data):
        """
        Called when user clicks on C{Cancel} button. Calls the callback
        contained in L{__confirm_cb}, if any.

        @type   data: tuple
        @param  data: Not used.
        """
        self.view.hide()
        self.kb_model.del_key_down_cb(self.__key_down_cb)

        if self.__cancel_cb:
            self.__cancel_cb()
        del self.__confirm_cb, self.__cancel_cb
