#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{ScreenLoader}.

@todo: Document this module.
"""

import os, edje

from common.carlog import WARNING, DEBUG
from gauges.bar import BarView
from gauges.digital import DigitalView
from gauges.analogue import AnalogueView
from gauges.arcbar import ArcBarView
from gauges.screen import ScreenController
from gauges.gaugectrl import GaugeController
from common.sensorreader import SensorReader
from common.carmanconfig import CarmanConfig, METRIC

class ScreenLoader(object):
    """
    @todo: Document this class.
    """
    def __init__(self, canvas, file):
        """
        Screen Constructor
        @param canvas: Evas canvas object
        @param file: Edje file
        """
        self.file = file
        self.canvas = canvas
        self.config = CarmanConfig()
        self.metric = self.config.get_unit() == METRIC
        self.sensors = SensorReader("/usr/share/carman/sensors.xml")

    def __read_data_item(self, group, part, param):
        """
        Return an item from data section in edje file
        @param group: group name
        @param part: part name
        @param param: parameter name
        """
        return edje.file_data_get(self.file, group + "." + part + "." + param)

    def __read_data_item_with_preset(self, group, part, param, preset):
        """
        Return an item or a preset item from data section in edje file
        @param group: group name
        @param part: part name
        @param param: parameter name
        @param preset: preset name
        """
        value = self.__read_data_item(group, part, param)
        if not value and preset:
            return edje.file_data_get(self.file, "preset:" + preset + "." + \
                param)
        return value

    def __add_param(self, obj_params, group, part, param, preset):
        """
        Add a string parameter to dictionary
        @param obj_params: dictionary of the parameters
        @param group: group name
        @param part: part name
        @param param: parameter name
        @param preset: preset name
        """
        value = self.__read_data_item_with_preset(group, part, param, preset)
        if value:
            obj_params[param] = value

    def __add_param_int(self, obj_params, group, part, param, preset):
        """
        Add a integer parameter to dictionary
        @param obj_params: dictionary of the parameters
        @param group: group name
        @param part: part name
        @param param: parameter name
        @param preset: preset name
        """
        value = self.__read_data_item_with_preset(group, part, param, preset)
        if value:
            try:
                obj_params[param] = int(value)
            except:
                WARNING("Error converting value \"%s\" from %s.%s" % \
                    (value, group, part))

    def __add_param_image(self, obj_params, group, part, param, preset):
        """
        Add a image parameter to dictionary
        @param obj_params: dictionary of the parameters
        @param group: group name
        @param part: part name
        @param param: parameter name
        @param preset: preset name
        """
        value = self.__read_data_item_with_preset(group, part, param, preset)
        if value:
            obj_params[param] = (self.file, "images/" + value)

    def __add_param_font(self, obj_params, group, part, param, preset):
        """
        Add a font parameter to dictionary
        @param obj_params: dictionary of the parameters
        @param group: group name
        @param part: part name
        @param param: parameter name
        @param preset: preset name
        """
        def read_font_data_item(param, default=None):
            value = edje.file_data_get(self.file, "font:" + font_preset + \
                "." + param)
            return value and value or default

        font = {"font_source": self.file}
        font_preset = self.__read_data_item_with_preset(group, part, param,
            preset)
        name = "fonts/" + read_font_data_item("font", "default")
        try:
            size = int(read_font_data_item("size"))
            font["font"] = (name, size)
        except ValueError:
            font["font"] = name
        font["color"] = read_font_data_item("color", "#FFFFFFFF")
        obj_params["font"] = font

    def __create_bar_view(self, group, part, preset):
        """
        Return a new Bar smartobject
        @param group: group name
        @param part: part name
        @param preset: preset name
        """
        params = {}
        self.__add_param_image(params, group, part, "image", preset)
        self.__add_param_int(params, group, part, "pieces", preset)
        self.__add_param(params, group, part, "mode", preset)
        try:
            return BarView(self.canvas, **params)
        except Exception, e:
            WARNING("Error creating bar view %s.%s: %s" % \
                (group, part, e))

    def __create_arcbar_view(self, group, part, preset):
        """
        Return a new ArcBar smartobject
        @param group: group name
        @param part: part name
        @param preset: preset name
        """
        params = {}
        self.__add_param_image(params, group, part, "image", preset)
        self.__add_param_int(params, group, part, "pieces", preset)
        self.__add_param_int(params, group, part, "initial_angle", preset)
        self.__add_param_int(params, group, part, "total_angle", preset)
        self.__add_param_int(params, group, part, "radius_v", preset)
        self.__add_param_int(params, group, part, "radius_h", preset)
        self.__add_param(params, group, part, "clock_wise", preset)
        try:
            return ArcBarView(self.canvas, **params)
        except Exception, e:
            WARNING("Error creating arcbar view %s.%s: %s" % \
                (group, part, e))

    def __create_analogue_view(self, group, part, preset):
        """
        Return a new Gauge smartobject
        @param group: group name
        @param part: part name
        @param preset: preset name
        """
        params = {}
        self.__add_param_image(params, group, part, "image", preset)
        self.__add_param_int(params, group, part, "radius", preset)
        self.__add_param_int(params, group, part, "initial_angle", preset)
        self.__add_param_int(params, group, part, "total_angle", preset)
        self.__add_param(params, group, part, "clock_wise", preset)
        self.__add_param_int(params, group, part, "pieces", preset)
        self.__add_param_font(params, group, part, "font", preset)
        try:
            return AnalogueView(self.canvas, **params)
        except Exception, e:
            WARNING("Error creating gauge analogue view %s.%s: %s" % \
                (group, part, e))

    def __create_digital_view(self, scr_edje, group, part, preset):
        """
        Return a new Label object
        @param group: group name
        @param part: part name
        @param preset: preset name
        """
        params = {}
        self.__add_param_int(params, group, part, "round", preset)
        self.__add_param(params, group, part, "has_unit", preset)
        try:
            return DigitalView(scr_edje, part, **params)
        except Exception, e:
            WARNING("Error creating gauge digital view %s.%s: %s" % \
                (group, part, e))

    def __load_module(self, module):
        return __import__(os.path.join(self.config.get_theme_root_path(),
            "modules", module))

    def __create_custom_view(self, scr_edje, group, part, preset):
        params = {}
        plugin = self.__read_data_item(group, part, "plugin")
        if plugin:
            try:
                module = self.__load_module(plugin)
                param_list = module.get_param_list()
                for param in param_list:
                    if param[1] == "str":
                        self.__add_param(params, group, part, param[0],
                            preset)
                    elif param[1] == "int":
                        self.__add_param_int(params, group, part, param[0],
                            preset)
                    elif param[1] == "img":
                        self.__add_param_image(params, group, part, param[0],
                            preset)
                    elif param[1] == "fnt":
                        self.__add_param_font(params, group, part, param[0],
                            preset)
                return module.get_view(self.canvas, **params)
            except Exception, e:
                WARNING("Error creating custom gauge view %s.%s: %s" % \
                    (group, part, e))

    def __load_screen_plugin(self, plugin, scr_edje, scr_name):
        try:
            module = self.__load_module(plugin)
            return module.get_screen(self.file, scr_edje, scr_name,
                self.metric)
        except Exception, e:
            WARNING("error loading module %s: %s" % (plugin, e))

    def load(self, group):
        """
        Load screen group and create smartobject
        @param group: edje group
        """
        scr = None
        scr_edje = edje.Edje(self.canvas, file=self.file, group=group)
        plugin = edje.file_data_get(self.file, "plugin:" + group +
            ".filename")
        if plugin:
            scr = self.__load_screen_plugin(plugin, scr_edje, group)
        if not scr:
            scr = ScreenController(scr_edje)
        parts = edje.file_data_get(self.file, group + ".parts")
        if not parts:
            return scr

        for part in parts.split():
            pid = self.__read_data_item(group, part, "pid")
            if not pid:
                WARNING("Sensor PID not specified for %s.%s" % (group, part))
                continue

            sensor = self.sensors.get_sensor(pid)
            if not sensor:
                WARNING("Sensor PID \"%s\" unknown for %s.%s" % (pid, group,
                    part))
                continue

            type = self.__read_data_item(group, part, "type")
            if not type:
                WARNING("Object type not specified for %s.%s" % (group, part))
                continue
            type = type.lower()

            ctrl = GaugeController(scr_edje, sensor, self.metric)
            ctrl.set_title(self.__read_data_item(group, part, "title"))
            ctrl.set_unit(self.__read_data_item(group, part, "unit"))
            ctrl.set_min(self.__read_data_item(group, part, "min_metric"),
                self.__read_data_item(group, part, "min_imperial"))
            ctrl.set_max(self.__read_data_item(group, part, "max_metric"),
                self.__read_data_item(group, part, "max_imperial"))
            ctrl.set_frequency(self.__read_data_item(group, part,
                "frequency"))
            ctrl.set_offset(self.__read_data_item(group, part, "offset"))

            preset = self.__read_data_item(group, part, "preset")
            if type == "bar":
                view = self.__create_bar_view(group, part, preset)
            elif type == "arcbar":
                view = self.__create_arcbar_view(group, part, preset)
            elif type == "analogue":
                ctrl.set_scale(self.__read_data_item(group, part,
                    "scale_metric"), self.__read_data_item(group, part,
                    "scale_imperial"))
                view = self.__create_analogue_view(group, part, preset)
            elif type == "digital":
                view = self.__create_digital_view(scr_edje, group, part,
                    preset)
            elif type == "custom":
                view = self.__create_custom_view(scr_edje, group, part,
                    preset)
            else:
                WARNING("Gauge type \"%s\" unknown for %s.%s" % \
                    (type, group, part))
                continue
            if view:
                ctrl.set_view(part, view)
                scr.add_controller(ctrl)

        return scr

    def get_transition_type(self):
        """ Return transition type """
        return edje.file_data_get(self.file, "transition_type")

    def get_transition_time(self):
        """ Return transition time """
        time = edje.file_data_get(self.file, "transition_time")
        try:
            return float(time)
        except:
            WARNING("Transition time invalid \"%s\"" % time)
            return 1
