#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""
Implements L{BarsView} and L{AccelerationRankView}.

@todo: Document this module.
"""

import evas, edje, math
from datetime import datetime

from common.carlog import DEBUG
from main.mainview import MainView
from common.carmanconfig import CarmanConfig
from models.kbmodel import KeyBoardModel

class BarsView(evas.ClippedSmartObject):
    """
    @todo: Document this class.
    """
    def __init__(self, canvas, theme):
        self.bars = []
        self.theme = theme
        evas.ClippedSmartObject.__init__(self, canvas)

    def set_scale(self, scale):
        self.scale = scale

    def add_bar(self, value, date, time):
        bar = edje.Edje(self.evas, file=self.theme, group="bar")
        self.member_add(bar)
        self.bars.append((bar, value))

        bar.part_text_set("num", str(value))
        bar.part_text_set("date", date)
        bar.part_text_set("time", time)
        bar.show()

        self.changed()

    def move(self, x, y):
        self.x = x
        self.y = y

    def resize(self, w, h):
        self.h = h

    def calculate(self):
        pos = 0
        for bar, value in self.bars:
            width = bar.size[0]
            height = self.h * value / self.scale
            bar.size_set(width, height)
            bar.move(self.x + pos * width, self.y + self.h - height)
            pos += 1

class AccelerationRankView(edje.Edje):
    """
    @todo: Document this class.
    """
    def __init__(self):
        self.max = 0
        self.nguides = 0

        canvas = MainView().get_evas_canvas()
        theme = CarmanConfig().get_current_theme()
        edje.Edje.__init__(self, canvas, file=theme, group="accelerationrank")
        self.signal_callback_add("screen-hidden", "", self.__destroy)
        self.layer_set(25)

        self.bars = BarsView(canvas, theme)
        self.part_swallow("screen", self.bars)

        self.kb_model = KeyBoardModel()
        self.kb_model.add_key_down_cb(self.__key_down_cb)

    def __key_down_cb(self, obj, event):
        if event.key == "Escape":
            self.signal_emit("hide-screen", "")
            return True

        return False

    def __calc_nun_guides(self, max):
        for nguides in [5, 3, 4]:
            mod = max % nguides
            if mod == 0:
                return (max, nguides)
        return self.__calc_nun_guides(max+1)

    def __change_nun_guide(self):
        if self.nguides == 3:
            self.signal_emit("three-guides", "")
        elif self.nguides == 4:
            self.signal_emit("four-guides", "")

    def set_max_range(self, max):
        max = max + (max / 10)
        max = int(math.ceil(max))
        max, self.nguides = self.__calc_nun_guides(max)
        step = max / self.nguides
        self.__change_nun_guide()
        self.part_text_set("guide_max_label", str(max)[:4])
        for i in range(1, self.nguides):
            self.part_text_set("guide%d_label" % i, str(step*i)[:4])
        self.bars.set_scale(max)

    def set_bar(self, values):
        self.set_max_range(max([b for a, b in values]))

        for value in values:
            ref = datetime.fromtimestamp(value[0])
            self.bars.add_bar(value[1], ref.strftime("%d/%m/%Y"),
                ref.strftime("%H:%M:%S"))

    def __destroy(self, *params):
        self.bars.delete()
        self.delete()
        self.kb_model.del_key_down_cb(self.__key_down_cb)
