#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python
#
#  carman-python is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License as
#  published by the Free Software Foundation; either version 3 of the
#  License, or (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program. If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{SpeedAlertModel}.

@var    __SOUND_FILE__: Default sound filename.
"""

import gst, time, dbus
from os import path

from common.carmanconfig import CarmanConfig
from common.carlog import ERROR
from common.singleton import Singleton
from models.obdmodel import OBDModel
from models.gpsmodel import GPSModel

__SOUND_FILE__ = "alert.mp3"


class SpeedAlertModel(Singleton):
    """
    Implements the speed alert feature on Carman. This feature is used to
    alert the driver when the speed is higher than a pre-configured value.
    """

    def __init__(self):
        Singleton.__init__(self)
        self.__play_sound = True
        self.__player = gst.element_factory_make("playbin", "player")
        self.__snd_time = 0

        self.obd_model = OBDModel()
        self.obd_model.add_status_changed_cb(self.__obd_status_changed_cb)
        self.gps_model = GPSModel()
        self.gps_model.add_status_changed_cb(self.__gps_status_changed_cb)
        self.__obd_handle = None

    def __sound_alert(self, cur_speed):
        """
        Checks if it has to play the sound alert according to the current speed.

        @param cur_speed: Current speed
        @type cur_speed: number
        """
        def play_sound():
            try:
                themepath = CarmanConfig().get_current_theme_path()
            except dbus.exceptions.DBusException, err:
                ERROR("Error in get current theme path: %s" % err)
                return
            sound_file = "file://"+path.join(themepath, __SOUND_FILE__)

            try:
                self.__player.set_state(gst.STATE_READY)
                self.__player.set_property('uri', sound_file)
                self.__player.set_property('volume', 10.0)
                self.__player.set_state(gst.STATE_PLAYING)
            except:
                ERROR("Error playing sound alert sound!")
            self.__play_sound = False

        if ((time.time() - self.__snd_time) < 3):
            return

        self.__snd_time = time.time()

        if CarmanConfig().get_speed_alert() == "ON":
            max_speed = CarmanConfig().get_max_speed()
            if ((max_speed - 5) <= 0) or (cur_speed < (max_speed - 5)):
                self.__play_sound = True

            if (cur_speed > max_speed) and self.__play_sound:
                play_sound()

    def __obd_status_changed_cb(self, status):
        """
        Calls the registered methods relative to the OBD status changing.

        @param status: OBD status
        @type status: string
        """
        if status == self.obd_model.CONNECTED:
            self.gps_model.del_data_available_cb(self.__gps_speed_alert_cb)
            self.__obd_handle = self.obd_model.add_data_available_cb( \
                        (("0D", 1, 0), ), self.__obd_speed_alert_cb)
        else:
            self.obd_model.del_data_available_cb(self.__obd_handle)
            self.__obd_handle = None
            if self.gps_model.Status() == self.gps_model.FIXED:
                self.gps_model.add_data_available_cb(self.__gps_speed_alert_cb)

    def __gps_status_changed_cb(self, status):
        """
        Calls the registered methods relative to the GPS status changing.

        @param status: GPS status
        @type status: string
        """
        if not self.__obd_handle:
            if status == self.gps_model.FIXED:
                self.gps_model.add_data_available_cb(self.__gps_speed_alert_cb)
            else:
                self.gps_model.del_data_available_cb(self.__gps_speed_alert_cb)

    def __obd_speed_alert_cb(self, model, pid, *data):
        """
        Calls method __sound_alert sending the current speed as a parameter
        when the OBD speed has changed.

        @param model: Model
        @type model: object
        @param pid: OBD PID
        @type pid: string
        @param data: Set of arguments from callback caller containing the current speed.
        @type data: set of arguments
        """
        self.__sound_alert(data[0])

    def __gps_speed_alert_cb(self, *data):
        """
        Calls method __sound_alert sending the current speed as a paramenter
        when the GPS position has changed.

        @param data: Set of arguments from callback caller containing the current speed.
        @type data: set of arguments.
        """
        self.__sound_alert(data[3]) # speed
