#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{BtModel}.
"""

from common.singleton import Singleton
from common.carlog import DEBUG, ERROR
from common import dbus_bt
from models.dbusmodel import EDbusModel


class BtModel(Singleton):
    """
    This class implements all methods used to access system bluetooth methods
    through serial bluetooth connection created by Carman.
    It uses bluetooth DBus service. For more detailed information, see Bluez
    documentation.

    @raise e: If bluetooth DBus service is down or bluetooth is not present.
    """

    def __init__(self):
        Singleton.__init__(self)

        self._discovery_start_cb = None
        self._name_update_cb = None
        self._device_found_cb = None
        self._discovery_completed_cb = None
        self._error_cb = None
        self._registered = {}

        self._dbusbt = dbus_bt.DBusBT(EDbusModel().get_main_loop(),
                self._bt_error_cb)
        try:
            self._dbusbt.set_disc_cb(self._bt_discovery_started_cb, \
                    self._bt_name_updated_cb, \
                    self._bt_device_found_cb, \
                    self._bt_discovery_completed_cb)
        except Exception, e:
            ERROR("Error initializing bluetooth model '%s'" % e)

    def connect_callbacks(self, discovery_start_cb=None, name_update_cb=None, \
            device_found_cb=None, discovery_completed_cb=None, error_cb=None):
        """
        Connect callbacks to bluetooth discovery callbacks.

        @type discovery_start_cb: callback
        @param discovery_start_cb: Callback function that will be registered into
        Bluetooth DBus callbacks for 'discovery mode'.
        @type name_update_cb: callback
        @param name_update_cb: Callback function that will be registered into
        Bluetooth DBus callbacks for 'discovery mode'.
        @type device_found_cb: callback
        @param device_found_cb: Callback function that will be registered into
        Bluetooth DBus callbacks for 'discovery mode'.
        @type discovery_completed_cb: callback
        @param discovery_completed_cb: Callback function that will be registered into
        Bluetooth DBus callbacks for 'discovery mode'.
        @type error_cb: callback
        @param error_cb: Callback function that will be registered into
        Bluetooth DBus callbacks for 'discovery mode'. This callback will be
        called if something got wrong on bluetooth discovery mode.
        """
        self._discovery_start_cb = callable(discovery_start_cb)\
                and discovery_start_cb or None
        self._name_update_cb = callable(name_update_cb)\
                and name_update_cb or None
        self._device_found_cb = callable(device_found_cb)\
                and device_found_cb or None
        self._discovery_completed_cb = callable(discovery_completed_cb)\
                and discovery_completed_cb or None
        self._error_cb = callable(error_cb) and error_cb or None

    def start_discovery(self):
        """
        Starts bluetooth discovery operation.

        @raise err: When a Bluetooth DBus error occured.
        """
        try:
            self._dbusbt.start_discovery()
        except Exception, err:
            self._dbusbt._disc_completed_signal()
            self._dbusbt.cancel_discovery()
            DEBUG('Got a D-Bus error: %s' % err)

    def cancel_discovery(self):
        """
        Stops bluetooth discovery operation.
        """
        self._dbusbt.cancel_discovery()

    def has_bonding(self, address):
        """
        Returns the bonded bluetooth device address.

        @type address: string
        @param address: bluetooth device address to check if has bonding or
        not.
        @rtype: boolean
        @return: C{True} if has bonding, C{False} otherwise.
        """
        if address == "internal":
            return True
        return self._dbusbt.has_bonding(address)

    def passkey_register(self, address, bonding_cb):
        """
        Register a passkey for a bluetooth device.

        @type address: string
        @param address: remote bluetooth device address.
        @type bonding_cb: callback
        @param bonding_cb: bonding callback function.
        """
        if not callable(bonding_cb):
            return

        if address in self._registered:
            self._registered[address] = None

        passkey = dbus_bt.CarmanPasskeyAgent(EDbusModel().get_main_loop(), \
                    address, bonding_cb)
        passkey.register()
        self._registered[address] = passkey


    def passkey_success(self, address, passwd):
        """
        Sets a password for registered bluetooth device.

        @type address: string
        @param address: remote bluetooth device address.
        @type passwd: string
        @param passwd: password to be registered.
        """
        if address in self._registered:
            self._registered[address].success(passwd)

    def _bt_error_cb(self, error=None):
        """
        Call registered bt_error callback.
        """
        if self._error_cb is not None:
            self._error_cb()

    def _bt_discovery_started_cb(self):
        """
        Call registered discovery_started callback.
        """
        if self._discovery_start_cb is not None:
            self._discovery_start_cb()

    def _bt_discovery_completed_cb(self):
        """
        Call registered callback function when discovery has completed.
        """
        if self._discovery_completed_cb is not None:
            self._discovery_completed_cb()

    def _bt_device_found_cb(self, address):
        """
        Call registered callback when a new device is found.

        @type address: string
        @param address: new bluetooth device address.
        """
        if self._device_found_cb is not None:
            self._device_found_cb(address)

    def _bt_name_updated_cb(self, address, name):
        """
        Call registered callback when a bluetooth address has its name updated.

        @type address: string
        @param address: bluetooth device address which name was updated.
        @type name: string
        @param name: new device's name.
        """
        if self._name_update_cb is not None:
            self._name_update_cb(address, name)
