#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{ProgressBarView}.
"""

import evas

class ProgressBarView(evas.SmartObject):
    """
    L{ProgressBarController}'s view.

    @type   canvas: L{evas.Canvas}
    @param  canvas: Carman canvas.
    """
    def __init__(self, canvas):
        evas.SmartObject.__init__(self, canvas)
        self.x = self.y = 0
        self.last_width = -1
        self.clipper = self.Rectangle(size=(0, 0))
        self.progressbar = self.Image()
        self.progressbar.clip_set(self.clipper)
        self.clipper.show()

    def set_theme(self, theme):
        """
        Updates progress bar view with given theme.

        @type   theme: string
        @param  theme: Theme filename with full path.
        """
        self.progressbar.file_set(theme, "images/download-bar-full")
        self.sx, self.sy = self.progressbar.image_size_get()
        self.progressbar.size_set(self.sx, self.sy)
        self.progressbar.fill_set(0, 0, self.sx, self.sy)
        self.progressbar.show()

    def clip_set(self, obj):
        """
        Clips object into clipper.

        @type   obj: C{evas.Rectangle}
        @param  obj: object to clip.
        """
        self.clipper.clip_set(obj)

    def clip_unset(self):
        """
        Unclips all object inside clipper.
        """
        self.clipper.clip_unset()

    def show(self):
        """
        Shows download progress bar.
        """
        self.clipper.show()

    def hide(self):
        """
        Hides download progress bar.
        """
        self.clipper.hide()

    def move(self, x, y):
        """
        Moves progress bar position.

        @type   x: number
        @param  x: Cartesian progress bar M{X} position.
        @type   y: number
        @param  y: Cartesian progress bar M{Y} position.
        """
        self.clipper.move(x - self.sx / 2, y - self.sy / 2)
        self.progressbar.move(x - self.sx / 2, y - self.sy / 2)

    def resize(self, w, h):
        """
        Resizes the progress bar using the given coordinates. This
        method overrides the L{evas.SmartObject} resize method.

        @type   w: number
        @param  w: Width.
        @type   h: number
        @param  h: Height.
        """
        pass

    def update(self, pos):
        """
        Updates progress bar position.

        @type   pos: number
        @param  pos: Progress bar position to be updated (eg. 3 from 10
                     downloads have finished so far).
        """
        width = int(round(pos * self.sx))
        if width != self.last_width:
            self.last_width = width
            self.clipper.size_set(width, self.sy)
