#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{MapTileView}.
"""

import os, evas
from common.carlog import DEBUG

class MapTileView(evas.SmartObject):
    """
    L{MapTileController}'s view.

    @type   canvas: L{evas.Canvas}
    @param  canvas: Carman canvas.
    @type   controller: L{MapTileController}
    @param  controller: Parent controller window.
    @type   tile_sizex: number
    @param  tile_sizex: Cartesian tile size M{X} position.
    @type   tile_sizey: number
    @param  tile_sizey: Cartesian tile size M{Y} position.
    """
    def __init__(self, canvas, controller, tile_sizex, tile_sizey):
        evas.SmartObject.__init__(self, canvas)
        self.zoom = None
        self.tile_cache = {}
        self.cx = self.cy = 0
        self.last_zoom = None
        self.last_basex = None
        self.last_basey = None
        self.base_changed = False
        self.posx = self.posy = 0
        self.controller = controller
        self.tile_sizex = tile_sizex
        self.tile_sizey = tile_sizey
        self.clipper = self.Rectangle()

    def __request_extra_tiles(self):
        """
        Requests extra tiles.

        @deprecated: This method is not used.
        """
        for x in range(self.sx + 2):
            self.controller.request_tile(self.basex + x - 1,
                self.basey - 1, self.zoom, False)
            self.controller.request_tile(self.basex + x - 1,
                self.basey + self.sy, self.zoom, False)
        for y in range(self.sy):
            self.controller.request_tile(self.basex - 1, self.basey + y,
                self.zoom, False)
            self.controller.request_tile(self.basex + self.sx,
                self.basey + y, self.zoom, False)

        basex = (self.cx * 2 - self.w / 2) / self.tile_sizex
        basey = (self.cy * 2 - self.h / 2) / self.tile_sizey
        for y in range(self.sy):
            for x in range(self.sx):
                self.controller.request_tile(basex + x, basey + y,
                    self.zoom - 1, False)

        basex = (self.cx / 2 - self.w / 2) / self.tile_sizex
        basey = (self.cy / 2 - self.h / 2) / self.tile_sizey
        for y in range(self.sy):
            for x in range(self.sx):
                self.controller.request_tile(basex + x, basey + y,
                    self.zoom + 1, False)

    def __empty_cache(self):
        """
        Called when tile cache is emptied.
        """
        for tile in self.tile_cache.values():
            self.member_del(tile)
            tile.delete()
        self.tile_cache = {}
        self.clipper.hide()

    def __clear_tile_cache(self):
        """
        Called when tile cache is cleared.
        """
        for key, value in self.tile_cache.items():
            zoom, y, x = [int(v) for v in key.split("/")]
            if zoom != self.zoom or x < self.basex - 3 or \
              y < self.basey - 3 or x >= self.basex + self.sx + 3 or \
              y >= self.basey + self.sy + 3:
                self.tile_cache.pop(key)
                self.member_del(value)
                value.delete()
                self.controller.cancel_download(x, y, zoom)
        if not self.tile_cache:
            self.clipper.hide()

    def __create_tile(self, filename, fill=None):
        """
        Called when tile is created.

        @type   filename: string
        @param  filename: Map tile filename.
        @type   fill: tuple
        @param  fill: tile coordinates.
        @rtype: C{evas.Image}
        @return: Map tile.
        """
        tile = self.Image()
        if isinstance(filename, (list, tuple)):
            tile.file_set(*filename)
        else:
            try:
                tile.file_set(filename)
            except Exception, e:
                DEBUG("error loading %s: %s" % (filename, e))
                try:
                    os.remove(filename)
                except:
                    pass
                fill = None
                tile.file_set(self.theme, "images/tile-none")
        tile.clip_set(self.clipper)
        tile.smooth_scale_set(False)
        tile.resize(self.tile_sizex, self.tile_sizey)
        if fill:
            tile.fill_set(*fill)
        else:
            tile.fill_set(0, 0, self.tile_sizex, self.tile_sizey)
        self.clipper.show()
        return tile

    def __get_tile(self, x, y, zoom):
        """
        Returns map tile.

        @type   x: number
        @param  x: Cartesian map M{X} position.
        @type   y: number
        @param  y: Cartesian map M{Y} position.
        @type   zoom: number
        @param  zoom: Map zoom.
        @rtype: C{evas.Image}
        @return: Map tile.
        """
        key = "%d/%d/%d" % (zoom, y, x)
        if key in self.tile_cache:
            return self.tile_cache[key]

        if self.controller.is_valid_tile(x, y, zoom):
            filename = self.controller.request_tile(x, y, zoom, True,
                self.on_tile_received)
            if filename:
                tile = self.__create_tile(filename)
                self.tile_cache[key] = tile
                return tile

            filename, fill = self.controller.request_last_tile(x, y, zoom)
            if filename:
                tile = self.__create_tile(filename, fill)
                self.tile_cache[key] = tile
                return tile

        tile = self.__create_tile((self.theme, "images/tile-none"))
        self.tile_cache[key] = tile
        return tile

    def __update_tiles(self):
        """
        Called when tile is updated.
        """
        for tile in self.tile_cache.values():
            tile.hide()

        for y in range(self.sy):
            py = self.posy + y * self.tile_sizey - self.offsety
            if py < self.posy + self.h:
                for x in range(self.sx):
                    px = self.posx + x * self.tile_sizex - self.offsetx
                    if px < self.posx + self.w:
                        tile = self.__get_tile(self.basex + x,
                            self.basey + y, self.zoom)
                        if tile:
                            tile.move(px, py)
                            tile.show()

        if self.base_changed:
            self.base_changed = False
#            self.__request_extra_tiles()
            self.__clear_tile_cache()

    def set_area(self, map_geometry, gps_geometry):
        """
        Sets L{MapTileView} area.

        @type   map_geometry: tuple
        @param  map_geometry: Tuple of C{(int, int, int ,int)}
        @type   gps_geometry: tuple
        @param  gps_geometry: Tuple of C{(int, int, int ,int)}
        """
        self.gps_geometry = (gps_geometry[0] - map_geometry[0],
            gps_geometry[1] - map_geometry[1], gps_geometry[0] + \
            gps_geometry[2] - map_geometry[0], gps_geometry[1] + \
            gps_geometry[3] - map_geometry[1])

    def set_theme(self, theme):
        """
        Sets L{MapTileView} view with given theme.

        @type   theme: string
        @param  theme: Theme filename with full path.
        """
        self.theme = theme
        self.__empty_cache()

    def is_coord_out(self, x, y):
        """
        Verifies if the given tile coordinates is out of tile.

        @type   x: number
        @param  x: Cartesian map M{X} position.
        @type   y: number
        @param  y: Cartesian map M{Y} position.
        @rtype: boolean
        @return: C{True} if tile coordinates are valid, C{False} otherwise.
        """
        px = self.w / 2 - self.cx + x
        py = self.h / 2 - self.cy + y
        return px < self.gps_geometry[0] or py < self.gps_geometry[1] or \
            px > self.gps_geometry[2] or py > self.gps_geometry[3]

    def set_position(self, x, y, zoom):
        """
        Sets the map tile position.

        @type   x: number
        @param  x: Cartesian map M{X} position.
        @type   y: number
        @param  y: Cartesian map M{Y} position.
        @type   zoom: number
        @param  zoom: Map zoom.
        """
        self.cx = x
        self.cy = y
        self.zoom = zoom
        self.basex, self.offsetx = divmod(x - self.w / 2, self.tile_sizex)
        self.basey, self.offsety = divmod(y - self.h / 2, self.tile_sizey)

        if self.last_basex != self.basex or self.last_basey != self.basey \
          or self.last_zoom != zoom:
            self.last_zoom = zoom
            self.last_basex = self.basex
            self.last_basey = self.basey
            self.base_changed = True

        self.__update_tiles()

    def on_tile_received(self, filename, x, y, zoom):
        """
        Called when map tile download finishes.

        @type   filename:string
        @param  filename: Map tile filename.
        @type   x: number
        @param  x: Cartesian map M{X} position.
        @type   y: number
        @param  y: Cartesian map M{Y} position.
        @type   zoom: number
        @param  zoom: Map zoom.
        """
        if filename:
            key = "%d/%d/%d" % (zoom, y, x)
            tile = self.tile_cache.get(key)
            if tile:
                try:
                    tile.file_set(filename)
                    tile.fill_set(0, 0, self.tile_sizex, self.tile_sizey)
                except Exception, e:
                    DEBUG("error loading %s: %s" % (filename, e))
                    try:
                        os.remove(filename)
                        self.controller.request_tile(x, y, zoom, True,
                            self.on_tile_received)
                    except:
                        pass
            elif self.zoom == zoom:
                x -= self.basex
                y -= self.basey
                if x >= 0 and x < self.sx and y >= 0 and y < self.sy:
                    tile = self.__create_tile(filename)
                    self.tile_cache[key] = tile
                    tile.move(self.posx + x * self.tile_sizex - self.offsetx,
                        self.posy + y * self.tile_sizey - self.offsety)
                    tile.show()

    def reset(self, tile_sizex, tile_sizey):
        """
        Resets tile.

        @type   tile_sizex: number
        @param  tile_sizex: Cartesian tile size M{X} position.
        @type   tile_sizey: number
        @param  tile_sizey: Cartesian tile size M{Y} position.
        """
        self.__empty_cache()
        self.tile_sizex = tile_sizex
        self.tile_sizey = tile_sizey

    def clip_set(self, obj):
        """
        Clips object into clipper.

        @type   obj: C{evas.Rectangle}
        @param  obj: object to clip.
        """
        self.clipper.clip_set(obj)

    def clip_unset(self):
        """
        Unclips all objects inside clipper.
        """
        self.clipper.clip_unset()

    def show(self):
        """
        Shows the C{evas.Rectangle} object.
        """
        if self.tile_cache:
            self.clipper.show()

    def hide(self):
        """
        Hides the C{evas.Rectangle} object.
        """
        self.clipper.hide()

    def move(self, x, y):
        """
        Moves tile position.

        @type   x: number
        @param  x: Cartesian tile M{X} position.
        @type   y: number
        @param  y: Cartesian tile M{Y} position.
        """
        self.posx = x
        self.posy = y
        evas.SmartObject.move(self, x, y)

    def resize(self, w, h):
        """
        Overrides evas.SmartObject resize method.

        @type   w: int
        @param  w: Width.
        @type   h: int
        @param  h: Height.
        """
        self.w = w
        self.h = h
        self.sx = w / self.tile_sizex + 2
        self.sy = h / self.tile_sizey + 2
        self.clipper.resize(w, h)
        if self.zoom is not None:
            self.set_position(self.cx, self.cy, self.zoom)
