#  
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{ManageNewBuddyDlgView}.
"""

import etk
from main.mainview import MainView
from main.basedlgview import BaseDlgView
from models.kbmodel import KeyBoardModel


class ManageNewBuddyDlgView(BaseDlgView):
    """
    Creates and shows new buddy dialog window.

    @type   cb: callback
    @param  cb: Callback used when C{OK} button is clicked.
    """
    def __init__(self, cb):
        canvas = MainView().get_evas_canvas()
        BaseDlgView.__init__(self, "managenewbuddy1", 30)
        self.part_text_set("title", "NEW BUDDY")

        self.name_entry = etk.Entry()
        self.alias_entry = etk.Entry()

        embed = etk.Embed(canvas)
        embed.add(self.name_entry)
        self.__connect_unfocused()
        self.name_entry.on_key_down(self.__key_down)
        self.name_entry.on_mouse_down(self.__mouse_down)
        embed.show_all()
        self.part_swallow("display_label", embed.object)

        embed2 = etk.Embed(canvas)
        embed2.add(self.alias_entry)
        self.alias_entry.on_key_down(self.__key_down)
        self.alias_entry.on_mouse_down(self.__mouse_down)
        embed2.show_all()
        self.part_swallow("display_label2", embed2.object)

        self.name_entry.focus()
        self.cb = callable(cb) and cb or None
        self.signal_callback_add("hide", "", self.__destroy)
        self.signal_callback_add("confirm_bt_clicked", "", self.__confirm_cb)
        self.signal_callback_add("cancel_bt_clicked", "", self.__cancel_cb)

    def focus_out_cb(self, obj):
        """
        Sets focus on window when L{etk.Entry} object loses focus.

        @type   obj: L{etk.Entry}
        @param  obj: etk entry object.
        """
        obj.focus()

    def __mouse_down(self, obj, event):
        """
        Called when mouse button is pressed over the L{etk.Entry} object.

        @type   obj: L{etk.Entry}
        @param  obj: etk entry object.
        @type   event: EventMouseDown
        @param  event: Event mouse down object.
        """
        if obj == self.name_entry:
            self.name_entry_focus()
        else:
            self.alias_entry_focus()

    def __key_down(self, obj, event):
        """
        Handles keys pressed(C{ESC} or C{Enter}).

        @type   obj: object
        @param  obj: Not used.
        @type   event: EventKeyDown
        @param  event: Event key down object.

        @rtype: boolean
        @return: C{True}
        """
        if event.key == "Escape":
            self.__cancel_cb()
        elif event.key == "Return" or event.key == "KP_Enter":
            if obj == self.name_entry:
                self.alias_entry_focus()
            else:
                self.__confirm_cb()
        return True

    def __connect_unfocused(self):
        """
        Connects on_unfocused signal on L{etk.Entry} object.
        """
        self.name_entry.on_unfocused(self.focus_out_cb)
        self.frst_focus = True

    def __disconnect_unfocused(self):
        """
        Disconnects on_unfocused signal on L{etk.Entry} object.
        """
        self.name_entry.disconnect(self.name_entry.UNFOCUSED_SIGNAL,
                self.focus_out_cb)
        self.frst_focus = False

    def __confirm_cb(self, *data):
        """
        Called when the user selects the C{OK} button.

        @type   data: object
        @param  data: Not used.
        """
        if self.frst_focus:
            self.__disconnect_unfocused()
        if self.cb:
            self.cb()

    def __cancel_cb(self, *data):
        """
        Called when the user selects the C{Cancel} button.

        @type   data: object
        @param  data: Not used.
        """
        self.hide(True)

    def __destroy(self, *data):
        """
        Called when the dialog is closed.

        @type   data: object
        @param  data: Not used.
        """
        self.cb = None
        if self.frst_focus:
            self.__disconnect_unfocused()
        self.name_entry.destroy()
        self.alias_entry.destroy()
        KeyBoardModel().set_focus()

    def name_entry_focus(self):
        """
        Sets focus on L{etk.Entry}'s name.
        """
        if not self.frst_focus:
            self.__connect_unfocused()
            self.frst_focus = True
        self.alias_entry.unfocus()
        self.name_entry.focus()

    def alias_entry_focus(self):
        """
        Sets focus on L{etk.Entry}'s alias.
        """
        if self.frst_focus:
            self.__disconnect_unfocused()
        self.name_entry.unfocus()
        self.alias_entry.focus()
