#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{ConnectBuddyController}.
"""

from main.messagedlgctrl import MessageDlgController
from maps.connectbuddyview import ConnectBuddyView
from models.buddymodel import BuddyModel
from models.infosharingmodel import InfoSharingModel


class ConnectBuddyController(object):
    """
    Implements buddy connection features (controls L{ConnectBuddyView} in
    order to display a window to user containing connectable buddies).

    @type   parent: L{MapsController}
    @param  parent: Parent window.
    @ivar   __buddies: List of buddies to connect.
    """

    def __init__(self, parent):
        self._parent = parent
        self.__buddies = []

        self.buddy_model = BuddyModel()
        self.is_model = InfoSharingModel()
        self.is_service = self.is_model.service
        self.view = ConnectBuddyView()
        self.view.signal_callback_add("confirm_bt_clicked", "", \
                self.__confirm_cb)
        self.view.signal_callback_add("cancel_bt_clicked", "", \
                self.__cancel_cb)

    def __confirm_cb(self, obj, item, param):
        """
        Callback called when user presses Confirm button. In this case,
        L{ConnectBuddyView} is closed, and a request authorization message is
        sent for each selected buddy.

        @type   obj: class
        @param  obj: Instance of L{ConnectBuddyView} which originated signal.
        @type   item: string
        @param  item: Signal originated (C{confirm_bt_clicked}).
        @type   param: object
        @param  param: Not used.
        """
        self.view.hide()
        if callable(getattr(self._parent, "hide", None)):
            self._parent.hide(True)

        for buddy in self.__buddies:
            self.is_model.send_request_authorization(buddy)

    def __cancel_cb(self, obj, item, param):
        """
        Callback called when user presses Cancel button. In this case,
        L{ConnectBuddyView} is closed.

        @type   obj: class
        @param  obj: Instance of L{ConnectBuddyView} which originated signal.
        @type   item: string
        @param  item: Signal originated (C{cancel_bt_clicked}).
        @type   param: object
        @param  param: Not used.
        """
        self.view.hide()

    def __buddy_selected_cb(self, obj, item, param):
        """
        Adds selected buddy to L{__buddies} list. If the number of selected
        buddies reaches L{BuddyModel.MAX_BUDDIES}, user is not able to
        select another buddy to connect.

        @type   obj: class
        @param  obj: Instance of L{ConnectBuddyView} which originated signal.
        @type   item: string
        @param  item: Signal originated (C{cancel_bt_clicked}).
        @type   param: object
        @param  param: Not used.
        """
        connected_buddies = self.buddy_model.get_connected_buddies()
        if param not in connected_buddies:
            if param not in self.__buddies:
                if (len(self.__buddies) + len(connected_buddies) \
                        + len(self.buddy_model.requests)) \
                        < BuddyModel.MAX_BUDDIES:
                    self.__buddies.append(param)
                else:
                    item.signal_emit("unchecked-button", "")
                    msg = MessageDlgController()
                    msg.show_message("You have reached the maximum<br>" \
                        "amount of 4 friends to<br>share information", \
                        title="CONNECT BUDDY")
            else:
                item.signal_emit("unchecked-button", "")
                self.__buddies.remove(param)
        else:
            msg = MessageDlgController()
            msg.show_message("Buddy is already connected", \
                title="CONNECT BUDDY")

    def load_list(self):
        """
        Verifies if all exceptions are validated before displaying list of
        connectable buddies.

        @rtype: boolean
        @return: C{True} if all exceptions are validated, C{False} otherwise.
        """
        account = self.is_service.get_accounts_active()
        if not account:
            self.view.hide(True)
            msg = MessageDlgController()
            msg.show_message("No account currently active", \
                    title="CONNECT BUDDY")
            return False

        buddies = self.is_service.get_buddies_online()
        if not buddies:
            self.view.hide(True)
            msg = MessageDlgController()
            msg.show_message("No buddies online", title="CONNECT BUDDY")
            return False

        for buddy in buddies:
            checked = False
            if self.buddy_model.buddy_exists(buddy[0]):
                checked = True
            has_alias = lambda alias: alias if alias else buddy[0]
            self.view.add_item(has_alias(buddy[1]), self.__buddy_selected_cb, \
                buddy[0], check=checked)

        if self.view.has_items():
            return True
        else:
            self.view.hide(True)
            msg = MessageDlgController()
            msg.show_message("No buddies left to connect", \
                title="CONNECT BUDDY")
            return False

    def show(self):
        """
        Shows L{ConnectBuddyView}.
        """
        self.view.show()
