#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{LoginAccountView}.
"""

import etk
from main.basedlgview import BaseDlgView
from main.mainview import MainView
from models.kbmodel import KeyBoardModel


class LoginAccountView(BaseDlgView):
    """
    Creates and shows login account dialog window.

    @type   title: string
    @param  title: Dialog title.
    @type   cb: callback
    @param  cb: Callback used when C{OK} button is clicked.
    """

    def __init__(self, title, cb):
        canvas = MainView().get_evas_canvas()
        self.title = title
        BaseDlgView.__init__(self, "manageaccountdlg2", 30)
        self.part_text_set("title", self.title)

        self.password_entry = etk.Entry()
        self.password_entry.password_mode = True
        self.remember = False
        self.signal_callback_add("remember_password_clicked", "",
                    self.__remember_selected_cb)
        self.remember_rect = self.part_object_get('remember_password')
        self.remember_rect.on_key_down_add(self.__key_down)
        self.remember_rect.on_mouse_down_add(self.__mouse_down)

        embed = etk.Embed(canvas)
        embed.add(self.password_entry)
        self.__connect_unfocused()
        self.password_entry.on_key_down(self.__key_down)
        self.password_entry.on_mouse_down(self.__mouse_down)
        embed.show_all()
        self.part_swallow("display_label", embed.object)
        self.password_entry.focus()

        self.part_text_set("cancel_bt_label", "Cancel")
        self.part_text_set("confirm_bt_label", "Ok")

        self.cb = callable(cb) and cb or None
        self.signal_callback_add("hide", "", self.__destroy)
        self.signal_callback_add("confirm_bt_clicked", "", self.__confirm_cb)
        self.signal_callback_add("cancel_bt_clicked", "", self.__cancel_cb)

    def focus_out_cb(self, obj):
        """
        Sets focus on window when L{etk.Entry} object loses focus.

        @type   obj: L{etk.Entry}
        @param  obj: etk entry object.
        """
        obj.focus()

    def __mouse_down(self, obj, event):
        """
        Called when mouse button is pressed over the L{etk.Entry} object.

        @type   obj: L{etk.Entry}
        @param  obj: etk entry object.
        @type   event: EventMouseDown
        @param  event: Event mouse down object.
        """
        if obj == self.password_entry:
            self.password_entry_focus()
        else:
            self.remember_focus()

    def __key_down(self, obj, event):
        """
        Handles keys pressed(C{ESC} or C{Enter}).

        @type   obj: object
        @param  obj: Not used.
        @type   event: EventKeyDown
        @param  event: Event key down object.

        @rtype: boolean
        @return: C{True}
        """
        if event.key == "Escape":
            self.__cancel_cb()
        elif event.key == "Return" or event.key == "KP_Enter":
            if obj == self.password_entry:
                self.remember_focus()
            else:
                self.__confirm_cb()
        elif obj == self.remember_rect and event.key == "space":
            self.__remember_selected_cb()
        return True

    def __connect_unfocused(self):
        """
        Connects C{on_unfocused} signal on L{etk.Entry} object.
        """
        self.password_entry.on_unfocused(self.focus_out_cb)
        self.frst_focus = True

    def __disconnect_unfocused(self):
        """
        Disconnects C{on_unfocused} signal on L{etk.Entry} object.
        """
        self.password_entry.disconnect(self.password_entry.UNFOCUSED_SIGNAL,
                self.focus_out_cb)
        self.frst_focus = False

    def __confirm_cb(self, *data):
        """
        Called when the user selects the C{OK} button.

        @type   data: object
        @param  data: Not used.
        """
        if self.frst_focus:
            self.__disconnect_unfocused()
        if self.cb:
            self.cb(self)

    def __cancel_cb(self, *data):
        """
        Called when the user selects the C{Cancel} button.

        @type   data: object
        @param  data: Not used.
        """
        self.hide(True)

    def __remember_selected_cb(self, *data):
        """
        Called when the user clicks on the remember password checkbox.

        @type   data: object
        @param  data: Not used.
        """
        if self.remember:
            self.signal_emit("disable-remember", "")
        else:
            self.signal_emit("enable-remember", "")
        self.remember = not self.remember

    def __destroy(self, *data):
        """
        Called when the dialog is closed.

        @type   data: object
        @param  data: Not used.
        """
        self.cb = None
        if self.frst_focus:
            self.__disconnect_unfocused()
        self.password_entry.destroy()
        KeyBoardModel().set_focus()

    def password_entry_focus(self):
        """
        Sets focus on L{etk.Entry}'s password.
        """
        if not self.frst_focus:
            self.__connect_unfocused()
            self.frst_focus = True
        self.remember_rect.focus_set(False)
        self.password_entry.focus()

    def remember_focus(self):
        """
        Sets focus on L{etk.Entry}'s remember password.
        """
        if self.frst_focus:
            self.__disconnect_unfocused()
        self.password_entry.unfocus()
        self.remember_rect.focus_set(True)
