#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{StatusController}.
"""

from common.carlog import DEBUG
from common.carmanconfig import CarmanConfig
from main.basedlgview import BaseDlgView
from main.configurebt import ConfigureOBD, ConfigureGPS
from maps.connectaccountctrl import ConnectAccountController
from models.connmodel import ConnectionModel
from models.dbusmodel import CarmandDbusModel
from models.gpsmodel import GPSModel
from models.infosharingmodel import InfoSharingModel
from models.kbmodel import KeyBoardModel
from models.obdmodel import OBDModel


class StatusController(object):
    """
    Controls "Status" window features (GPS, OBD, network and InfoShare account
    statuses).
    """

    def __init__(self):
        self.view = BaseDlgView("status")
        self.view.signal_callback_add("gps_bt-clicked", "", \
            self.__gps_button_clicked)
        self.view.signal_callback_add("obd_bt-clicked", "", \
            self.__obd_button_clicked)
        self.view.signal_callback_add("wlan_bt-clicked", "", \
            self.__wlan_button_clicked)
        self.view.signal_callback_add("is_bt-clicked", "", \
            self.__is_button_clicked)
        self.view.signal_callback_add("hide", "", self.__destroy)
        self.view.signal_callback_add("hide-anime", "", self.__destroy)

        self.gps_model = GPSModel()
        self.gps_model.add_status_changed_cb(self.__set_gps_status)

        self.obd_model = OBDModel()
        self.obd_model.add_status_changed_cb(self.__set_obd_status)

        self.con_model = ConnectionModel()
        self.con_model.add_status_changed_cb(self.__set_wlan_status)

        self.is_model = InfoSharingModel()
        if CarmanConfig().is_service:
            self.is_model.add_status_changed_cb(self.__set_is_status)
        else:
            self.view.signal_emit("is_not_installed", "")

        self.kb_model = KeyBoardModel()
        self.kb_model.add_key_down_cb(self.__key_down_cb)

        self.carmand_model = CarmandDbusModel()

    def __set_gps_status(self, status):
        """
        Called when GPS status has changed.

        @type   status: string
        @param  status: GPS status (L{GPSModel.CONNECTING},
                        L{GPSModel.DISCONNECTED}, L{GPSModel.FIXING},
                        L{GPSModel.FIXED}).
        """
        if status == GPSModel.CONNECTING:
            self.view.signal_emit("gps-connecting", "")
        elif status == GPSModel.DISCONNECTED:
            self.view.signal_emit("gps-disconnected", "")
        elif status == GPSModel.FIXING:
            self.view.signal_emit("gps-fixing", "")
        elif status == GPSModel.FIXED:
            self.view.signal_emit("gps-fixed", "")

    def __set_obd_status(self, status):
        """
        Called when OBD status has changed.

        @type   status: string
        @param  status: OBD status (L{OBDModel.CONNECTED},
                        L{OBDModel.CONNECTING}, L{OBDModel.DISCONNECTED}).
        """
        if status == OBDModel.CONNECTED:
            self.view.signal_emit("obd-connected", "")
        elif status == OBDModel.CONNECTING:
            self.view.signal_emit("obd-connecting", "")
        elif status == OBDModel.DISCONNECTED:
            self.view.signal_emit("obd-disconnected", "")

    def __set_wlan_status(self, status):
        """
        Called when network status has changed.

        @type   status: string
        @param  status: Network status (L{ConnectionModel.CONNECTED},
                        L{ConnectionModel.CONNECTING},
                        L{ConnectionModel.IDLE}).
        """
        if status == ConnectionModel.CONNECTED:
            self.view.signal_emit("wlan-connected", "")
        elif status == ConnectionModel.CONNECTING:
            self.view.signal_emit("wlan-connecting", "")
        elif status == ConnectionModel.IDLE:
            self.view.signal_emit("wlan-disconnected", "")

    def __set_is_status(self, status):
        """
        Called when InfoShare status has changed.

        @type   status: string
        @param  status: InfoShare status (L{InfoSharingModel.CONNECTED},
                        L{InfoSharingModel.CONNECTING},
                        L{InfoSharingModel.DISCONNECTED}).
        """
        if status == InfoSharingModel.CONNECTED:
            self.view.signal_emit("is-connected", "")
        elif status == InfoSharingModel.CONNECTING:
            self.view.signal_emit("is-connecting", "")
        elif status == InfoSharingModel.DISCONNECTED:
            self.view.signal_emit("is-disconnected", "")

    def __gps_button_clicked(self, *data):
        """
        Called when GPS button is clicked.

        @type   data: tuple
        @param  data: Not used
        """
        def connect_gps(obj, item, param):
            obj.hide()
            self.gps_model.connect(param)

        status = self.gps_model.Status()
        if status == GPSModel.DISCONNECTED:
            addr = self.carmand_model.GetGPSDevice()
            if addr != "none" or self.carmand_model.GetMode() != "normal":
                self.gps_model.connect(addr)
            else:
                confgps = ConfigureGPS(connect_gps)
                confgps.show()
        elif status == GPSModel.FIXING or status == GPSModel.FIXED:
            self.gps_model.disconnect()

    def __obd_button_clicked(self, *data):
        """
        Called when OBD button is clicked.

        @type   data: tuple
        @param  data: Not used
        """
        def connect_obd(obj, item, param):
            obj.hide()
            self.obd_model.connect(param)

        status = self.obd_model.Status()
        if status == OBDModel.DISCONNECTED:
            addr = self.carmand_model.GetOBDDevice()
            if addr != "none" or self.carmand_model.GetMode() != "normal":
                self.obd_model.connect(addr)
            else:
                confobd = ConfigureOBD(connect_obd)
                confobd.show()
        elif status == OBDModel.CONNECTED:
            self.obd_model.disconnect()

    def __wlan_button_clicked(self, *data):
        """
        Called when network button is clicked.

        @type   data: tuple
        @param  data: Not used
        """
        if self.con_model.Status() != ConnectionModel.CONNECTED:
            self.con_model.try_connect()

    def __is_button_clicked(self, *data):
        """
        Called when InfoShare button is clicked.

        @type   data: tuple
        @param  data: Not used
        """
        if CarmanConfig().is_service:
            if not self.is_model.service:
                self.is_model.instantiate_isservice()
            if self.is_model.account_status != self.is_model.DISCONNECTED:
                self.is_model.service.account_disconnect()
            else:
                connect = ConnectAccountController()
                if connect.exist_account():
                    connect.connect_account()

    def __key_down_cb(self, obj, event):
        """
        Associates key presses with callbacks.

        @type   obj: class
        @param  obj: Not used.
        @type   event: class
        @param  event: Instance of L{ecore.x.EventKeyDown}.
        """
        if event.key == "Escape":
            self.view.signal_emit("hide-anime", "")
            return True
        else:
            return False

    def __destroy(self, *data):
        """
        Called when status view receives a C{hide} or C{hide-anime} signal.

        @type   data: tuple
        @param  data: Not used
        """
        self.gps_model.del_status_changed_cb(self.__set_gps_status)
        self.obd_model.del_status_changed_cb(self.__set_obd_status)
        self.con_model.del_status_changed_cb(self.__set_wlan_status)
        if CarmanConfig().is_service:
            self.is_model.del_status_changed_cb(self.__set_is_status)
        self.kb_model.del_key_down_cb(self.__key_down_cb)

    def show(self):
        """
        Called when status view needs to be shown.
        """
        self.__set_gps_status(self.gps_model.Status())
        self.__set_obd_status(self.obd_model.Status())
        self.__set_wlan_status(self.con_model.Status())
        if CarmanConfig().is_service:
            self.__set_is_status(self.is_model.account_status)
        self.view.show()
