#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{ListItemView}, L{ListContainer} and L{ListView}.

@var    __TIME_ANIM__: Default timer value for list container "move" animation.
"""

import evas, edje, ecore, time
from common.carlog import DEBUG
from main.mainview import MainView
from models.kbmodel import KeyBoardModel
from common.carmanconfig import CarmanConfig

__TIME_ANIM__ = 0.35

class ListItemView(edje.Edje):
    """
    Handles an individual list item view from the L{ListView} list items.

    @type   controller: class
    @param  controller: Class which controls this list item view.
    @type   canvas: class
    @param  canvas: Edje canvas.
    @type   theme: string
    @param  theme: Current theme filename.
    @type   group: string
    @param  group: Name of the Edje group.
    @type   label: string
    @param  label: Label to be applied on the list item.
    @type   cb: callback
    @param  cb: Callback to be called when the list item is selected.
    @type   param: tuple
    @param  param: Parameters to be passed to C{cb} when exectued.
    @type   toggled: boolean
    @param  toggled: C{True} it the list item checkbox is toggled, C{False} if
                     not.
    @ivar   checked: C{True} if list item checkbox is checked, C{False} if
                     unchecked.
    @ivar   thumbnail: List item thumbnail image, when applicable.
    @ivar   cb: Callback to be called when list item is selected.
    @ivar   __cb_param: Parameters passed to L{cb} when called.
    """

    def __init__(self, controller, canvas, theme, group, label, cb, param, toggled):
        edje.Edje.__init__(self, canvas, file=theme, group=group)

        self.canvas = canvas
        self.checked = False
        self.toggled = toggled
        self.thumbnail = None
        self.controller = controller
        self.set_label(label)
        if callable(cb):
            self.cb = cb
            self.param = param
            self.signal_callback_add("item-pressed", "", self.__item_pressed)
            self.signal_callback_add("item-selected", "", self.__item_selected)

    def __del__(self):
        """
        L{ListItemView}'s destructor.
        """
        DEBUG("deleting ListItemView %s" % self)
        if self.thumbnail is not None:
            self.thumbnail.delete()

    def __item_pressed(self, *params):
        """
        Called when the list item checkbox is pressed.

        @type   params: tuple
        @param  params: Not used.
        """
        if self.checked and self.toggled:
            self.checked = False
            self.signal_emit("unchecked-button-select", "")
            return

        if not self.checked:
            self.controller.uncheck_items()
        self.controller.signal_emit("show-click-off", "")
        self.signal_emit("start-anime", "")

    def __item_selected(self, *params):
        """
        Called when the list item is selected.

        @type   params: tuple
        @param  params: Not used.
        """
        self.controller.signal_emit("hide-click-off", "")
        self.cb(self.controller, self, self.param)

    def set_label(self, label):
        """
        Applies the given text label on the list item.

        @type   label: string
        @param  label: Text label to be applied.
        """
        self.part_text_set("label", label.encode("utf-8"))

    def is_checked(self):
        """
        Verifies if the list item is checked.

        @rtype: boolean
        @return: C{True} if the list item is checked, C{False} otherwise.
        """
        return self.checked

    def set_checkbox(self, value):
        """
        Checks or unchecks the list item, according to the given value.

        @type   value: boolean
        @param  value: C{True} if list item is checked, C{False} if unchecked.
        """
        if self.checked == value:
            return
        self.checked = value
        if value:
            self.signal_emit("checked-button", "")
        else:
            self.signal_emit("unchecked-button", "")

    def set_thumbnail(self, image):
        """
        Applies the given thumbnail image on the list item.

        @type   image: string
        @param  image: Thumbnail image filename.
        """
        self.thumbnail = self.canvas.Image(file=image)
        w, h = self.thumbnail.image_size_get()
        self.thumbnail.size_set(w, h)
        self.thumbnail.fill_set(0, 0, w, h)
        self.part_swallow("thumbnail", self.thumbnail)

    def set_tag(self, tag):
        """
        Applies the given text tag on the list item.

        @type   tag: string
        @param  tag: Text tag to be applied.
        """
        self.part_text_set("tag", tag.encode("utf-8"))


class ListContainer(evas.SmartObject):
    """
    Handles a container which stores a fixed number of list items,
    according to their properties.

    @type   canvas: class
    @param  canvas: Edje canvas
    @type   parent: class
    @param  parent: Instance of L{ListView}.
    @ivar   height: Height size (in pixels) of the list container clipper.
    @ivar   moving: C{True} if container is moving list items, C{False} if not.
    """

    def __init__(self, canvas, parent):
        evas.SmartObject.__init__(self, canvas)
        self.items = []
        self.posx = self.posy = 0
        self.__pos = 0
        self.anim = 0
        self.height = 0
        self.moving = False
        self._parent = parent
        self.clipper = self.Rectangle()

    def __anim_move(self, data):
        """
        Called by list container to animate the "moving" effect.

        @type   data: tuple
        @param  data: Tuple of time data (current time, animation end time).
        @rtype: boolean
        @return: C{False} if moving animation has finished, C{True} otherwise.
        """
        init_anim, acce = data
        part_time = time.time() - init_anim
        anim_current = int(self.__pos * (part_time / acce))

        if abs(anim_current) >= abs(self.__pos):
            self.moving = False
            if self._parent._move is None:
                anim_current = self.__pos
                self.__pos = 0
            else:
                self.__pos = anim_current - self.__pos

        offset = abs(anim_current) - self.anim
        if anim_current < 0:
            offset *= -1

        for item in self.items:
            posx, posy = item.pos
            item.move(posx, posy + offset)
        self.anim = abs(anim_current)

        if not self.moving:
            self._parent.end_anim()
            return False

        return self.moving

    def destroy(self):
        """
        Called when the list container is destroyed.
        """
        self._parent = None
        self.clear()
        self.delete()

    def clear(self):
        """
        Clears list container items.
        """
        for item in self.items:
            self.member_del(item)
            item.delete()
        self.items = []
        self.clipper.hide()

    def clip_set(self, obj):
        """
        Clips object into area.

        @type   obj: C{evas.Rectangle}
        @param  obj: Object to clip.
        """
        self.clipper.clip_set(obj)

    def clip_unset(self):
        """
        Unsets all clipped objects from area.
        """
        self.clipper.clip_unset()

    def show(self):
        """
        Shows the C{evas.Rectangle} object.
        """
        if self.items:
            self.clipper.show()

    def hide(self):
        """
        Hides the C{evas.Rectangle} object.
        """
        self.clipper.hide()

    def move(self, x, y):
        """
        Moves the C{evas.Rectangle} object.

        @type   x: number
        @param  x: M{X} position.
        @type   y: number
        @param  y: M{Y} position.
        """
        self.posx = x
        self.posy = y
        evas.SmartObject.move(self, x, y)

    def resize(self, w, h):
        """
        Resizes the list container clipper using the given coordinates. This
        method overrides the L{evas.SmartObject} resize method.

        @type   w: number
        @param  w: Width.
        @type   h: number
        @param  h: Height.
        """
        self.height = h
        self.clipper.resize(w, h)

    def add_item(self, item):
        """
        Adds a list item into the list container.

        @type   item: class
        @param  item: Instance of L{ListItemView} to be added.
        """
        if self.items:
            pos = self.items[-1].pos[1] + self.items[-1].size[1]
        else:
            pos = self.posy
        self.clipper.show()
        self.member_add(item)
        self.items.append(item)
        item.clip_set(self.clipper)
        item.move(self.posx, pos)
        item.show()

    def remove_item(self, item):
        """
        Removes the given list item from the list container.

        @type   item: class
        @param  item: Instance of L{ListItemView} to be removed.
        @rtype: boolean
        @return: C{True} if list item was found and removed, C{False} otherwise.
        """
        if item not in self.items:
            return False

        height = item.size[1]
        idx = self.items.index(item)
        self.items.remove(item)
        self.member_del(item)
        item.delete()
        if self.items:
            for item in self.items[idx:]:
                posx, posy = item.pos
                item.move(posx, posy - height)
        else:
            self.clipper.hide()
        self.changed()
        return True

    def is_moving(self):
        """
        Verifies if the list container is moving.

        @rtype: boolean
        @return: C{True} if list container is moving, C{False} otherwise.
        """
        return self.moving

    def uncheck_items(self):
        """
        Unchecks all checkboxes from the list container items.
        """
        for item in self.items:
            if item.is_checked:
                item.set_checkbox(False)

    def move_items(self, pos, acce=0):
        """
        Moves list container items according to the given position (in pixels)
        and acceleration (in seconds).

        @type   pos: number
        @param  pos: Used to notify the container how much from the current
                     position it has to move.
        @type   acce: number
        @param  acce: Time (in seconds) in which the moving animation will
                      last.
        """
        self.anim = 0
        self.moving = True
        acce = __TIME_ANIM__ - acce
        self.__pos = pos - self.__pos
        self.changed()
        ecore.animator_add(self.__anim_move, (time.time(), acce))

    def has_items(self):
        """
        Verifies if the list container has items.

        @rtype: boolean
        @return: C{True} if the list container has items, C{False} if not.
        """
        return len(self.items) > 0

    def calculate(self):
        """
        Calls L{ListView.update_scroll}'s method to update its scroll bars.
        """
        self._parent.update_scroll()


class ListView(edje.Edje):
    """
    List item main view. Uses L{ListContainer} to store its L{ListItemView}
    items.

    @type   screen: string
    @param  screen: Name of the Edje screen group
    @type   item: string
    @param  item: Name of the Edje item group
    @type   title: string
    @param  title: Title to be applied on the list view.
    """

    def __init__(self, screen, item, title):
        self.canvas = MainView().get_evas_canvas()
        self.theme = CarmanConfig().get_current_theme()
        edje.Edje.__init__(self, self.canvas, file=self.theme, group=screen)
        self.group = item
        self.item_pos = 0
        self.item_height = 0
        self.items_size = 0
        self.acce_anin = 0
        self.part_text_set("title", title)

        self.layer_set(20)
        self.container = ListContainer(self.canvas, self)
        self.part_swallow("container", self.container)

        self.kb_model = KeyBoardModel()
        self.kb_model.add_key_down_cb(self.__key_down_cb)

        self.signal_callback_add("scroll-up-pressed", "",
                                    self.__mouse_move_up)
        self.signal_callback_add("scroll-down-pressed", "",
                                    self.__mouse_move_down)
        self.signal_callback_add("list-hidden", "", self.destroy)
        self.on_mouse_up_add(self.__on_mouse_up)
        self._move = None

    def update_scroll(self):
        """
        Updates scroll bar, according to the disposal of list item elements on
        the list container.
        """
        if self.item_pos < 0:
            self.signal_emit("enable-scroll-up", "")
        else:
            self.signal_emit("disable-scroll-up", "")
        if self.item_pos + self.items_size > self.container.height:
            self.signal_emit("enable-scroll-down", "")
        else:
            self.signal_emit("disable-scroll-down", "")

    def __on_mouse_up(self, *params):
        """
        Called when mouse is up.

        @type   params: tuple
        @param  params: Not used.
        """
        self._move = None
        self.acce_anin = 0

    def __mouse_move_up(self, *params):
        """
        Called when the mouse button is clicked over up scroll.

        @type   params: tuple
        @param  params: Not used.
        """
        self._move = self.__move_up
        self.__move_up()

    def __mouse_move_down(self, *params):
        """
        Called when the mouse button is clicked over down scroll.

        @type   params: tuple
        @param  params: Not used.
        """
        self._move = self.__move_down
        self.__move_down()

    def __move_up(self, *params):
        """
        Moves the list container items up.

        @type   params: tuple
        @param  params: Not used.
        """
        if not self.container.is_moving() and self.item_pos < 0:
            self.item_pos += self.item_height
            self.container.move_items(self.item_height, self.acce_anin)
            if not self.item_pos < 0:
                self._move = None

    def __move_down(self, *params):
        """
        Moves the list container items down.

        @type   params: tuple
        @param  params: Not used.
        """
        if not self.container.is_moving() and\
                self.item_pos + self.items_size > self.container.height:
            self.item_pos -= self.item_height
            self.container.move_items(-self.item_height, self.acce_anin)
            if not self.item_pos + self.items_size > self.container.height:
                self._move = None

    def end_anim(self):
        if callable(self._move):
            if self.acce_anin < 0.25:
                self.acce_anin += 0.05
            self._move()

    def destroy(self, *params):
        """
        Called when the list view is destroyed.
        """
        self.container.destroy()
        self.delete()
        self.kb_model.del_key_down_cb(self.__key_down_cb)

    def __key_down_cb(self, obj, event):
        """
        Associates key presses with callbacks.

        @type   obj: class
        @param  obj: Not used.
        @type   event: class
        @param  event: Instance of L{ecore.x.EventKeyDown}.

        @rtype: boolean
        @return: C{True} if event is used, C{False} otherwise.
        """
        if event.key == "Up":
            self.__move_up()
            return True
        elif event.key == "Down":
            self.__move_down()
            return True
        elif event.key == "Escape":
            self.signal_emit("button-back-pressed", "")
            self.hide()
            return True

        return False

    def hide(self, hide_now=False):
        """
        Notifies the list view to hide immediately or fading, according to
        the given L{hide_now} parameter.

        @type   hide_now: boolean
        @param  hide_now: C{True} if list view must hide immediately, or
                          C{False} if fading.
        """
        if hide_now:
            self.signal_emit("list-hidden", "")
        else:
            self.signal_emit("hide-list", "")

    def uncheck_items(self):
        """
        Notifies the internal list container to uncheck all checkboxes from
        its list items.
        """
        self.container.uncheck_items()

    def add_item(self, label, cb, param=None, check=None, thumbnail=None,
            tag=None, toggled=False):
        """
        Adds a list item into the internal list container.

        @type   label: string
        @param  label: Label to be applied on the list item.
        @type   cb: callback
        @param  cb: Callback to be called when the list item is selected.
        @type   param: tuple
        @param  param: Parameters to be passed to C{cb} when exectued.
        @type   check: boolean
        @param  check: C{True} if list item checkbox is checked, C{False} if
                         unchecked.
        @type   thumbnail: string
        @param  thumbnail: List item thumbnail image, when applicable.
        @type   tag: string
        @param  tag: Text tag to be applied.
        @type   toggled: boolean
        @param  toggled: C{True} it the list item checkbox is toggled, C{False}
                         if not.
        @rtype: class
        @return: Instance of L{ListItemView} to be added.
        """
        item = ListItemView(self, self.canvas, self.theme, self.group, label,
                            cb, param, toggled)
        if check:
            item.set_checkbox(True)
        if thumbnail is not None:
            item.set_thumbnail(thumbnail)
        if tag is not None:
            item.set_tag(tag)

        self.container.add_item(item)
        self.container.changed()

        self.item_height = item.size[1]
        self.items_size += self.item_height
        return item

    def remove_item(self, item):
        """
        Removes the given list item from the internal list container.

        @type   item: class
        @param  item: Instance of L{ListItemView} to be removed.
        """
        if self.container.remove_item(item):
            self.items_size -= self.item_height

    def has_items(self):
        """
        Verifies if the internal list container has items.

        @rtype: boolean
        @return: C{True} if the list container has items, C{False} if not.
        """
        return self.container.has_items()
