#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{GaugesController}.

@todo: Document this module.
"""

import edje

from main.container import Controller
from main.mainview import MainView
from main.transition import Transition
from models.obdmodel import OBDModel
from models.kbmodel import KeyBoardModel
from gauges.event import EventView
from gauges.screenloader import ScreenLoader
from gauges.optionctrl import OptionController

class GaugesController(Controller):
    """
    @todo: Document this class.
    """
    NAME = "GaugesController"

    def __init__(self, parent, canvas, theme):
        """
        Gauges controller constructor
        @param canvas: Evas canvas object
        @param theme: Theme directory
        """
        self.canvas = canvas
        self._parent = parent
        self.transition = None
        self.mil_status = False

        self.set_theme(theme)

        self.main_view = MainView()
        self.event_view = EventView(canvas)
        self.event_view.set_previous_cb(self.__move_previous)
        self.event_view.set_next_cb(self.__move_next)

        self.model = OBDModel()
        self.model.add_status_changed_cb(self.__obd_status_changed)
        self.mil_handle = self.model.add_data_available_cb((("01", 30), ),
            self.__mil_status_cb)

        self.kb_model = KeyBoardModel()
        self.option_ctrl = OptionController(self)

    def __obd_status_changed(self, status):
        if status == OBDModel.DISCONNECTED:
            self.mil_status = None
            for screen in self.screens:
                screen.reset_values()

    def __mil_status_cb(self, model, pid, *data):
        status = int(data[0]) & 128
        if status != self.mil_status:
            self.mil_status = status
            for screen in self.screens:
                screen.set_dtc_button_status(status)

    def get_name(self):
        return self.NAME

    def get_view(self):
        if self.current:
            return self.current.get_view()

    def get_icon(self):
        return (self.theme_file, "images/icon-gauges")

    def __screen_changed(self):
        """ Transition finalized """
        self.transition = None
        self.last.get_view().hide()
        self.current.activate()

    def __change_screen(self, index, move_to_right):
        """
        Change current screen
        @param index: screen index to change
        """
        scr = self.screens[index]
        if scr != self.current:
            self.current.deactivate()
            self.transiting = True
            self.last = self.current
            self.current = scr
            self.transition = Transition(self.canvas, self.last.get_view(),
                self.current.get_view(), self.transition_time)
            if self.transition_type == "fade":
                self.transition.fade(self.__screen_changed)
            elif self.transition_type == "scroll":
                self.transition.scroll(move_to_right, self.__screen_changed)
            elif self.transition_type == "fall":
                self.transition.fall(self.__screen_changed)
            else:
                self.__screen_changed()

    def __move_next(self):
        """ Move to next screen """
        if self.current and not self.transition:
            index = (self.screens.index(self.current) + 1) % len(self.screens)
            self.__change_screen(index, False)

    def __move_previous(self):
        """ Move to previous screen """
        if self.current and not self.transition:
            index = (self.screens.index(self.current) - 1) % len(self.screens)
            self.__change_screen(index, True)

    def key_down_cb(self, obj, event):
        """ Handle key down event """
        if self.main_view.transiting:
            return False

        if event.key == "Left":
            self.__move_previous()
        elif event.key == "Right":
            self.__move_next()
        elif event.key == "F4":
            self.show_options()
        else:
            return False

        return True

    def set_global_unit(self, system):
        self._parent.set_unit_system(system)

    def set_unit_system(self, value):
        """
        Set unit system to metric or imperial
        @param value: True for metric or False for imperial
        """
        for scr in self.screens:
            scr.set_unit_system(value)

    def set_global_theme(self):
        self._parent.set_theme()

    def set_theme(self, theme):
        """ Load a new theme """
        self.current = self.last = None
        self.screens = []
        self.theme_file = theme
        scr_loader = ScreenLoader(self.canvas, self.theme_file)
        self.transition_type = scr_loader.get_transition_type()
        self.transition_time = scr_loader.get_transition_time()
        groups = edje.file_data_get(theme, "gauges.screens")
        if groups:
            for group in groups.split():
                screen = scr_loader.load(group)
                if screen:
                    self.screens.append(screen)
                    screen.set_dtc_button_status(self.mil_status)
        self.current = self.screens and self.screens[0] or None

    def show_options(self):
        """
        Shows L{OptionController}.
        """
        self.option_ctrl.show()

    def activate(self):
        """ Activate gauges controller with current screen """
        if self.current:
            self.current.activate()
            self.event_view.show()
            self.kb_model.add_key_down_cb(self.key_down_cb)

    def deactivate(self):
        """ Stop transition and OBD sensors """
        if self.transition:
            self.transition.stop()
            self.transition = None
        if self.current:
            self.current.deactivate()
            self.event_view.hide()
        self.kb_model.del_key_down_cb(self.key_down_cb)

    def finalize(self):
        self.model.del_data_available_cb(self.mil_handle)
        self.deactivate()
