#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{DigitalView}.
"""

from common.carlog import DEBUG

class DigitalView(object):
    """
    Digital gauge view.

    @type   screen: Edje object
    @param  screen: Edje group object.
    @type   part_name: string
    @param  part_name: Edje text part name.
    @type   round: int
    @param  round: number to round precision in decimal digits.
    @type   has_unit: string
    @param  has_unit: Show value with unit.
    """
    def __init__(self, screen, part_name, round=0, has_unit="no"):
        self.unit = ""
        self.last_value = 0
        self.screen = screen
        self.part_name = part_name
        self.round = round >= 0 and round or 0
        self.has_unit = has_unit.lower() == "yes"
        self.set_value(0, 0)

    def set_value(self, value_abs, value_rel):
        """
        Changes current value.

        @type   value_abs: int
        @param  value_abs: absolute value to show.
        @type   value_rel: int
        @param  value_rel: relative value (not used).
        """
        self.last_value = value_abs
        value = ("%%.%df" % self.round) % value_abs
        if self.has_unit:
            value += "<unit>" + self.unit + "</unit>"
        self.screen.part_text_set(self.part_name, value.encode("utf-8"))

    def set_unit_system(self, min, max, scale, unit):
        """
        Change unit system values.

        @type   min: int
        @param  min: integer mininum value
        @type   max: int
        @param  max: integer maximum value
        @type   scale: int
        @param  scale: integer scale value
        @type   unit: string
        @param  unit: string unit value
        """
        if self.has_unit:
            self.unit = unit
            self.set_value(self.last_value, 0)

