# profilereader.py - Reader for profile files
#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""
Implements L{SensorReader}.
"""

import os
from common.carlog import WARNING
from common.singleton import Singleton
from common.xmlparser import get_xml_parser
from common.sensors import Sensor

class SensorReader(Singleton):
    """
    This class reads the XML sensor file associated with a profile.

    @type   sensors_file_name: string
    @param  sensors_file_name: Sensors XML filename.
    """

    def __init__(self, sensors_file_name):
        Singleton.__init__(self)
        self._sensors_xml = get_xml_parser().parse(sensors_file_name)
        self._sensors_functions = None
        self._filename = sensors_file_name
        self._sensors = {}

        self._process_sensors_xml()

    def _process_sensors_xml(self):
        """
        Processes the XML sensor file and creates a dictionary of sensors.

        @raise ValueError: Sensor value not found.
        """
        self._load_sensor_functions()

        for sensor in self._sensors_xml.findall("sensor"):
            items = dict(sensor.items())
            if 'pid' not in items or 'description' not in items or \
              'short_description' not in items:
                WARNING('%s conatains an invalid line: %s. Discarting it.'\
                        % (self._filename, items))
                continue
            pid = items['pid']
            desc = items['description']
            short_dec = items['short_description']
            (calc_func, conv_func) = self._get_sensor_functions(pid)
            (unit, mmin, mmax) = self._load_unit_metric(sensor)
            (iunit, imin, imax) = self._load_unit_imperial(sensor)
            try:
                sensor_inst = Sensor(pid, desc, short_dec, [unit, iunit],
                                     [mmin, imin], [mmax, imax], calc_func,
                                     conv_func)
            except ValueError, err:
                WARNING('Ignoring sensor %s - %s' % (pid, err))
                continue
            self._sensors[pid] = sensor_inst

    def _load_sensor_functions(self):
        """
        Loads an external module with custom OBD-II data conversion functions.

        @raise ImportError: Unable to read sensors module.
        @raise AttributeError: Unable to find get_sensor_functions function
                               from the given sensors module.
        """
        items = dict(self._sensors_xml.getroot().items())
        if "code" in items:
            try:
                module = __import__(os.path.join("modules", items["code"]))
                self._sensors_functions = module.get_sensor_functions
            except ImportError, err:
                WARNING('Unable to read sensors %s module %s.'\
                        ' The error was: %s' % (os.path.join("modules",\
                         items["code"], self._filename,err)))
            except AttributeError:
                WARNING('%s from %s has no get_sensor_functions function.' \
                        % (os.path.join("modules", items["code"])),
                        self._filename)
            if not callable(self._sensors_functions):
                WARNING('s from %s provides no get_sensor_functions'\
                        'callable attribute'\
                        % (os.path.join("modules", items["code"])),\
                        self._filename)
                self._sensors_functions = None
            # TODO: add further module verificartions

    def _get_sensor_functions(self, pid):
        """
        Collects the data conversion function for a given PID.

        @type   pid: number
        @param  pid: The hexadecimal OBD-II pid (e.g. 0C).
        @rtype: tuple
        @return: A tuple of sensor functions.
        """
        if self._sensors_functions is not None:
            functions = self._sensors_functions(pid)
            if isinstance(functions, tuple) or \
              isinstance(functions, list) and len(functions) == 2:
                return (functions[0], functions[1])
        return (None, None)

    @staticmethod
    def _load_unit_metric(sensor):
        """
        Loads the metric unit data from the given sensor.

        @type   sensor: object
        @param  sensor: the XML root of the sensor tag.
        @rtype: tuple
        @return: A tuple of metric unit data (C{unit}, C{min}, C{max}).
        """
        metric = sensor.find("metric")

        if metric is None:
            return (None, None, None)

        metric_items = dict(metric.items())
        return (metric_items.get("unit"), metric_items.get("min"),
            metric_items.get("max"))

    @staticmethod
    def _load_unit_imperial(sensor):
        """
        Loads the imperial unit data from the given sensor.

        @type   sensor: object
        @param  sensor: the XML root of the sensor tag.
        @rtype: tuple
        @return: A tuple of imperial unit data (C{unit}, C{min}, C{max}).
        """
        imperial = sensor.find("imperial")

        if imperial is None:
            return (None, None, None)

        imperial_items = dict(imperial.items())
        return (imperial_items.get('unit'), imperial_items.get('min'),
                imperial_items.get('max'))

    def get_sensor(self, pid):
        """
        Returns the L{Sensor} class associated with L{pid}.

        @type   pid: number
        @param  pid: The hexadecimal OBD-II pid (e.g. 0C).
        @rtype: class
        @return: L{Sensor} class associated with L{pid}.
        """
        if pid in self._sensors:
            return self._sensors[pid]
        else:
            return None

    def get_pid_list(self):
        """
        Return the list of PIDs available in the associated sensor XML file.

        @rtype: list
        @return: List of PIDs available.
        """
        return self._sensors.keys()
