# ossohelper.py - Helper to osso functions
#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""
Implements libosso functions.

@var    DEVICE_STATE_NORMAL: Normal device state string.
@var    OSSO_DEVSTATE_MODE_FILE: Filename path with the device state.
@var    __osso_context__: Osso Context object.
@var    __osso_application__: Osso Application object.
@var    __osso_device_state__: Osso Device State object.
"""

import os

from common.carmanconfig import CarmanConfig
from common.carlog import WARNING, INFO, DEBUG

try:
    import osso
    __osso_imported__ = True
except ImportError:
    WARNING('osso module not found - are you running on a desktop?')
    __osso_imported__ = False

DEVICE_STATE_NORMAL     = "normal"
OSSO_DEVSTATE_MODE_FILE = "/tmp/.libosso_device_mode_cache"

__osso_context__ = None
__osso_application__ = None
__osso_device_state__ = None

def application_init(app_name):
    """
    Initializes the application.

    @type   app_name: string
    @param  app_name: Name of the application.

    @raise  Exception: Osso module found but could
                        not be initialized: I{error_name}.
    @raise  Exception: Error creating osso application: I{error_name}.
    """

    global __osso_context__, __osso_device_state__, __osso_application__

    app_version = CarmanConfig().get_app_version()

    if __osso_imported__:
        if has_osso():
            WARNING('osso application was already called. Ignoring...')
            return
        try:
            __osso_context__ = osso.Context(app_name, app_version, False)
        except Exception, err:
            WARNING('osso module found but could not be initialized: %s' % err)
            __osso_context__ = None
            return

        try:
            __osso_application__ = osso.Application(__osso_context__)
        except Exception, err:
            WARNING('error creating osso application: %s' % err)
            __osso_application__ = None

        INFO('osso application init sent - %s v%s' % (app_name, app_version))
        __osso_device_state__ = osso.DeviceState(__osso_context__)

def application_exit():
    """
    Closes the application.

    @raise  Exception: Application end could not be sent: I{error_name}.
    """
    if __osso_application__ is not None and __osso_context__ is not None:
        try:
            __osso_application__.close()
            __osso_context__.close()
        except Exception, err:
            WARNING('application end could not be sent: %s' % err)
        INFO('osso application end sent')

def application_top(app_name):
    """
    Changes the top application. If the application is
    already running, bring it to the foreground.

    @type   app_name: string
    @param  app_name: Name of the application.

    @raise  Exception: Osso application top for I{app_name} sent.
    """
    if __osso_imported__ and __osso_application__:
        try:
            __osso_application__.application_top(app_name)
        except Exception, err:
            WARNING("Error calling application top for %s: %s" %(app_name, err))
        INFO('osso application top for %s sent' % app_name)

def has_osso():
    """
    Checks if the osso module was initialized
    and all objects were created.

    @rtype: boolean
    @return: C{True} it the module was initialized properly,
             C{False} otherwise.
    """
    return __osso_imported__ and \
        not None in ( __osso_context__, __osso_device_state__, __osso_application__)

def display_on():
    """
    Turns on the display.
    """
    if __osso_device_state__ is not None:
        __osso_device_state__.display_state_on()
        __osso_device_state__.display_blanking_pause()
        INFO('osso display on')

def display_blanking_pause():
    """
    Keep the backlight on. Should be called every 45 seconds.
    """
    if __osso_device_state__ is not None:
        __osso_device_state__.display_blanking_pause()
        DEBUG('osso blanking screen')

def get_device_state():
    """
    Returns the device state.

    @rtype: string
    @return: Device state.

    @raise  Exception: Failure to read device state
                        from I{cache_file_name}.
    """
    if __osso_device_state__ is not None:
        cache_file_name = OSSO_DEVSTATE_MODE_FILE + "-" + str(os.getuid())
        try:
            state = os.readlink(cache_file_name)
        except:
            state = None
        if not state:
            DEBUG("Failure to read device state from %s" % cache_file_name)
            state = DEVICE_STATE_NORMAL
        return state
    else:
        return DEVICE_STATE_NORMAL

