# _errors.py - Custom dbus-bt errors
#
# Copyright (C) 2008 Instituto Nokia de Tecnologia - INdT
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Implements methods which handles bluetooth DBus connection exceptions.

@var __DICT_ERROR__: Dictionary of bluetooth DBus connection exceptions.
"""

from dbus.exceptions import DBusException

class BluezDBusException(DBusException):
    """
    Custom Bluez Dbus Exception class.

    @type   reason: string
    @param  reason: Description of exception reason.
    """
    def __init__(self, reason):
        self.reason = reason

    def __str__(self):
        """
        Returns the exception reason.

        @rtype: string
        @return: Exception reason.
        """
        return self.reason

class DBusFailedError(BluezDBusException):
    """
    DBus failed exception class.
    """
    pass

class DBusInvalidArgumentsError(BluezDBusException):
    """
    DBus invalid arguments exception class.
    """
    pass

class DBusNotAuthorizedError(BluezDBusException):
    """
    DBus not authorized exception class.
    """
    pass

class DBusOutOfMemoryError(BluezDBusException):
    """
    DBus out of memory exception class.
    """
    pass

class DBusNoSuchAdapterError(BluezDBusException):
    """
    DBus no such adapter exception class.
    """
    pass

class DBusNotReadyError(BluezDBusException):
    """
    DBus not ready exception class.
    """
    pass

class DBusNotAvailableError(BluezDBusException):
    """
    DBus not available exception class.
    """
    pass

class DBusNotConnectedError(BluezDBusException):
    """
    DBus not connected exception class.
    """
    pass

class DBusConnectionAttemptFailedError(BluezDBusException):
    """
    DBus connection attempt failed exception class.
    """
    pass

class DBusAlreadyExistsError(BluezDBusException):
    """
    DBus already exists exception class.
    """
    pass

class DBusDoesNotExistError(BluezDBusException):
    """
    DBus does not exist exception class.
    """
    pass

class DBusNoReplyError(BluezDBusException):
    """
    DBus no reply exception class.
    """
    pass

class DBusInProgressError(BluezDBusException):
    """
    DBus in progress exception class.
    """
    pass

class DBusNotSupportedError(BluezDBusException):
    """
    DBus not supported exception class.
    """
    pass

class DBusAuthenticationFailedError(BluezDBusException):
    """
    DBus authentication failed exception class.
    """
    pass

class DBusAuthenticationTimeoutError(BluezDBusException):
    """
    DBus authentication timeout exception class.
    """
    pass

class DBusAuthenticationRejectedError(BluezDBusException):
    """
    DBus authentication rejected exception class.
    """
    pass

class DBusAuthenticationCanceledError(BluezDBusException):
    """
    DBus authentication canceled exception class.
    """
    pass

class DBusUnsupportedMajorClassError(BluezDBusException):
    """
    DBus unsupported major class exception class.
    """
    pass

class DBusNotSupportedError(BluezDBusException):
    """
    DBus not supported exception class.
    """
    pass

__DICT_ERROR__ = {'org.bluez.Error.Failed:':DBusFailedError,
                  'org.bluez.Error.InvalidArguments:':DBusInvalidArgumentsError,
                  'org.bluez.Error.NotAuthorized:':DBusNotAuthorizedError,
                  'org.bluez.Error.OutOfMemory:':DBusOutOfMemoryError,
                  'org.bluez.Error.NoSuchAdapter:':DBusNoSuchAdapterError,
                  'org.bluez.Error.NotReady:':DBusNotReadyError,
                  'org.bluez.Error.NotAvailable:':DBusNotAvailableError,
                  'org.bluez.Error.NotConnected:':DBusNotConnectedError,
                  'org.bluez.serial.Error.ConnectionAttemptFailed:':DBusConnectionAttemptFailedError,
                  'org.bluez.Error.AlreadyExists:':DBusAlreadyExistsError,
                  'org.bluez.Error.DoesNotExist:':DBusDoesNotExistError,
                  'org.bluez.Error.InProgress:':DBusInProgressError,
                  'org.bluez.Error.NoReply:':DBusNoReplyError,
                  'org.bluez.Error.NotSupported:':DBusNotSupportedError,
                  'org.bluez.Error.AuthenticationFailed:':DBusAuthenticationFailedError,
                  'org.bluez.Error.AuthenticationTimeout:':DBusAuthenticationTimeoutError,
                  'org.bluez.Error.AuthenticationRejected:':DBusAuthenticationRejectedError,
                  'org.bluez.Error.AuthenticationCanceled:':DBusAuthenticationCanceledError,
                  'org.bluez.serial.Error.NotSupported:':DBusNotSupportedError,
                  'org.bluez.Error.UnsupportedMajorClass:':DBusUnsupportedMajorClassError}

def parse_dbus_error(exception):
    """
    Returns a custom exception based in the DBus exception that was raised.

    @type   exception: string
    @param  exception: DBus exception.
    @rtype: string
    @return: Custom exception message.
    """
    global __DICT_ERROR__

    aux = "%s" % exception
    aux_splt = aux.split(None, 1)
    try:
        return __DICT_ERROR__[aux_splt[0]](aux_splt[1])
    except KeyError:
        return exception
