# carlog.py - logging facilities for carman
#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""
Implements methods used for application debugging. These methods print out to
log stream all debug messages according to their types, which are: L{DEBUG},
L{INFO}, L{WARNING}, L{ERROR}.

@var    NOT_INITIALIZED: Debug status flag (not initialized).
@var    INITIALIZING: Debug status flag (initializing).
@var    INITIALIZED: Debug status flag (initialized).
@var    DEBUG: Method used by the application to print out debug messages.
@var    INFO: Method used by the application to print out info messages.
@var    WARNING: Method used by the application to print out warning messages.
@var    ERROR: Method used by the application to print out error messages.
@var    LOGFORMATS: Dictionary of lof formats {C{COMPACT}, C{NORMAL},
                    C{VERBOSE}}.
@var    LOGLEVELS: List of log levels [L{DEBUG}, L{INFO}, L{WARNING},
                   L{ERROR}].
@var    __state__: Current debug status.
@var    __logstream__: Log stream (standard error/output, file, etc).
@var    __logformat__: Current Log format.
@var    __loglevel__: Current log level.
@var    __msg_buffer_: Debug messages buffer.
@var    __CARMAN_DEBUG__: Carman debug environment variable name, if any.
"""

import sys, os, fcntl, time

(NOT_INITIALIZED, INITIALIZING, INITIALIZED) = range(3)
LOGLEVELS = ['DEBUG', 'INFO', 'WARNING', 'ERROR']
__CARMAN_DEBUG__ = 'CARMAN_DEBUG'
__state__ = NOT_INITIALIZED
__logstream__ = None
__loglevel__ = 2
__msg_buffer__ = []

def setup():
    """
    Configures where the logging messages will be written.
    """
    global __state__
    global __msg_buffer__
    global DEBUG, INFO, WARNING, ERROR

    if __state__ == NOT_INITIALIZED:
        __state__ = INITIALIZING
        from common.carmanconfig import CarmanConfig

        config = CarmanConfig()

        if not config.is_log_enabled():
            __state__ = INITIALIZED
            __msg_buffer__ = []
            DEBUG = INFO = WARNING = ERROR = _get_log_dumb_method
            return

        _setup_log_level(config.get_log_level())
        _setup_log_path(config.get_log_path(), config)
        _setup_log_format(config.get_log_format())

        __state__ = INITIALIZED

        for level, msg in __msg_buffer__:
            if __logstream__ and level >= __loglevel__:
                _write_log(msg)
        __msg_buffer__ = []

def _setup_log_level(loglevel):
    """
    Initializes the loglevel.

    @type   loglevel: number
    @param  loglevel: one of L{LOGLEVELS}.
    """
    global __loglevel__
    if loglevel in LOGLEVELS:
        __loglevel__ = LOGLEVELS.index(loglevel)
    else:
        __loglevel__ = LOGLEVELS.index('WARNING')

def _setup_log_path(logpath, carman_config):
    """
    Initializes logpath. The log path is the path where the log file should be
    saved. If empty, we use the [~user/.carman]. If not informed or an error
    happens while trying to create a log file, we use the stderror.

    @type   logpath: string
    @param  logpath: The path to create a file.
    @type   carman_config: class
    @param  carman_config: Instance of L{CarmanConfig}.
    @except IOError: Unable to open log file.
    """
    global __logstream__
    if logpath is None:
        __logstream__ = sys.stderr
    else:
        if not os.path.dirname(logpath):
            filename = os.path.join(carman_config.get_user_directory(),
                    os.path.splitext(carman_config.get_app_name())[0] \
                        +'.log')
        else:
            filename = os.path.join(logpath,
                    os.path.splitext(carman_config.get_app_name())[0] \
                    +'.log')
        try:
            __logstream__ = open(filename, 'w')
        except IOError, err:
            ERROR('Unable to open %s log file: %s. Using stderr instead.' \
                  % (filename, err))
            __logstream__ = sys.stderr

        filedesc = __logstream__.fileno()
        arg = fcntl.fcntl(filedesc, fcntl.F_GETFL)
        fcntl.fcntl(filedesc, fcntl.F_SETFL, arg | os.O_NONBLOCK)

def _setup_log_format(logformat):
    """
    Initializes the logformat.

    @type   logformat: number
    @param  logformat: one of L{LOGFORMATS}.
    """
    global __logformat__
    if logformat in LOGFORMATS.keys():
        __logformat__ = LOGFORMATS[logformat]
    else:
        __logformat__ = LOGFORMATS['COMPACT']

def finalize():
    """
    Closes any used resource.
    """
    global __logstream__
    global __initialized__

    if __logstream__ is not None and __logstream__ != sys.stderr:
        __logstream__.close()
    __logstream__ = None
    __state__ = NOT_INITIALIZED

def _log_format_compact(level, msg):
    """
    Writes the log message to the stream in compact format.

    @type   level: string
    @param  level: Message log level.
    @type   msg: string
    @param  msg: Log message.
    @rtype: string
    @return: Compact formatted log message.
    """
    format = '%(level)s: %(message)s\n'

    kargs = {
        'level' : level,
        'message' : msg,
    }

    return format % kargs

def _log_format_normal(level, msg):
    """
    Writes the log message to the stream in normal format.

    @type   level: string
    @param  level: Message log level.
    @type   msg: string
    @param  msg: Log message.
    @rtype: string
    @return: Normal formatted log message.
    """
    frame = sys._getframe(2)
    if frame.f_code.co_name != '?':
        frameinfo = '%s:%s' % (os.path.splitext(
                            os.path.basename(frame.f_code.co_filename))[0],
                               frame.f_code.co_name)
    else:
        frameinfo = '%s' % os.path.splitext(
                            os.path.basename(frame.f_code.co_filename))[0]

    format = '%(level)s: %(frameinfo)s - %(message)s\n'

    kargs = {
        'frameinfo' : frameinfo,
        'level' : level,
        'message' : msg,
    }

    return format % kargs

def _log_format_verbose(level, msg):
    """
    Writes the log message to the stream in verbose format.

    @type   level: string
    @param  level: Message log level.
    @type   msg: string
    @param  msg: Log message.
    @rtype: string
    @return: Verbose formatted log message.
    """
    currenttime = time.time()
    frame = sys._getframe(2)

    if frame.f_code.co_name != '?':
        frameinfo = '%s:%s:%s' % (os.path.splitext(
                              os.path.basename(frame.f_code.co_filename))[0],
                               frame.f_code.co_name,
                               frame.f_lineno)
    else:
        frameinfo = '%s:%s' % (os.path.splitext(
                              os.path.basename(frame.f_code.co_filename))[0],
                               frame.f_lineno)

    msecs = (currenttime - long(currenttime)) * 1000
    asctime = time.strftime('%H:%M:%S', time.localtime(currenttime))
    format = '%(asctime)s:%(msecs)03d: %(level)s: %(frameinfo)s -' \
             ' %(message)s\n'

    kargs = {
        'asctime' : asctime,
        'msecs' : msecs,
        'frameinfo' : frameinfo,
        'level' : level,
        'message' : msg,
    }

    return format % kargs

def _write_log(msg):
    """
    Writes the message into the logstream.

    @type   msg: string
    @param  msg: Log message.
    @except IOError: Unable to write log to log stream.
    """
    try:
        __logstream__.write(msg)
        __logstream__.flush()
    except IOError:
        pass

def log_method_call(method):
    """
    Decorator to debug method calls.

    @type   method: object
    @param  method: Method to be decorated.
    @rtype: object
    @return: Method to be called.
    """
    def wrap(*args, **kargs) :
        if __logstream__ is not None and __loglevel__ == 0:
            _write_log(_log_format_verbose('DEBUG', 'CALL: %s.%s(%r, %r)' % \
                               (args[0].__class__.__name__,
                                method.__name__, args[1:], kargs)))
        reply = method(*args, **kargs)
        if __logstream__ is not None and __loglevel__ == 0:
            _write_log(_log_format_verbose('DEBUG', 'RET: %s.%s(%r, %r) = %r' \
                               % (args[0].__class__.__name__,
                                method.__name__, args[1:], kargs, reply)))
        return reply
    if __CARMAN_DEBUG__ in os.environ and os.environ[__CARMAN_DEBUG__] == '1':
        return wrap
    else:
        return method

def log_function_call(function):
    """
    Decorator to debug function calls.

    @type   function: object
    @param  function: Function to be decorated.
    @rtype: object
    @return: Function to be called.
    """
    def wrap(*args, **kargs):
        if __logstream__ is not None and __loglevel__ == 0:
            _write_log(_log_format_verbose('DEBUG', 'CALL: %s.%s(%r, %r)' % \
                               (function.__module__, function.__name__,
                                args, kargs)))
        reply = function(*args, **kargs)
        if __logstream__ is not None and __loglevel__ == 0:
            _write_log(_log_format_verbose('DEBUG', 'RET: %s.%s(%r, %r) = %r' \
                               % (function.__module__, function.__name__,
                                args, kargs, reply)))
        return reply
    if __CARMAN_DEBUG__ in os.environ and os.environ[__CARMAN_DEBUG__] == '1':
        return wrap
    else:
        return function

def _get_log_dumb_method(msg):
    """
    This method will do nothing, that is, ignore the message.

    @type   msg: string
    @param  msg: Log message.
    """
    pass

def _get_log_method(level):
    """
    Wraps a function to the debug, info, warning and error object.

    @type   level: string
    @param  level: Message log level.
    @rtype: object:
    @return: Sub-method which writes the log message.
    """
    def do_log(msg) :
        global __msg_buffer__
        if __state__ == INITIALIZED:
            if __logstream__ is not None and num_level >= __loglevel__:
                _write_log(__logformat__(level, msg))
        elif __state__ == INITIALIZING:
            __msg_buffer__.append((num_level, __logformat__(level, msg)))

    # do_log()
    num_level = LOGLEVELS.index(level)
    return do_log

def get_log_stream():
    """
    Returns the stream that the messages should be sent to.

    @rtype: object
    @return: Log stream object.
    """
    return __logstream__

def get_log_level():
    """
    Returns the loglevel string.

    @rtype: string
    @return: Current log level.
    """
    return LOGLEVELS[__loglevel__]

def get_log_format():
    """
    Returns the current log format string.

    @rtype: string
    @return: Current log format.
    """
    if __logformat__ in LOGFORMATS:
        return LOGFORMATS[__logformat__]

LOGFORMATS = {'COMPACT': _log_format_compact, \
              'NORMAL': _log_format_normal, \
              'VERBOSE': _log_format_verbose,}

__logformat__ = LOGFORMATS['COMPACT']

DEBUG = _get_log_method('DEBUG')
INFO = _get_log_method('INFO')
WARNING = _get_log_method('WARNING')
ERROR = _get_log_method('ERROR')
