/*
   Copyright (C) 2009 Roman Belov

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <math.h>

#include <png.h>

#ifdef OPENGL
#include <GL/gl.h>
#include <GL/glext.h>
#endif

#include "draw.h"
#include "caph.h"

#define	FB_ADDR(x,y)		(((color_t *)screen_pixels) + screen_w * (y) + (x))
#define FB_DRAW(x,y,col)	(*FB_ADDR((x),(y)) = (col))

static int
clip_code(point_t *s)
{
	int		code = 0;
	float		lim_w = (float) (screen_w - 1);
	float		lim_h = (float) (screen_h - 1);

	if (s->x < 0.0f)
		code |= 0x01;
	else if (s->x > lim_w)
		code |= 0x02;
	if (s->y < 0.0f)
		code |= 0x04;
	else if (s->y > lim_h)
		code |= 0x08;

	return code;
}

static int
clip_line(point_t *s, point_t *e)
{
	int		s_cc = clip_code(s);
	int		e_cc = clip_code(e);
	float		lim_w = (float) (screen_w - 1);
	float		lim_h = (float) (screen_h - 1);
	float		dx, dy;

	do {
		if (s_cc & e_cc)
			return -1;

		if (!(s_cc | e_cc))
			return 0;

		if (!s_cc) {

			int tmp = s_cc;
			s_cc = e_cc;
			e_cc = tmp;

			point_t *ptmp = s;
			s = e;
			e = ptmp;
		}

		dx = e->x - s->x;
		dy = e->y - s->y;

		if ((fabsf(dx) < 1.0f) && (s_cc & 0x03))
			return -1;

		if ((fabsf(dy) < 1.0f) && (s_cc & 0x0c))
			return -1;

		if (s_cc & 0x01) {
			s->y = e->y + ((0.0f - e->x) * dy) / dx;
			s->x = 0.0f;
			goto __cl_recalc;
		}

		if (s_cc & 0x02) {
			s->y = s->y + ((lim_w - s->x) * dy) / dx;
			s->x = lim_w;
			goto __cl_recalc;
		}

		if (s_cc & 0x04) {
			s->x = e->x + ((0.0f - e->y) * dx) / dy;
			s->y = 0.0f;
			goto __cl_recalc;
		}

		if (s_cc & 0x08) {
			s->x = s->x + ((lim_h - s->y) * dx) / dy;
			s->y = lim_h;
			goto __cl_recalc;
		}

__cl_recalc:
		s_cc = clip_code(s);
	}
	while (1);
}

static void
draw_int_line(int xs, int ys, int xe, int ye, color_t col)
{
	int	dx, dy;
	int	vx, vy;
	int	i = 0;

	if (xs < xe) {
		dx = xe - xs;
		vx = 1;
	}
	else {
		dx = xs - xe;
		vx = -1;
	}

	if (ys < ye) {
		dy = ye - ys;
		vy = 1;
	}
	else {
		dy = ys - ye;
		vy = -1;
	}

	if (dx < dy) {
		while (ys != ye) {
			FB_DRAW(xs, ys, col);

			ys += vy;
			i += dx;

			if (i >= dy) {
				i -= dy;
				xs += vx;
			}
		}
	}
	else if (dx > dy) {
		while (xs != xe) {
			FB_DRAW(xs, ys, col);

			xs += vx;
			i += dy;

			if (i >= dx) {
				i -= dx;
				ys += vy;
			}
		}
	}
	else {
		while (ys != ye) {
			FB_DRAW(xs, ys, col);

			xs += vx;
			ys += vy;
		}
	}

	FB_DRAW(xe, ye, col);
}

void draw_init()
{
#ifdef _OPENGL
	float w = (float) screen_w;
	float h = (float) screen_h;

	float mx[16] = {
		2.0f / w, 0.0f, 0.0f, 0.0f,
		0.0f, - 2.0f / h, 0.0f, 0.0f,
		0.0f, 0.0f, 0.0f, 0.0f,
		- 1.0f, 1.0f, - 1.0f, 1.0f };

	glViewport(0, 0, screen_w, screen_h);

	glMatrixMode(GL_PROJECTION);
	glLoadMatrixf(mx);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	glEnable(GL_COLOR_MATERIAL);
	glEnable(GL_TEXTURE_2D);

	glDisable(GL_DEPTH_TEST);
	glDisable(GL_CULL_FACE);
#endif
}

typedef struct raw_img raw_img_t;

struct raw_img {
	int width;		/* image width */
	int height;		/* image height */
	void *raw;		/* raw color data */
	int alpha;		/* RGBA or RGB */
};

static raw_img_t bgimg;
#ifdef OPENGL
static GLuint bgimg_id;
#endif


int static
img_load_png(const char *png, raw_img_t *raw)
{
	if (raw == NULL)
		return -1;

	FILE *fp = fopen(png, "rb");
	
	if (fp == NULL) {
		fprintf(stderr, "%s:%i [ERROR] fopen(\"%s\") failed: %s \n",
						__FILE__, __LINE__,
						png, strerror(errno));
		return -1;
	}

	unsigned char sig[8];
	fread(sig, 8, 1, fp);

	if (png_sig_cmp(sig, 0, 8))
	{
		fprintf(stderr, "%s:%i [ERROR] \"%s\" is not png file \n",
						__FILE__, __LINE__, png);
		return -1;
	}

	png_structp png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING,
							NULL, NULL, NULL);
	if (png_ptr == NULL)
	{
		fprintf(stderr, "%s:%i [ERROR] libpng failed \n",
						 __FILE__, __LINE__);
		return -1;
	}
	
	png_infop info_ptr = png_create_info_struct(png_ptr);
	if (info_ptr == NULL)
	{
		fprintf(stderr, "%s:%i [ERROR] libpng failed \n",
						 __FILE__, __LINE__);
		png_destroy_read_struct(&png_ptr, NULL, NULL);
		return -1;
	}

	png_init_io(png_ptr, fp);
	png_set_sig_bytes(png_ptr, 8);
	png_read_info(png_ptr, info_ptr);

	png_uint_32 w, h;
	int bits, ltype;
	png_get_IHDR(png_ptr, info_ptr, &w, &h, &bits, &ltype, 0, 0, 0);

	if (bits > 8)
		png_set_strip_16(png_ptr);
	
	if (ltype == PNG_COLOR_TYPE_PALETTE)
		png_set_palette_to_rgb(png_ptr);
	
	if ((ltype == PNG_COLOR_TYPE_GRAY) ||
		(ltype == PNG_COLOR_TYPE_GRAY_ALPHA))
	{
		if (bits < 8)
			png_set_gray_1_2_4_to_8(png_ptr);
		png_set_gray_to_rgb(png_ptr);
	}

	png_read_update_info(png_ptr, info_ptr);
	png_get_IHDR(png_ptr, info_ptr, ( void* ) &w, ( void* ) &h,
						&bits, &ltype, 0, 0, 0);

	if (((ltype != PNG_COLOR_TYPE_RGB) &&
		(ltype != PNG_COLOR_TYPE_RGB_ALPHA)) ||
		(bits > 8))
	{
		fprintf(stderr, "%s:%i [ERROR] FIXME: Invalid png reading \n",
						__FILE__, __LINE__);
		return -1;
	}

	int alpha = (ltype == PNG_COLOR_TYPE_RGB_ALPHA);
	int bytes = alpha ? 4 : 3 ;

	png_byte *data = malloc(w * h * bytes);
	png_byte **row_pointers = malloc(h * sizeof(unsigned *));

	int i ;
	for (i = 0; i < h; i++)
		row_pointers[i] = data + i * w * bytes;

	png_read_image(png_ptr, row_pointers);

	png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
	free(row_pointers);
	fclose(fp);

	raw->width = w;
	raw->height = h;
	raw->raw = data;
	raw->alpha = alpha;

	return 0;
}

void draw_load_bg(const char *img)
{
	if (img_load_png(img, &bgimg) < 0) {
		fprintf(stderr, "%s:%i [ERROR] Background image not loaded\n",
				__FILE__, __LINE__);
		return ;
	}

#ifdef _OPENGL
	GLuint id ;
	glGenTextures(1, &id);
	glBindTexture(GL_TEXTURE_2D, id);

	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);

	if (bgimg.alpha)
		glTexImage2D(GL_TEXTURE_2D, 0, GL_COMPRESSED_RGBA,
				bgimg.width, bgimg.height,
				0, GL_RGBA, GL_UNSIGNED_BYTE, bgimg.raw);
	else
		glTexImage2D(GL_TEXTURE_2D, 0, GL_COMPRESSED_RGB,
				bgimg.width, bgimg.height,
				0, GL_RGB, GL_UNSIGNED_BYTE, bgimg.raw);

	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);

	free(bgimg.raw);
	bgimg_id = id;
#else
#endif
}

void draw_clear()
{
#ifdef _OPENGL
	glColor4f(1.0f, 1.0f, 1.0f, 1.0f);
	glBindTexture(GL_TEXTURE_2D, bgimg_id);

	glDisable(GL_BLEND);

	glBegin(GL_QUADS);

		glVertex2f(0.0f, 0.0f);
		glTexCoord2f(1.0f, 0.0f);	

		glVertex2f((float) screen_w, 0.0f);
		glTexCoord2f(1.0f, 1.0f);

		glVertex2f((float) screen_w,
				(float) screen_h);
		glTexCoord2f(0.0f, 1.0f);

		glVertex2f(0.0f, (float) screen_h);
		glTexCoord2f(0.0f, 0.0f);

	glEnd();
#else
	uint32_t *ptr;
	uint32_t *fin;
	uint32_t fill;

	fill = 0xffffffff;

	ptr = (uint32_t*) FB_ADDR(0, 0);
	fin = ptr + screen_w * screen_h;

	while (ptr != fin) {
		*ptr++ = fill;
		*ptr++ = fill;
		*ptr++ = fill;
		*ptr++ = fill;
	}
#endif
}

void draw_point(const point_t *s, color_t col)
{
#ifdef _OPENGL
	glColor4ub(col.r, col.g, col.b, 200);
	glBindTexture(GL_TEXTURE_2D, 0);

	glPointSize(5.0f);
	glEnable(GL_BLEND);
	glBlendFunc(GL_ONE, GL_ONE);

	glBegin(GL_POINTS);

		glVertex2fv((void *) s);

	glEnd();
#endif
}

void draw_line(const point_t *s, const point_t *e, color_t col)
{
#ifdef _OPENGL
	glBindTexture(GL_TEXTURE_2D, 0);

	if (col.a == 1) {
		glColor4ub(col.r, col.g, col.b, 100);
		glLineWidth(4.0f);
		glEnable(GL_BLEND);
		glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	}
	else if (col.a == 2) {
		if (col.r < 150 || col.g < 150)
			glColor4ub(col.r, col.g, col.b, 255);
		else
			glColor4ub(255, 255, 0, 255);
		glLineWidth(3.0f);
		glDisable(GL_BLEND);
	}
	else if (col.a == 3) {
		glColor4ub(col.r, col.g, col.b, 100);
		glLineWidth(2.0f);
		glEnable(GL_BLEND);
		glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	}
	else if (col.a == 4) {
		if (col.r < 150 || col.g < 150 || col.b < 150)
			glColor4ub(col.r, col.g, col.b, 60);
		else
			glColor4ub(255, 255, 255, 200);
		glLineWidth(4.0f);
		glEnable(GL_BLEND);
		glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	}
	else {
		glColor4ub(col.r, col.g, col.b, 255);
		glLineWidth(2.0f);
		glDisable(GL_BLEND);
	}

	glBegin(GL_LINES);

		glVertex2fv((void *) s);
		glVertex2fv((void *) e);

	glEnd();	
#else
	point_t		ls, le;

	ls = *s;
	le = *e;

	if (isnan(ls.x) || isnan(ls.y))
		return;

	if (isnan(le.x) || isnan(le.y))
		return;

	if (clip_line(&ls, &le) < 0)
		return;

	draw_int_line(	(int) ls.x,
			(int) ls.y,
			(int) le.x,
			(int) le.y,
			col);
#endif
}

void draw_fade()
{
#ifdef _OPENGL
	glColor4f(0.0f, 1.0f, 0.0f, 0.5f);
	glBindTexture(GL_TEXTURE_2D, 0);

	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

	glBegin(GL_QUADS);

		glVertex2f(0.0f, 0.0f);

		glVertex2f((float) screen_w, 0.0f);

		glVertex2f((float) screen_w,
				(float) screen_h);

		glVertex2f(0.0f, (float) screen_h);

	glEnd();
#endif
}

void draw_fail()
{
#ifdef _OPENGL
	glColor4f(1.0f, 0.0f, 0.0f, 0.5f);
	glBindTexture(GL_TEXTURE_2D, 0);

	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

	glBegin(GL_QUADS);

		glVertex2f(0.0f, 0.0f);

		glVertex2f((float) screen_w, 0.0f);

		glVertex2f((float) screen_w,
				(float) screen_h);

		glVertex2f(0.0f, (float) screen_h);

	glEnd();
#endif
}
