#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import ecore
import edje
import evas.decorators
import edje.decorators

from terra.core.manager import Manager
from terra.ui.base import EdjeWidget, Widget
from terra.ui.scrollbar import Scrollbar
from terra.core.plugin_prefs import PluginPrefs
from terra.ui.list import RowRenderer, KineticList
from terra.core.terra_object import TerraObject

mger = Manager()


class BaseRowRendererWidget(EdjeWidget, RowRenderer, TerraObject):
    terra_type = "Widget/BaseRowRenderer"
    row_group = None

    def __init__(self, parent, theme=None):
        EdjeWidget.__init__(self, parent.evas, self.row_group, parent, theme)
        self._model = None
        self._state = None
        self._selected = None
        self._visible = None

    def theme_changed(self, end_callback=None):
        EdjeWidget.theme_changed(self, end_callback)
        self.force_redraw()

    def force_redraw(self):
        m = self._model
        self._model = None
        self.value_set(m)

    def do_value_set(self, v):
        raise NotImplementedError("do_value_set")

    def value_set(self, v):
        if self._model is v or v is None:
            return
        self.do_value_set(v)
        self._model = v

    state_mapping = {RowRenderer.STATE_FIRST: "state,first",
                     RowRenderer.STATE_LAST: "state,last",
                     RowRenderer.STATE_DEFAULT: "state,default"}
    def state_set(self, state):
        self.signal_emit(self.state_mapping[state], "")
        self._state = state

    def state_hidden(self):
        self.signal_emit("clip,state,hidden", "")
        self._visible = False

    def state_default(self):
        self.signal_emit("clip,state,default", "")
        self._visible = True

    def select(self):
        self.signal_emit("selection,show", "")
        self._selected = True

    def unselect(self):
        self.signal_emit("selection,hide", "")
        self._selected = False

    def blink(self):
        self.signal_emit("selection,blink", "")
        self._selected = False

    @evas.decorators.del_callback
    def _on_delete(self, *ignored):
        self._model = None


class RowRendererWidget(BaseRowRendererWidget):
    terra_type = "Widget/RowRenderer"
    resizable_row_renderer = None

    def resizable_version(self):
        if self.resizable_row_renderer is None:
            return None
        o = self.resizable_row_renderer(self._parent_widget, self._theme)
        o.copy_renderer(self)
        return o


class ResizableRowRendererWidget(BaseRowRendererWidget):
    terra_type = "Widget/ResizableRowRenderer"

    def select(self, end_callback=None):
        def end(*ignored):
            self.signal_callback_del("animate,finished", "", end)
            if end_callback:
                end_callback(self)
        self.signal_callback_add("animate,finished", "", end)
        self.signal_emit("animate", "")

    def copy_renderer(self, renderer):
        self.geometry_set(*renderer.geometry_get())
        self._model = None # force redraw
        if renderer._model is not None:
            self.value_set(renderer._model)
        if renderer._state is not None:
            self.state_set(renderer._state)
        if renderer._visible:
            self.state_default()
        else:
            self.state_hidden()
        if renderer._selected:
            self.select()
        else:
            self.unselect()


class SimpleRowRendererWidgetFactory(TerraObject):
    terra_type = "Widget/SimpleRowRendererFactory"

    def __new__(cls, value_setter, row_group, resizable_row_group=None,
                provide_resizable_row=True):
        if not provide_resizable_row:
            resizable_renderer = None
        else:
            if resizable_row_group is None:
                resizable_row_group = row_group + "_resizable"

            def get_resizable_row_renderer(edje, value_setter):
                class ResizableRowRenderer(ResizableRowRendererWidget):
                    row_group = edje
                    do_value_set = value_setter
                return ResizableRowRenderer

            resizable_renderer = get_resizable_row_renderer(resizable_row_group,
                                                            value_setter)

        def get_row_renderer(edje, value_setter, resizable_renderer):
            class RowRenderer(RowRendererWidget):
                row_group = edje
                resizable_row_renderer = resizable_renderer
                do_value_set = value_setter
            return RowRenderer

        return get_row_renderer(row_group, value_setter, resizable_renderer)


class KineticListWidget(KineticList, Widget, TerraObject):
    terra_type = "Widget/KineticList"
    hold_const = 1.1

    def __init__(self, parent, renderer_new, elements, theme=None):
        Widget.__init__(self, parent, theme)

        height = self.row_renderer_height_get(renderer_new)
        KineticList.__init__(self, parent.evas, renderer_new, elements, height)
        self._resizable_renderer = None

        self.callback_on_mouse_up = None
        self.callback_on_mouse_down = None
        self.callback_on_mouse_move = None
        self.callback_on_item_selected = None
        self.callback_on_show_selection = None

        self.hold_cb = None
        self.hold_timer = None
        self.hold_t_ok = False

        prefs = PluginPrefs("settings")
        self.click_constant = int(prefs.get("kinetic_list_click_constant", 25))
        self.move_constant = int(prefs.get("kinetic_list_move_constant", 20))
        self.click_init_time = float(prefs.get("kinetic_list_click_selection_time", 0.15))
        self.click_block_time = float(prefs.get("kinetic_list_click_block_time", 0.30))

    def row_renderer_set(self, renderer_new):
        if self._selection is not None:
            self._selection.delete()

        self.renderer_new = renderer_new
        self.renderer_height = self.row_renderer_height_get(renderer_new)

        for renderer in self.renderers:
            renderer.delete()

        # recreate renderers
        self.renderers = []
        self.model_updated()

    def row_renderer_height_get(self, renderer):
        row_renderer = renderer(self._parent_widget)

        self._selection = row_renderer.resizable_version()
        if self._selection is not None:
            self.member_add(self._selection)
            height = row_renderer.size_min[1]

        height = row_renderer.size_min[1]
        row_renderer.delete()

        return height

    def theme_changed(self):
        for r in self.renderers:
            r.theme_changed()
        self._refill_renderers()
        if self._selection is not None:
            self._selection.hide()
            self._selection.theme_changed()

    def _hold_t_func(self, idx):
        if self.hold_cb and self.hold_cb(idx):
            self.blink_selection()
            self.hold_t_ok = True
        self.hold_timer.delete()
        self.hold_timer = None

    def _cb_on_init_click(self, obj, event):
        if self._is_click_possible(self.actual_pos_y):
            idx = self.index_at_y(self.actual_pos_y)
            if idx >= 0:
                self.show_selection_for_item(idx)
                if self.hold_cb is not None:
                    self.hold_timer = ecore.timer_add(self.hold_const,
                                                      self._hold_t_func, idx)
        KineticList._cb_on_init_click(self, obj, event)

    def _cb_on_mouse_down(self, obj, event):
        if event.button == 1:
            self.hold_t_ok = False
            if self.callback_on_mouse_down:
                self.callback_on_mouse_down(obj, event)
            KineticList._cb_on_mouse_down(self, obj, event)

    def _cb_on_mouse_move(self, obj, event):
        if event.buttons == 1:
            if self.callback_on_mouse_move:
                self.callback_on_mouse_move(obj, event)
            KineticList._cb_on_mouse_move(self, obj, event)
            if self._resizable_renderer is not None and \
                   not self._is_click_possible(event.position.canvas.y):
                self.hide_selection()
                if self.hold_timer is not None:
                    self.hold_timer.delete()
                    self.hold_timer = None

    def _cb_on_mouse_up(self, obj, event):
        if event.button == 1:
            if not self.hold_t_ok:
                self.hide_selection()
                if self.hold_timer is not None:
                    self.hold_timer.delete()
                    self.hold_timer = None
            if self.callback_on_mouse_up:
                self.callback_on_mouse_up(obj, event)
            KineticList._cb_on_mouse_up(self, obj, event)

    def emit_clicked(self, *ignored):
        if not self.hold_t_ok:
            KineticList.emit_clicked(self, *ignored)

    def show_selection_for_item(self, index):
        model = self.elements[index]
        renderer = self.renderer_for_index(index)
        if renderer is None:
            return
        renderer.select()
        self._resizable_renderer = renderer
        if self.callback_on_show_selection:
            self.callback_on_show_selection(index)

    def hide_selection(self):
        if self._resizable_renderer is not None:
            self._resizable_renderer.unselect()
            self._resizable_renderer = None

    def blink_selection(self):
        if self._resizable_renderer is not None:
            self._resizable_renderer.blink()
            self._resizable_renderer = None

    def select_item(self, index, end_callback=None):
        model = self.elements[index]
        renderer = self.renderer_for_index(index)
        if renderer is None:
            if self._selection is not None:
                self._selection.hide()
            return

        if self.callback_on_item_selected:
            self.callback_on_item_selected(index)

        def end(*ignored):
            if self._selection is not None:
                self._selection.hide()
            if end_callback:
                end_callback(self)

        if self._selection is not None:
            self._selection.copy_renderer(renderer)
            self.member_add(self._selection)
            self._selection.select(end)
            self._selection.show()
            renderer.state_hidden()

        # XXX: immediate visual feedback, force edje programs to run.
        edje.message_signal_process()
        self.evas.render()
        if self._selection is None and end_callback is not None:
            end_callback(self)

    @evas.decorators.del_callback
    def _on_delete(self, *ignored):
        if self._selection is not None:
            self._selection.delete()
        for r in self.renderers:
            r.delete()
        self.renderers = None
        self.elements = None


class KnobWidget(EdjeWidget, TerraObject):
    terra_type = "Widget/Knob"

    def __init__(self, parent, group=None, theme=None):
        if group is None:
            group = "list_scrollbar"
        EdjeWidget.__init__(self, parent.evas, group, parent, theme)

    def tooltip_show(self):
        self.signal_emit("tooltip,show", "")

    def tooltip_hide(self):
        self.signal_emit("tooltip,hide", "")

    def tooltip_set(self, value):
        self.part_text_set("tooltip", value)


class ScrollbarWidget(Scrollbar, Widget, TerraObject):
    terra_type = "Widget/Scrollbar"

    def __init__(self, parent, knob, theme=None):
        Widget.__init__(self, parent, theme)
        Scrollbar.__init__(self, parent.evas, knob)
        knob.show()

    def theme_changed(self):
        self.knob.theme_changed()

    def tooltip_show(self):
        self.knob.tooltip_show()

    def tooltip_hide(self):
        self.knob.tooltip_hide()

    def tooltip_set(self, value):
        self.knob.tooltip_set(value)


class ActionButton(EdjeWidget, TerraObject):
    group = "widget/list/action_button"
    terra_type = "Widget/ActionButton"

    (
        STATE_FETCH,
        STATE_WORKING,
        STATE_PAUSED,
        STATE_QUEUED,
        STATE_TRASH,
        STATE_DELETE
    ) = range(6)

    dispatch_table = {
        STATE_FETCH: "state,initial",
        STATE_WORKING: "state,downloading",
        STATE_PAUSED: "state,paused",
        STATE_QUEUED: "state,queued",
        STATE_TRASH: "state,downloaded",
        STATE_DELETE: "delete,show"
    }

    def __init__(self, parent, theme=None):
        EdjeWidget.__init__(self, parent.evas, self.group, parent, theme)

        self.contents_box_collapsed_cb = None
        self.contents_box_expanded_cb = None
        self.button_fetch_pressed_cb = None
        self.button_pause_pressed_cb = None
        self.button_resume_pressed_cb = None
        self.button_delete_pressed_cb = None
        self.delete_shown_cb = None
        self.delete_hidden_cb = None

        self.progress_set(0.0)

        obj = self.part_object_get("contents_area")
        obj.on_resize_add(self._resize)

    def _resize(self, obj):
        w, h = obj.size_get()
        edje.extern_object_min_size_set(self, w, h)
        edje.extern_object_max_size_set(self, w, h)

    def theme_changed(self):
        EdjeWidget.theme_changed(self)
        obj = self.part_object_get("contents_area")
        obj.on_resize_add(self._resize)

    def state_set(self, state):
        self.signal_emit(self.dispatch_table[state], "")

    def _call_if_exists(self, cb, args=()):
        if cb is not None:
            cb(*args)

    def delete_text_set(self, text):
        self.part_text_set("bt_delete_confirm_text", text)

    def disable_download(self):
        self.signal_emit("disable,download", "")

    def on_contents_box_collapsed_set(self, cb):
        self.contents_box_collapsed_cb = cb

    def on_contents_box_expanded_set(self, cb):
        self.contents_box_expanded_cb = cb

    def on_button_fetch_pressed_set(self, cb):
        self.button_fetch_pressed_cb = cb

    def on_button_pause_pressed_set(self, cb):
        self.button_pause_pressed_cb = cb

    def on_button_resume_pressed_set(self, cb):
        self.button_resume_pressed_cb = cb

    def on_button_delete_pressed_set(self, cb):
        self.button_delete_pressed_cb = cb

    def on_delete_shown_set(self, cb):
        self.delete_shown_cb = cb

    def on_delete_hidden_set(self, cb):
        self.delete_hidden_cb = cb

    @edje.decorators.signal_callback("contents_box,collapsed", "")
    def _on_contents_box_collapsed(self, *ignore):
        self._call_if_exists(self.contents_box_collapsed_cb)

    @edje.decorators.signal_callback("contents_box,expanded", "")
    def _on_contents_box_expanded(self, *ignore):
        self._call_if_exists(self.contents_box_expanded_cb)

    @edje.decorators.signal_callback("button,start,pressed", "")
    def _on_button_fetch_pressed(self, *ignore):
        self._call_if_exists(self.button_fetch_pressed_cb)

    @edje.decorators.signal_callback("button,pause,pressed", "")
    def _on_button_pause_pressed(self, *ignore):
        self._call_if_exists(self.button_pause_pressed_cb)

    @edje.decorators.signal_callback("button,resume,pressed", "")
    def _on_button_resume_pressed(self, *ignore):
        self._call_if_exists(self.button_resume_pressed_cb)

    @edje.decorators.signal_callback("button,delete,pressed", "")
    def _on_button_delete_pressed(self, *ignore):
        self._call_if_exists(self.button_delete_pressed_cb)

    @edje.decorators.signal_callback("delete,shown", "")
    def _on_delete_shown(self, *ignore):
        self._call_if_exists(self.delete_shown_cb)

    @edje.decorators.signal_callback("delete,hidden", "")
    def _on_delete_hidden(self,*ignore):
        self._call_if_exists(self.delete_hidden_cb)

    def progress_set(self, value):
        self.part_drag_value_set("knob", value, 0.0)

    def inform_online(self):
        self.signal_emit("network,true", "")

    def inform_offline(self):
        self.signal_emit("network,false", "")

    def inform_fetch_finished(self):
        self.signal_emit("download,finished", "")

    def progress_show_animated(self):
        self.signal_emit("contents_box,open,download", "")

    def delete_show(self):
        self.signal_emit("delete,show", "")


class DeletableRowRenderer(RowRendererWidget):
    """The parent model need to have [is_removable_child] and [on_remove_child]
    methods to handle delete actions."""

    terra_type = "Widget/BaseDeletableRowRenderer"

    def __init__(self, parent, theme=None):
        RowRendererWidget.__init__(self, parent, theme)

        self.delete_button = ActionButton(self)
        self.delete_button.state_set(ActionButton.STATE_TRASH)
        self.delete_button.on_button_delete_pressed_set(self.cb_delete_pressed)
        self.part_swallow("delete_button", self.delete_button)

        self.delete_button.on_contents_box_expanded_set(self.cb_box_expanded)
        self.delete_button.on_contents_box_collapsed_set(self.cb_box_collapsed)
        self.delete_button.disable_download()
        self.delete_button.delete_text_set("Delete")

    def theme_changed(self, end_callback=None):
        def cb(*ignored):
            self.part_swallow("delete_button", self.delete_button)
            if end_callback is not None:
                end_callback(self)

        self.delete_button.theme_changed()
        self.delete_button.state_set(ActionButton.STATE_TRASH)
        self.delete_button.disable_download()
        RowRendererWidget.theme_changed(self, cb)

    def cb_box_expanded(self, *ignored):
        self._model._deletable_select_state = True

    def cb_box_collapsed(self, *ignored):
        self._model._deletable_select_state = False

    def cb_delete_pressed(self, *ignored):
        def cb_collapsed(*ignored):
            self.delete_button.signal_callback_del("contents_box,collapsed", "",
                                                   cb_collapsed)
            self._model.parent.on_remove_child(self._model)
            self.delete_button.signal_emit("unblock,events", "")
            self.delete_button.state_set(ActionButton.STATE_TRASH)

        self.delete_button.signal_callback_add("contents_box,collapsed", "",
                                               cb_collapsed)

    def value_set(self, model):
        if not model or model is self._model:
            return

        if model.parent.is_removable_child(model):
            self.signal_emit("delete_button,show", "")
        else:
            self.signal_emit("delete_button,hide", "")

        # XXX: faster hash lookups
        if not hasattr(model, "_deletable_select_state"):
            model._deletable_select_state = False

        if model._deletable_select_state:
            self.delete_button.state_set(ActionButton.STATE_DELETE)
        else:
            self.delete_button.state_set(ActionButton.STATE_TRASH)

        self._model = model
        self.part_text_set("text", model.name)

    @evas.decorators.del_callback
    def __on_delete(self):
        self.delete_button.delete()


class ResizableDeletableRowRenderer(DeletableRowRenderer, ResizableRowRendererWidget):
    row_group="list_item_text_resizable"

    def __init__(self, parent, theme=None):
        DeletableRowRenderer.__init__(self, parent, theme)


class DeletableRowRendererWidget(DeletableRowRenderer):
    terra_type = "Widget/DeletableRowRenderer"
    row_group="list_item_text_deletable"
    resizable_row_renderer = ResizableDeletableRowRenderer
