# -*- coding: utf-8 -*-
#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import os
from logging import getLogger
from datetime import datetime
from time import time, mktime

from ecore import timer_add

from terra.core.manager import Manager
from terra.core.model import ModelFolder
from terra.core.task import Task
from terra.utils.encoding import to_utf8
from terra.core.plugin_prefs import PluginPrefs
from sqlite3 import OperationalError

from common import STATE_DOWNLOADED
from folder import OnDemandFolder

__all__ = ("OnDemandTaskFolder",
           "FeedFolder", "OnDemandPlayerOptionsModel",
           "InfoOnDemandPlayerOptionsModel", "MarkAsNewOptionsActionModel",
           "TopRatedGenericOnDemandFolder", "NewEpisodesAudioOnDemandFolder",
           "TodaysEpisodesGenericOnDemandFolder",
           "CurrentDownloadsGenericOnDemandFolder")


log = getLogger("plugins.canola-core.ondemand.model.model")
mger = Manager()
db = mger.canola_db

FatalError = mger.get_class("Model/Notify/Error/Fatal")
OptionsModelFolder = mger.get_class("Model/Options/Folder")
OptionsActionModel = mger.get_class("Model/Options/Action")
OnDemandFolderOptionsModel = \
    mger.get_class("Model/Options/Folder/Media/OnDemand")

DownloadManager = mger.get_class("DownloadManager")
download_mger = DownloadManager()


class OnDemandTaskFolder(OnDemandFolder, Task):
    terra_type = "Model/Folder/Task/Generic/OnDemand"
    title = "OnDemand source"
    preset_section = ""

    feed_cls = None
    model_cls = None

    # XXX Change this key in the future
    db_version_settings_key = "podcast_db_version"
    db_version = 3

    def __init__(self, parent):
        Task.__init__(self)
        OnDemandFolder.__init__(self, self.title, None, parent)
        try:
            self.check_db_version()
        except OperationalError, e:
            log.debug("OnDemand model found problem : %s" % e)
            msg = "Problem with the database version.<br>Going to exit Canola."
            err = FatalError(msg)
            self.parent.callback_notify(err)
        try:
            self.reinit_downloads()
        except OperationalError, e:
            log.debug("OnDemand model found problem : %s" % e)
            msg = "Local drive is out of space.<br>Going to exit Canola."
            err = FatalError(msg)
            self.parent.callback_notify(err)

    def reinit_downloads(self):
        rows = self.model_cls.execute_stmt('select-all')
        cache = dict()
        for r in rows:
            # creating an item resumes interruped transfers
            c = self.child_model_from_db_row(r, cache)
        cache = None

    def check_db_version(self):
        key = self.db_version_settings_key
        cur_db_version = int(PluginPrefs("settings").get(key, 1))

        def exists(cls):
            return bool(cls.execute_stmt('table-info'))

        def create(cls):
            cls.execute_stmt('create', layout=cls.layout[3])

        def insert_presets(cls):
            from ConfigParser import ConfigParser
            if not cls is self.feed_cls:
                return

            try:
                presets_cfg = mger.terra_config.presets_cfg
                parser = ConfigParser()
                parser.readfp(open(presets_cfg, "r"))

                num = int(parser.get(self.preset_section, "NumberOfEntries")) + 1
            except:
                return

            def get(prop, idx):
                try:
                    return parser.get(self.preset_section, prop + str(idx))
                except:
                    return None

            for i in xrange(1, num):
                values = (get("Location", i),
                          get("Title", i),
                          get("Description", i),
                          None, int(time()))

                if values[0] and values[1]:
                    cls.execute_stmt('insert', *values)
                else:
                    continue
            db.commit()

        def migrate(cls):
            ### FIXME: migrate is not working correctly, should be fixed later
            msg = "Problem with database.<br>Please run Canola2-Cleanup."
            err = FatalError(msg)
            self.parent.callback_notify(err)

        set_version = True
        for cls in [self.feed_cls, self.model_cls]:
            if not exists(cls):
                create(cls)
                insert_presets(cls)
            elif self.db_version > cur_db_version:
                migrate(cls)
            else:
                set_version = False

        if set_version:
            s = PluginPrefs("settings")
            s[key] = self.db_version
            s.save()


class FeedFolder(ModelFolder):
    terra_type = "Model/Folder/Media/Feeds"
    title = "My feeds"
    feed_cls = OnDemandFolder
    model_cls = None

    def __init__(self, parent):
        ModelFolder.__init__(self, self.title, parent)

    def do_load(self):
        rows = self.feed_cls.execute_stmt('select-order-by', field="title")
        for r in rows:
            c = self.child_model_from_db_row(r)
            self.append(c)

    def child_model_exists_in_db(self, uri, cur=None):
        try:
            c = self.feed_cls
            if cur:
                already_exists = c.execute_stmt_with_cursor('exists', cur, uri)
            else:
                already_exists = c.execute_stmt('exists', uri)

            if already_exists:
                return True
            else:
                return False
        except:
            return False

    def child_model_from_db_row(self, row):
        uri, title, desc, cache_tag = row[1:5]

        item = self.feed_cls(to_utf8(title), to_utf8(uri))
        item.id = row[0]
        item.desc = to_utf8(desc or "")
        item._cache_tag = cache_tag

        return item


class OnDemandPlayerOptionsModel(OptionsModelFolder):
    terra_type = "Model/Options/Folder/Player/OnDemand"
    title = "Options"
    children_order = ["/Info", "/MarkAsUnheard"]
    children_prefixes = ["Model/Options/Folder/Player/OnDemand",
                         "Model/Options/Action/Player/OnDemand"]


class InfoOnDemandPlayerOptionsModel(OptionsModelFolder):
    terra_type = "Model/Options/Folder/Player/OnDemand/Info"
    title = "Info"


class MarkAsNewOptionsActionModel(OptionsActionModel):
    terra_type = "Model/Options/Action/Player/OnDemand/MarkAsNew"
    name = "Mark as new"

    def execute(self):
        player = self.screen_controller
        episode = player.model
        episode.unheard = True
        return True


class QueryBasedOnDemandFolder(OnDemandFolder):
    title = "All entries"
    model_cls = None
    select_query = "SELECT * FROM %(item-table)s"

    def __init__(self, parent=None):
        OnDemandFolder.__init__(self, self.title, None, parent)
        self._query = self.model_cls.preprocess_query(self.select_query)

    def do_load(self):
        rows = db.execute(self._query)
        cache = dict()
        for r in rows:
            item = self.child_model_from_db_row(r, cache)
            if not self.reject(item):
                self.children.append(item)
        cache = None

    def reject(self, item):
        return False

    def options_model_get(self, controller):
        return OnDemandFolderOptionsModel(None, controller)


class TopRatedGenericOnDemandFolder(QueryBasedOnDemandFolder):
    terra_type = "Model/Folder/Media/OnDemand/Generic/TopRated"
    title = "Top rated"
    model_cls = None

    min_rating = 3
    # More than average rating and limited to 25
    select_query = "SELECT * FROM %(item-table)s " + \
        "WHERE rating >= %d ORDER BY(rating) LIMIT 25" % min_rating

    def child_property_changed(self, child):
        if child.rating < self.min_rating:
            self.children.remove(child)


class NewEpisodesAudioOnDemandFolder(QueryBasedOnDemandFolder):
    terra_type = "Model/Folder/Media/OnDemand/Generic/NewEpisodes"
    title = "New episodes"
    model_cls = None

    select_query = "SELECT * FROM %(item-table)s " \
        "WHERE uri != 0 AND uri != '' AND read == 0 " \
        "ORDER BY(date) DESC"

    def reject(self, item):
        if not os.path.exists(item.uri or "") \
            or os.path.exists(item.uri + ".info"):
            return True
        else:
            return False

    def child_property_changed(self, child):
        if child.data_state != STATE_DOWNLOADED:
            # Not downloaded (deleted from list) thus not a new
            # episode any longer.
            self.children.remove(child)

        if child.unheard == False:
            # We only want to remove the child after we have
            # entered the player. Add delay
            timer_add(2.0, lambda: self.children.remove(child))


class TodaysEpisodesGenericOnDemandFolder(OnDemandFolder):
    terra_type = "Model/Folder/Media/OnDemand/Generic/TodaysEpisodes"
    title = "Today's episodes"
    model_cls = None
    feed_cls = None

    def __init__(self, parent=None):
        OnDemandFolder.__init__(self, self.title, None, parent)

    def feed_model_from_db_row(self, row):
        uri, title, desc, cache_tag = row[1:5]

        item = self.feed_cls(to_utf8(title), to_utf8(uri))
        item.id = row[0]
        item._cache_tag = cache_tag
        item.desc = to_utf8(desc or "")

        return item

    def do_load(self):
        rows = self.feed_cls.execute_stmt('select')
        self.feeds_left = len(rows)

        # This will make sure all feeds are refreshed before
        # refreshing ourselves.
        def on_refresh_finished(feed, was_updated):
            feed.refresh_finished_callback = None
            self.feeds_left -= 1
            if self.feeds_left == 0:
                self.load_from_db()

        for r in rows:
            feed = self.feed_model_from_db_row(r)
            feed.refresh_finished_callback = on_refresh_finished
            feed.do_refresh()

        self.is_loading = True

    def do_unload(self):
        self.refresh_finished_callback = None

    def load_from_db(self):
        today = datetime.now()
        # We look one day before because of feeds not specifying timezone,
        # and thus not showing up because they are in the past
        epoch = mktime((today.year, today.month,
            today.day - 1, 0, 0, 0, 0, 0, 0))

        query_tmpl = "SELECT * FROM %(item-table)s WHERE date > " + str(epoch)
        query = self.feed_cls.preprocess_query(query_tmpl)
        rows = db.execute(query)

        cache = dict()
        for r in rows:
            c = self.child_model_from_db_row(r, cache)
            self.children.append(c)
        cache = None

        self.inform_loaded()
        self.refresh_finished_callback = None

    def options_model_get(self, controller):
        return OnDemandFolderOptionsModel(None, controller)


class CurrentDownloadsGenericOnDemandFolder(OnDemandFolder):
    terra_type = "Model/Folder/Media/OnDemand/Generic/CurrentDownloads"
    title = "Current downloads"
    model_cls = None
    feed_cls = None
    watched_groups = []

    def __init__(self, parent=None):
        OnDemandFolder.__init__(self, self.title, None, parent)

    def on_added_or_removed(self, group):
        def do_reload():
            self.children.freeze()
            try:
                self.do_unload()
                self.do_load()
            finally:
                self.children.thaw()

        if group in self.watched_groups:
            do_reload()

    def do_load(self):
        download_mger.on_item_added_add(self.on_added_or_removed)
        download_mger.on_item_removed_add(self.on_added_or_removed)

        for downloader in download_mger.items.itervalues():
            if downloader.group not in self.watched_groups:
                continue

            feed_id, id, title = downloader.data
            rows = self.model_cls.execute_stmt('select-one', feed_id, id)

            cache = dict()
            if rows:
                c = self.child_model_from_db_row(rows[0], cache)
                self.append(c)
            cache = None

    def do_unload(self):
        download_mger.on_item_added_remove(self.on_added_or_removed)
        download_mger.on_item_removed_remove(self.on_added_or_removed)

        # parent method unloads and deletes .children
        OnDemandFolder.do_unload(self)
