#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import os
import re
import ecore
import logging

from album_titles_list import ListWidget
from album_covers_set import CellRendererWidget, KineticVGridWidget, \
    KnobWidget, ScrollbarWidget

__all__ = ("AlbumView",)

log = logging.getLogger("canola.ui.album_view")
s_cover_ln_re = re.compile(".*-small\.jpg")
s_covers_re = re.compile(".*-small-[0-9]+\.jpg")


class AlbumView(object):
    SCROLLBAR_TIMEOUT = 0.7
    SCROLLBAR_FIRST_TIMEOUT = 1.5

    def __init__(self, widget, view, cb_title_clicked=None, cb_cover_clicked=None):
        self.view = view
        self.titles = None
        self.cb_title_clicked = cb_title_clicked
        self.cb_cover_clicked = cb_cover_clicked
        self.cover = self.view.evas.FilledImage()
        self.w_cover = None
        self.w_cover_set = None
        self.set_cover_widget(widget)

    def cb_cover_right(self):
        def renderer_new(canvas):
            return CellRendererWidget(self.view)

        self.elements = []
        if os.path.exists(self._cover_path):
            files = os.listdir(self._cover_path)
            for cover in files:
                if s_covers_re.match(cover) and not s_cover_ln_re.match(cover):
                    self.elements.append(os.path.join(self._cover_path, cover))
        self.elements.append("cover-small-no_cover.jpg")

        if self.w_cover_set is not None:
            self.w_cover_set.delete()
            self._scroll.delete()

        if self.parent:
           self.w_cover_set = KineticVGridWidget(self.view,
                                                 renderer_new, self.elements)
           self.parent.part_swallow("album_set", self.w_cover_set)
           self.w_cover_set.show()
           self.w_cover_set.clicked_cb_set(self.cb_cover_clicked)
           self.w_cover_set.position_changed_cb_set(self._cb_position_changed)
           self.w_cover_set.on_resize_add(self._cb_grid_resized)
           self._setup_gui_scrollbar()
           self._show_scrollbar(self.SCROLLBAR_FIRST_TIMEOUT)

    def _setup_gui_scrollbar(self):
        knob = KnobWidget(self.view, group="album_list_scrollbar")
        self._scroll = ScrollbarWidget(self.view, knob)
        self.parent.part_swallow("scrollbar", self._scroll)
        self._hide_scroll_timer = None

    def _show_scrollbar(self, timeout=SCROLLBAR_TIMEOUT):
        if self.w_cover_set.visible_cols_count() < len(self.elements):
            self._hide_scroll_timer = \
                ecore.timer_add(timeout, self._hide_scroll_cb)
            self.parent.signal_emit("scrollbar,transition,in", "")

    def _cb_position_changed(self, grid, percent):
        self._scroll.value_set(percent)
        if self._hide_scroll_timer is not None:
            self._hide_scroll_timer.delete()
        self._hide_scroll_timer = ecore.timer_add(self.SCROLLBAR_TIMEOUT,
                                                  self._hide_scroll_cb)
        self.parent.signal_emit("scrollbar,transition,in", "")

    def _hide_scroll_cb(self):
        self.parent.signal_emit("scrollbar,transition,out", "")
        self._hide_scroll_timer = None
        return False

    def _cb_grid_resized(self, grid):
        self._scroll.scale_set(grid.visible_rows_scale_get())

    def set_new_album(self, album=None):
        self.set_cover_image(album)
        if album:
            self.set_list(album=album)

    def set_cover_widget(self, widget):
        if widget.part_exists("titles_clipper"):
            self.parent = widget
            self.w_cover = widget.part_swallow_get("album_cover")
        else:
            self.parent = None
            self.w_cover = widget
        self.w_cover.part_swallow("album_cover", self.cover)

        if self.titles and self.parent:
            self.parent.part_swallow("titles", self.titles)

    def set_cover_image(self, album=None):
        album.w_cover.part_unswallow(album.cover)
        self.cover = album.cover
        self._cover_path = album._cover_path
        self.set_album_state(album.w_cover)
        self.w_cover.part_swallow("album_cover", self.cover)
        if self.w_cover_set is not None:
            self.w_cover_set.delete()
            self._scroll.delete()

    def set_cover(self, cover=None):
        if cover and os.path.exists(cover):
            self._cover_path = os.path.dirname(cover)
            #self.cover.load_size_set(256, 256)
            try:
                self.cover.file_set(cover)

                # hack to force evas load image from disk
                # as this method is 'lazy', we need to access
                # buffer()[0] in order to correctly load image
                # from disk.
                b = buffer(self.cover)[0]

                self.w_cover.part_swallow("album_cover", self.cover)
                self.w_cover.signal_emit("cover,show", "")
                return
            except Exception, e:
                log.error("Error during set_cover: %s", e)
        elif cover:
            self._cover_path = os.path.dirname(cover)
        else:
            self._cover_path = ""

        self.w_cover.signal_emit("cover,hide", "")
        self.cover.hide()

    def set_album_state(self, next_state):
        if next_state.part_state_get("album_cover")[0] == "default":
            self.w_cover.signal_emit("cover,show", "")
        else:
            self.w_cover.signal_emit("cover,hide", "")

    def set_list(self, titles=None, album=None):
        if album:
            if album.parent:
                album.parent.part_unswallow(album.titles)
            self.titles = album.titles
        else:
            self.titles = ListWidget(self.view, titles)

        if self.titles:
            self.titles.callback_clicked = self.cb_title_clicked
            if self.parent:
                self.parent.part_swallow("titles", self.titles)

    def theme_changed(self):
        if self.titles is not None:
           self.titles.theme_changed()

        if self.w_cover_set is not None:
            self.w_cover_set.theme_changed()
            self.parent.part_swallow("album_set", self.w_cover_set)
            self.w_cover_set.show()
            self._scroll.theme_changed()
            self.parent.part_swallow("scrollbar", self._scroll)


    def delete(self):
        if self.cover:
            self.w_cover.part_unswallow(self.cover)
            self.cover.delete()

        if self.parent:
            self.parent.part_unswallow(self.titles)

        if self.titles and not self.titles.is_deleted():
            self.titles.delete()

        if self.w_cover_set and not self.w_cover_set.is_deleted():
            self.w_cover_set.delete()
            self._scroll.delete()


