#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import logging
import extfeedparser
import urlparse

from terra.core.manager import Manager

from model import IRadioModelFolder, IRadioFeedModelFolder
from terra.utils.encoding import to_utf8

__all__ = ("SettingsIRadioModelFolder", "SettingsIRadioFolderController", )


log = logging.getLogger("plugins.canola-core.iradio.settings")
mger = Manager()
SettingsFeedFolderController = mger.get_class("Controller/Settings/Folder/Feed")


class SettingsIRadioModelFolder(IRadioModelFolder):
    terra_type = "Model/Settings/Folder/InternetMedia/IRadio"
    title = "Internet radio"
    select_order = "ORDER BY epoch DESC"

    def item_new(self, uri, contents):
        feed = contents.feed
        title = to_utf8(feed.title)

        # XXX: what's the correct field here?
        if feed.has_key("description"):
            desc = to_utf8(feed.description)
        else:
            desc = ""

        mimetype = contents.headers['content-type']

        item = IRadioFeedModelFolder(None, uri, mimetype, title, desc)
        item.commit()
        return item


class SettingsIRadioFolderController(SettingsFeedFolderController):
    terra_type = "Controller/Settings/Folder/InternetMedia/IRadio"

    acceptable_mime_types = [
        'application/pls', 'audio/x-scpls', 'pls', # dance.fm sends 'pls'
        'application/m3u', 'audio/x-mpegurl',
        'application/asx', 'video/x-ms-asf',
        'application/ram', 'application/rpm', 'audio/x-pn-realaudio',
        'application/rss+xml' # Allow RSS feeds as streaming radios
    ]

    def _get_dummy_title(self, url):
        title = urlparse.urlparse(url)[1]

        name_list = title.split(".")
        if len(name_list) > 2 and not name_list[-1].isdigit():
            title = ".".join(name_list[1:])

        title = title.capitalize()

        # Special handling!
        if "shoutcast.com" in url:
            try:
                for arg in url.split("?")[1].split("&"):
                    if arg.startswith("rn="):
                        no = arg.split("=")[1]
                return title + " (" + no + ")"
            except:
                return title
        else:
            return title

    def alternative_uri(self, uri, mimetype):
        if mimetype is None:
            # Happens with shoutcast streams.
            # return alternative URI, consisting of original URI plus
            # "/.pls". If it is a shoutcast server this will return
            # the feed.
            return uri + "/listen.pls"
        else:
            return None

    def process_non_feed(self, uri, mimetype):
        if not mimetype in ['audio/mpeg']:
            return None

        data = extfeedparser.Dummy(mimetype)

        feed = data.feed
        feed['title'] = self._get_dummy_title(uri)
        feed['link'] = uri

        return data

    def _parse(self, filename, orig_uri, mimetype):
        try:
            data = extfeedparser.parse(filename, mimetype)
            if data is None:
                return None
            feed = data.feed

            if not 'title' in feed:
                feed['title'] = self._get_dummy_title(orig_uri)
            if not 'link' in feed:
                feed['link'] = orig_uri

            return data
        except Exception, e:
            log.debug("Exception during extfeedparser.", exc_info=True)
            raise e
