#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import edje

from terra.ui.panel import Panel
from terra.ui.layout import Box
from terra.ui.base import EdjeWidget
from terra.core.terra_object import TerraObject


class PairBox(Box):
    """Layout a pair of items, one to the left and the other to the right."""
    def __init__(self, canvas, size, left=None, right=None,
                 hpadding=1, vpadding=1):
        self.hpadding = int(hpadding)
        self.vpadding = int(vpadding)
        self._size = size
        Box.__init__(self, canvas, [left, right], halign=0.5, valign=0.5)

    def required_size_get(self):
        return self._size

    def _setup_gui(self, x, y, w, h):
        if not self._dirty:
            return

        if not self._children:
            self.clipper.hide()
            self._dirty = False
            return

        self.clipper.visible = self.visible

        req_w, req_h = self.required_size_get()
        if req_w < w:
            x += int(((w - req_w) / 2) * self.halign)

        left, right = self._children
        if left is not None:
            left.size = left.size_min_calc()
            cx = self.hpadding
            cy = int((h - left.size[1]) * self.valign)
            left.move(x + cx, y + cy)

        if right is not None:
            right.size = right.size_min_calc()
            cx = int(req_w - right.size[0]) - self.hpadding
            cy = int((h - right.size[1]) * self.valign)
            right.move(x + cx, y + cy)

        self._dirty = False


class BasicPanel(Panel, TerraObject):
    terra_type = "Widget/BasicPanel"
    background_group = "panel_list_contents"

    def __init__(self, main_window, title, group=None, theme=None,
                 header_text=None):
        Panel.__init__(self, main_window, title, theme)
        TerraObject.__init__(self)

        self.contents = EdjeWidget(self.evas, self.background_group, self)
        self.callback_escape = None
        self.header_text = header_text
        self._button_box = None
        self._button_left = None
        self._button_right = None
        self._inner_contents = None

    def _setup_button_box(self, left=None, right=None):
        """Subclasses can use it to create a button bar."""
        self._button_left = left
        self._button_right = right

        # XXX: we shouldn't need to set those sizes explicitly like that.
        w, h = (left or right).size_min_get()
        ow, oh = self._contents.part_size_get("bottom_contents")

        W, H = ow, h + 20
        self._button_box = PairBox(self.evas, (W, H), left, right, hpadding=10)
        edje.extern_object_min_size_set(self._button_box, W, H)
        edje.extern_object_max_size_set(self._button_box, W, H)

        self._contents.part_swallow("bottom_contents", self._button_box)
        if left is not None:
            left.show()
        if right is not None:
            right.show()

    def handle_key_down(self, event):
        if event.key == "Escape" and self.callback_escape:
            self.callback_escape()

    def delete(self):
        self._contents.delete()
        if self._button_box is not None:
            self._button_box.clear()
        if self._button_left is not None:
            self._button_left.delete()
        if self._button_right is not None:
            self._button_right.delete()
        Panel.delete(self)

    def theme_changed(self):
        Panel.theme_changed(self)
        self._contents.theme_changed()
        if self._button_left is not None:
            self._button_left.theme_changed()
        if self._button_right is not None:
            self._button_right.theme_changed()

    def header_text_set(self, value):
        if not value:
            self._contents.signal_emit("header,hide", "")
            self._contents.part_text_set("header_text", "")
        else:
            self._contents.signal_emit("header,show", "")
            self._contents.part_text_set("header_text", value)

    header_text = property(fset=header_text_set)

    # XXX: maybe call it contents_set, since our parents contents_set
    # is just used internally?
    def inner_contents_set(self, inner_contents):
        self._inner_contents = inner_contents
        self._contents.part_swallow("contents", inner_contents)
