/*
 * callforwarding - Call forwarding control panel applet for N900
 * Copyright (C) 2009  Mikko Mehtonen <mikko.mehtonen@vincit.fi>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 * 
 */


#include <hildon/hildon.h>
#include <gtk/gtk.h>
#include <string.h>
#include <libosso-abook/osso-abook.h>
#include "callforwardingdialog.h"
#include "callforwarding.h"

#include <locale.h>
#include <config.h>
#include <glib/gi18n.h>

#define _(String) gettext (String)

static void dialog_response (GtkDialog *dialog,
                             gint response,
                             gpointer user_data);
static void callforwarding_status_reply (GError *error, gboolean status, gchar *number);
static void callforwarding_cancel_reply (GError *error);
static void callforwarding_activate_reply (GError *error);

static void set_dialog_sensitivity (void);
static void enable_callforwarding_clicked (HildonCheckButton *button, gpointer type);
static void select_contact_clicked (GtkWidget *widget, gpointer data);

static GtkWidget *dialog = NULL;
static GtkWidget *button = NULL;
static GtkWidget *entry = NULL;
static GtkWidget *label = NULL;
static GtkWidget *select_contact = NULL;
static GtkWidget *picker = NULL;
static gint call_forwarding_type = 0;

void callforwardingdialog_show (GtkWindow *parent,
                                const gchar *title,
                                gint type)
{
  call_forwarding_type = type;
  dialog = gtk_dialog_new_with_buttons (
                              title,
                              parent, GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT |
                                      GTK_DIALOG_NO_SEPARATOR,
                              _("Save"), GTK_RESPONSE_OK,
                              NULL);
  g_signal_connect (G_OBJECT(dialog), "response",
                    G_CALLBACK(dialog_response), NULL);

  GtkWidget *vbox = gtk_vbox_new (FALSE, HILDON_MARGIN_DEFAULT);

  button = hildon_check_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label (GTK_BUTTON(button), _("Enabled"));
  g_signal_connect(GTK_OBJECT(button), "clicked",
                   G_CALLBACK(enable_callforwarding_clicked), NULL);
  gtk_box_pack_start (GTK_BOX(vbox), button, FALSE, FALSE, 0);

  entry = hildon_entry_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
  hildon_gtk_entry_set_input_mode(GTK_ENTRY(entry), HILDON_GTK_INPUT_MODE_TELE);

  GtkWidget *hbox = gtk_hbox_new (FALSE, HILDON_MARGIN_DEFAULT);
  label = gtk_label_new (_("Forward to"));
  gtk_misc_set_alignment (GTK_MISC(label), 0.0, 0.5);
  gtk_box_pack_start (GTK_BOX(hbox), label, TRUE, TRUE, 0);
  gtk_box_pack_end (GTK_BOX(hbox), entry, TRUE, TRUE, 0);

  gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

  select_contact = hildon_button_new_with_text (
                            HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT, 
                            HILDON_BUTTON_ARRANGEMENT_HORIZONTAL,
                            _("Select a contact"), NULL);
  gtk_box_pack_start (GTK_BOX(vbox), select_contact, FALSE, FALSE, 0);
  g_signal_connect(GTK_OBJECT(select_contact), "clicked",
                   G_CALLBACK(select_contact_clicked), NULL);

  if (type == SS_DIVERT_NO_REPLY)
  {
    GtkWidget *selector = hildon_touch_selector_new_text ();
    hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR(selector), "5");
    hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR(selector), "10");
    hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR(selector), "15");
    hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR(selector), "20");
    hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR(selector), "25");
    hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR(selector), "30");

    picker = hildon_picker_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT,
                                       HILDON_BUTTON_ARRANGEMENT_VERTICAL);
    hildon_picker_button_set_selector (HILDON_PICKER_BUTTON(picker), HILDON_TOUCH_SELECTOR(selector));
    hildon_button_set_title (HILDON_BUTTON (picker), _("Seconds before fowarding"));
    hildon_button_set_alignment (HILDON_BUTTON (picker), 0, 0, 0, 0);
    gtk_box_pack_start (GTK_BOX(vbox), picker, FALSE, FALSE, 0);
  }

  gtk_container_add (GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), vbox);

  // NO_AVAIL is only for set, status can't queried
  if (type == SS_DIVERT_NO_AVAIL)
  {
    set_dialog_sensitivity ();
    gtk_widget_show_all (dialog);
  } else {
    /* Set dialog insensitive while settings are loaded */
    hildon_gtk_window_set_progress_indicator (GTK_WINDOW(dialog), TRUE);
    gtk_widget_set_sensitive (button, FALSE);
    gtk_widget_set_sensitive (entry, FALSE);
    gtk_widget_set_sensitive (label, FALSE);
    gtk_widget_set_sensitive (select_contact, FALSE);
    gtk_dialog_set_response_sensitive (GTK_DIALOG(dialog), GTK_RESPONSE_OK, FALSE);
    if (picker != NULL)
    {
      gtk_widget_set_sensitive (picker, FALSE);
    }
    gtk_widget_show_all (dialog);
    callforwarding_get_status (type, callforwarding_status_reply);
  }
}

static void set_dialog_sensitivity (void)
{
  gtk_dialog_set_response_sensitive (GTK_DIALOG(dialog), GTK_RESPONSE_OK, TRUE);
  gtk_widget_set_sensitive (button, TRUE);
  gboolean sensitive = hildon_check_button_get_active (HILDON_CHECK_BUTTON(button));

  gtk_widget_set_sensitive (entry, sensitive);
  gtk_widget_set_sensitive (label, sensitive);
  gtk_widget_set_sensitive (select_contact, sensitive);
  if (picker != NULL)
  {
    gtk_widget_set_sensitive (picker, sensitive);
  }

  if (sensitive)
  {
    gtk_widget_grab_focus(entry);
  }
}

static void enable_callforwarding_clicked (HildonCheckButton *button, gpointer type)
{
  set_dialog_sensitivity ();
}

static void dialog_response (GtkDialog *dialog,
                             gint response,
                             gpointer user_data)
{
  const gchar *number = NULL;
  switch (response)
  {
    case GTK_RESPONSE_OK:
      if (hildon_check_button_get_active (HILDON_CHECK_BUTTON(button)))
      {
        /* save */
        number = hildon_entry_get_text (HILDON_ENTRY(entry));
        if (number == NULL || strlen(number) < 1)
        {
            hildon_banner_show_information (GTK_WIDGET(dialog), NULL, _("Enter number"));
            return;
        }
        gtk_dialog_set_response_sensitive (GTK_DIALOG(dialog), GTK_RESPONSE_OK, FALSE);
        hildon_gtk_window_set_progress_indicator (GTK_WINDOW(dialog), TRUE);
        gint time = 20;
        if (picker != NULL)
        {
          gint index = hildon_picker_button_get_active (HILDON_PICKER_BUTTON(picker));
          if (index >= 0 && index < 6)
          {
            int times[] = {5, 10, 15, 20, 25, 30};
            time = times[index];
          }
        }
        callforwarding_activate (call_forwarding_type, number, time, callforwarding_activate_reply);
      } else {
        /* cancel */
        hildon_gtk_window_set_progress_indicator (GTK_WINDOW(dialog), TRUE);
        gtk_widget_set_sensitive (GTK_WIDGET(dialog), FALSE);
        callforwarding_cancel (call_forwarding_type, callforwarding_cancel_reply);
      }
      break;

    default:
      callforwarding_cancel_pending_call ();
      gtk_widget_destroy (GTK_WIDGET(dialog));
  }
}


static void callforwarding_status_reply (GError *error, gboolean status, gchar *number)
{
  if (error != NULL)
  {
    hildon_banner_show_information (dialog, NULL, _("Error while reading status"));
    g_clear_error (&error);
  } else {
    hildon_check_button_set_active (HILDON_CHECK_BUTTON(button), status);
    if (number != NULL)
    {
      hildon_entry_set_text (HILDON_ENTRY(entry), number);
    }
  }
  g_free (number);
  number = NULL;
  set_dialog_sensitivity ();
  hildon_gtk_window_set_progress_indicator (GTK_WINDOW(dialog), FALSE);
}

static void callforwarding_cancel_reply (GError *error)
{
  if (error != NULL)
  {
    hildon_banner_show_information (dialog, NULL, _("Error while canceling call forwarding"));
    printf("Error while canceling call forwarding: %s\n", error->message);
    g_clear_error (&error);
  }

  gtk_widget_destroy (GTK_WIDGET(dialog));
}

static void callforwarding_activate_reply (GError *error)
{
  if (error != NULL)
  {
    set_dialog_sensitivity ();
    hildon_gtk_window_set_progress_indicator (GTK_WINDOW(dialog), FALSE);
    hildon_banner_show_information (dialog, NULL, _("Not allowed"));
    g_clear_error (&error);
  } else {
    gtk_widget_destroy (GTK_WIDGET(dialog));
  }
}

static void select_contact_clicked (GtkWidget *widget, gpointer data)
{
  GtkWidget *chooser;
  GList *selection;
  EContact *contact = NULL;

  chooser = osso_abook_contact_chooser_new (NULL, _("Select a contact"));
  osso_abook_contact_chooser_set_capabilities (OSSO_ABOOK_CONTACT_CHOOSER(chooser), OSSO_ABOOK_CAPS_PHONE);

  gtk_dialog_run (GTK_DIALOG (chooser));
  gtk_widget_hide (chooser);

  selection = osso_abook_contact_chooser_get_selection
                    (OSSO_ABOOK_CONTACT_CHOOSER (chooser));

  if (selection) {
    contact = E_CONTACT (selection->data);
    GList *numbers = osso_abook_contact_get_attributes (contact, "TEL");
    if (g_list_length (numbers) < 2)
    {
      /* Contact has just one phone number, use it. */
      EVCardAttribute *attr = e_vcard_get_attribute (E_VCARD(contact), "TEL");
      hildon_entry_set_text (HILDON_ENTRY(entry), e_vcard_attribute_get_value (attr));
    } else {
      /* Contact has several phone numbers, let use choose which one should be used. */
      OssoABookContact *selected_contact = selection->data;
      GtkWidget *number_selector = osso_abook_contact_detail_selector_new_for_contact (
                                                      GTK_WINDOW(chooser),
                                                      selected_contact,
                                                      OSSO_ABOOK_CONTACT_DETAIL_PHONE);
      const int response = gtk_dialog_run (GTK_DIALOG (number_selector));
      if (response == GTK_RESPONSE_OK)
      {
        EVCardAttribute *attr;
        attr = osso_abook_contact_detail_selector_get_detail
                (OSSO_ABOOK_CONTACT_DETAIL_SELECTOR (number_selector));
        hildon_entry_set_text (HILDON_ENTRY(entry), e_vcard_attribute_get_value (attr));
      }
      gtk_widget_destroy (number_selector);
    }
  }
  g_list_free (selection);
  gtk_widget_destroy (chooser);
}
