#include "mainwindow.h"

MainWindow::MainWindow(QWidget *parent)
    : QMainWindow(parent)
{
    setAttribute(Qt::WA_Maemo5StackedWindow);
    setAttribute(Qt::WA_Maemo5AutoOrientation, true);

    readSettings();
    if (!QFile::exists("/opt/callerx/bin/callerx")){
        QMaemo5InformationBox::information(this, "There's no callerx daemon installed, This Application require callerx daemon to function properly");
        customlistspath.clear();
    }
    else {
        if (daemonenabled){
            if (!callerxDaemonRunning()){
                QMaemo5InformationBox::information(this, "Unable to communicate with callerx daemon");
            }
            else {
                if (!QDBusConnection::sessionBus().connect(BUS_NAME, BUS_PATH, BUS_NAME, "incoming", this, SLOT(inBoundCallUpdate(const QString&)))){
                    QMaemo5InformationBox::information(this, "Failed to connect to callerx daemon");
                }
            }
        }
    }

    listWidget = new QListWidget;
    listWidget->setContextMenuPolicy(Qt::ActionsContextMenu);
    listWidget->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    delSelected = new QAction(tr("Delete"), listWidget);
    listWidget->addAction(delSelected);
    connect(delSelected, SIGNAL(triggered()), this, SLOT(delList()));
    connect(listWidget, SIGNAL(itemClicked(QListWidgetItem*)), this, SLOT(showList(QListWidgetItem*)));

    inboundlist = new QListWidget;
    inboundlist->setContextMenuPolicy(Qt::ActionsContextMenu);
    inboundlist->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    inboundaddtolist = new QAction(tr("Add to list"), inboundlist);
    inbounddelete = new QAction(tr("Delete"), inboundlist);
    inboundlist->addAction(inboundaddtolist);
    inboundlist->addAction(inbounddelete);
    connect(inboundaddtolist, SIGNAL(triggered()), this, SLOT(inboundlistaddtolist()));
    connect(inbounddelete, SIGNAL(triggered()), this, SLOT(inboundlistdelete()));
    setupMenus();
    setupUi();
    statusLabel->setText(tr("Active list:\"%1\" [Mode:%2]").arg(listPath).arg(whitelistmode?"WhiteList":"BlackList"));
    recentInboundList();
}

MainWindow::~MainWindow()
{
}

void MainWindow::beBusy(bool on)
{
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, on);
}

void MainWindow::setupMenus()
{
    QActionGroup *filterGroup = new QActionGroup(this);
    filterGroup->setExclusive(true);
    actEnable = new QAction(tr("Enabled"), filterGroup);
    actEnable->setCheckable(true);
    actEnable->setChecked(daemonenabled);
    actDisable = new QAction(tr("Disabled"), filterGroup);
    actDisable->setCheckable(true);
    actDisable->setChecked(!daemonenabled);
    connect(filterGroup, SIGNAL(triggered(QAction *)), this, SLOT(daemonSwitch(QAction *)));
    menuBar()->addActions(filterGroup->actions());

    menuBar()->addAction(tr("New list"), this, SLOT(newList()));
    menuBar()->addAction(tr("Add list"), this, SLOT(addList()));
    menuBar()->addAction(tr("Setting"), this, SLOT(editSettings()));
    menuBar()->addAction(tr("Clear inbound calls"), this, SLOT(clearInboundCallLog()));
    menuBar()->addAction(tr("About"), this, SLOT(about()));
}

void MainWindow::setupUi()
{
    mLists = new QPushButton(tr("Manage Lists"), this);
    mLists->setIcon(QIcon::fromTheme("general_foldertree"));
    mLists->setCheckable(true);
    connect(mLists, SIGNAL(clicked()), this, SLOT(openLists()));
    recentInboundCall = new QPushButton(tr("Inbound calls"), this);
    recentInboundCall->setIcon(QIcon::fromTheme("general_received"));
    recentInboundCall->setCheckable(true);
    connect(recentInboundCall, SIGNAL(clicked()), this, SLOT(recentInboundList()));
    QVBoxLayout *mainLayout = new QVBoxLayout;
    QHBoxLayout *topHLayout = new QHBoxLayout;
    topHLayout->setContentsMargins(0,10,0,5);
    topHLayout->addWidget(recentInboundCall);
    topHLayout->addWidget(mLists);
    mainLayout->addLayout(topHLayout);
    mainLayout->addWidget(listWidget);
    mainLayout->addWidget(inboundlist);

    QWidget *widget = new QWidget;
    widget->setLayout(mainLayout);

    setCentralWidget(widget);
    setWindowTitle("Callerx");

    statusLabel = new QLabel(this);
    statusLabel->setStyleSheet("color: rgb(120, 120, 120); font-size: 20px");
    mainLayout->addWidget(statusLabel);
}

void MainWindow::openLists()
{
    mLists->setChecked(true);
    recentInboundCall->setChecked(false);
    listWidget->clear();
    QListWidgetItem *item;
    for (int i = 0; i < customlistspath.size(); ++i){
        QFileInfo f(customlistspath.at(i));
        item = new QListWidgetItem(QIcon::fromTheme("general_foldertree"), f.baseName());
        listWidget->addItem(item);
    }
    inboundlist->hide();
    listWidget->show();
}

void MainWindow::showList(QListWidgetItem *item)
{
    Q_UNUSED(item);
    QString fileName = customlistspath.at(listWidget->currentRow());
    if (!fileName.isEmpty()){
        if(!QFile::exists(fileName)){
            QMaemo5InformationBox::information(this, "File not found!, updating lists..");
            customlistspath.removeAt(listWidget->currentRow());
            delete(listWidget->takeItem(listWidget->currentRow()));
            QSettings settings("callerx", "callerx");
            QStringList list =  customlistspath.mid(customlistspath.size() > 1?2:1, -1);
            settings.setValue("customlistspath", list);
            return;
        }
        else {
            EditListWidget *listItems = new EditListWidget(fileName, this);
            listItems->show();
        }
    }
}

void MainWindow::delList()
{
    if (listWidget->currentRow() < 1 || customlistspath.at(listWidget->currentRow()) == listPath){
        QMaemo5InformationBox::information(this, "Deleting active list is not allowed!");
        return;
    }
    else {
        int reply = QMessageBox::question(this, "Warning!", tr("delete %1?").arg(listWidget->currentItem()->text()), QMessageBox::Yes | QMessageBox::No, QMessageBox::No);
        if (reply == QMessageBox::Yes){
            if (!QFile::remove(customlistspath.at(listWidget->currentRow()))){
                QMaemo5InformationBox::information(this, "Unable to delete file!");
                return;
            }
            customlistspath.removeAt(listWidget->currentRow());
            delete(listWidget->takeItem(listWidget->currentRow()));
            QSettings settings("callerx", "callerx");
            QStringList list =  customlistspath.mid(customlistspath.size() > 1?2:1, -1);
            settings.setValue("customlistspath", list);
        }
    }
}

void MainWindow::newList()
{
    QString fileName = QFileDialog::getSaveFileName(this, tr("Add New File"), "/home/user/MyDocs/newlist.txt");
    if (!fileName.isEmpty()){
        QFile blocklist(fileName);
        if (!blocklist.exists()) {
            if (!blocklist.open(QIODevice::WriteOnly)){
                QMaemo5InformationBox::information(this, "Unable to create new file");
                return;
            }
            blocklist.close();
        }
        customlistspath.append(fileName);
        openLists();
        QSettings settings("callerx", "callerx");
        QStringList list =  customlistspath.mid(customlistspath.size() > 1?2:1, -1);
        settings.setValue("customlistspath", list);
    }
}

void MainWindow::addList()
{
    QString fileName = QFileDialog::getOpenFileName(this, tr("Choose list/file"), "/home/user/MyDocs");
    if (!fileName.isEmpty()){
        QFile blocklist(fileName);
        if (!blocklist.exists()) {
            if (!blocklist.open(QIODevice::WriteOnly)){
                QMaemo5InformationBox::information(this, "Unable to add this list/file");
                return;
            }
            blocklist.close();
        }
        customlistspath.append(fileName);
        openLists();
        QSettings settings("callerx", "callerx");
        QStringList list =  customlistspath.mid(customlistspath.size() > 1?2:1, -1);
        settings.setValue("customlistspath", list);
    }
}

/* recent inbound call */
void MainWindow::recentInboundList()
{
    mLists->setChecked(false);
    recentInboundCall->setChecked(true);
    inboundlist->clear();
    QListWidgetItem *item;
    QString number;
    bool isBlocked;
    QIcon icon = QIcon::fromTheme("general_received");
    QIcon iconx = QIcon::fromTheme("callerx");
    QLabel *label;
    for (int i = 0; i < recentCalls.size(); ++i){
        number = recentCalls.at(i);
        isBlocked = number.startsWith("x");
        label = new QLabel(isBlocked?number.remove(0,1).replace(" [ ", "<br><font size=\"1\" color=\"gray\">").replace(" ]", "</font></br>"):number.replace(" [ ", "<br><font size=\"1\" color=\"gray\">").replace(" ]", "</font></br>"), inboundlist);
        item = new QListWidgetItem(inboundlist);
        item->setIcon(isBlocked?iconx:icon);
        inboundlist->setItemWidget(item, label);
    }
    listWidget->hide();
    inboundlist->show();
    statusBarUpdate();
}

void MainWindow::inboundlistdelete()
{
    if (inboundlist->count() < 1){
        return;
    }
    recentCalls.removeAt(inboundlist->currentRow());
    delete(inboundlist->takeItem(inboundlist->currentRow()));
    QSettings settings("callerx", "callerx");
    settings.setValue("recentinboundcalls", recentCalls);
    statusBarUpdate();
}

void MainWindow::inboundlistaddtolist()
{
    if (inboundlist->count() < 1){
        return;
    }
    QString temp = recentCalls.at(inboundlist->currentRow());;
    QString number = temp.split(" [ ")[0];
    bool ok;
    QString item = QInputDialog::getItem(this, number, tr("Lists:"), customlistspath, 0, false, &ok);
    if (ok && !item.isEmpty()){
        QFile blocklist(item);
        if (!blocklist.open(QIODevice::WriteOnly | QIODevice::Text | QIODevice::Append)){
            QMaemo5InformationBox::information(this, "Error: file not found");
            return;
        }
        QTextStream out(&blocklist);
        out << number << "\n";
        blocklist.close();
        QMaemo5InformationBox::information(this, "Successful Added!");
    }
}

void MainWindow::clearInboundCallLog()
{
    recentCalls.clear();
    inboundlist->clear();
    statusBarUpdate();
    QSettings settings("callerx", "callerx");
    settings.setValue("recentinboundcalls", recentCalls);
}

void MainWindow::statusBarUpdate()
{
    statusLabel->setText(tr("Current active list:\"%1\" [Mode:%2]").arg(listPath).arg(whitelistmode?"WhiteList":"BlackList"));
}

QString MainWindow::cleanString(const QString &str, bool prefix)
{
    QString cleanednumber = str;
    if (cleanednumber.contains("unknown")){
        return cleanednumber.trimmed();
    }
    else if (prefix){
        return cleanednumber.remove(QRegExp("[a-zA-Z.-#()@ \t\n\v\f\r]"));
    }
    else {
        return cleanednumber.remove(QRegExp("[a-zA-Z+.-#()@ \t\n\v\f\r]"));
    }
}

void MainWindow::editSettings()
{
    SettingDialog settingDialog;
    if(settingDialog.exec() && daemonenabled){
        readSettings();
        statusBarUpdate();
        callerxDaemon("reload");
    }
}

/* Parsing the stored settings */
void MainWindow::readSettings()
{
    QSettings settings("callerx", "callerx");
    daemonautostart = settings.value("auto-start", true).toBool();
    daemonenabled = settings.value("active", true).toBool();
    whitelistmode = settings.value("whitelistmode", false).toBool();
    daemonversion = settings.value("app_version", "n/a").toString();
    whitelistpath = settings.value("whitelistpath", DEFAULT_WHITELIST).toString();
    blacklistpath = settings.value("blacklistpath", DEFAULT_BLACKLIST).toString();
    customlistspath = settings.value("customlistspath", QStringList()).toStringList();
    recentCalls = settings.value("recentinboundcalls", QStringList()).toStringList();
    if (whitelistmode) {
        listPath = whitelistpath;
    }
    else {
        listPath = blacklistpath;
    }
    customlistspath.prepend(whitelistmode?blacklistpath:whitelistpath);
    customlistspath.prepend(listPath);
    for (int i = 0; i < customlistspath.size(); ++i){
        if (!QFile::exists(customlistspath.at(i))){
            customlistspath.removeAt(i);
        }
    }
    QStringList list =  customlistspath.mid(customlistspath.size() > 1?2:1, -1);
    settings.setValue("customlistspath", list);
}

/* Functions to control Callerx Daemon via Dbus */
void MainWindow::daemonSwitch(QAction *action)
{
    QSettings settings("callerx", "callerx");
    if(action == actEnable){
        daemonenabled = true;
        settings.setValue("active", true);
        callerxDaemon("start");
    }
    else {
        daemonenabled = false;
        settings.setValue("active", false);
        callerxDaemon("stop");
    }
}

void MainWindow::loadWhiteListDaemon()
{
    callerxDaemon("loadwhitelist");
}

void MainWindow::loadBlackListDaemon()
{
    callerxDaemon("loadblacklist");
}

bool MainWindow::callerxDaemonRunning()
{
    QDBusInterface iface(BUS_NAME, BUS_PATH, BUS_NAME, QDBusConnection::sessionBus(), this);
    QDBusReply<QString> reply = iface.call("ping");
    if (reply.isValid()){
        if (reply.value()=="pong"){
            return true;
        }
    }
    return false;
}

void MainWindow::callerxDaemon(const QString &m)
{
    QStringList args;
    args << m;
    QProcess *daemon = new QProcess;
    daemon->start("/etc/init.d/callerx", args);
}

void MainWindow::inBoundCallUpdate(const QString &msg)
{
    recentCalls.prepend(msg);
    QString number = msg;
    QIcon icon = QIcon::fromTheme("general_received");
    QIcon iconx = QIcon::fromTheme("callerx");
    QLabel *label;
    bool isBlocked = number.startsWith("x");
    label = new QLabel(isBlocked?number.remove(0,1).replace(" [ ", "<br><font size=\"1\" color=\"gray\">").replace(" ]", "</font></br>"):number.replace(" [ ", "<br><font size=\"1\" color=\"gray\">").replace(" ]", "</font></br>"));
    QListWidgetItem *item = new QListWidgetItem;
    item->setIcon(isBlocked?iconx:icon);
    inboundlist->insertItem(0, item);
    inboundlist->setItemWidget(item, label);
}

void MainWindow::about() const
{
    QMessageBox msgBox;
    msgBox.setAttribute(Qt::WA_Maemo5AutoOrientation, true);
    msgBox.setWindowTitle(tr("About"));
    msgBox.setText( QString(tr( "<font size=3>Callerx: ui - v%1 ; daemon - v%2</font><br>"
                                 "<font size=2>This is a callerx manager for callerx daemon writen in Qt.<br>"
                                 "It required callerx version 0.1.6 and later.<br><br>"
                                 "Bug report: <a href=\"http://talk.maemo.org/showthread.php?p=1052812#post1052812\">http://talk.maemo.org/showthread.php?p=1052812#post1052812</a><br><br></font>").arg( APP_VERSION ).arg( daemonversion )));
    msgBox.exec();
}
